/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RandomNormalTest.C
 *    Test cases for MIRA random number generators
 *
 * @author Erik Einhorn
 * @date   2012/11/09
 */

#include <boost/test/unit_test.hpp>
#if BOOST_VERSION >= 107100
#include <boost/test/tools/floating_point_comparison.hpp>
#else
#include <boost/test/floating_point_comparison.hpp>
#endif

#include <math/NormalRandomGenerator.h>
#include <math/UniformRandomGenerator.h>
#include <boost/random/uniform_01.hpp>

#include <utils/Time.h>
#include <math/Random.h>

using namespace mira;

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( RandomGeneratorTest )
{
	// basically tests if everything compiles fine
	const int N=1;

	RandomGenerator<boost::uniform_01<>> rnd;
	rnd.seed();

	for(int i=0; i<N; ++i)
		std::cout << rnd() << std::endl;

	UniformRandomGenerator<int> urnd(10,20);
	urnd.seed();
	for(int i=0; i<N; ++i)
		std::cout << urnd() << std::endl;

}

BOOST_AUTO_TEST_CASE( UniformRandomGeneratorTest )
{
	float a = 10.0f;
	float b = 20.0f;
	const int N = 10000;

	UniformRandomGenerator<float> urnd(a, b);
	urnd.seed();

	std::vector<float> values;
	for(int i = 0; i < N; i++)
		values.push_back(urnd());

	float mean = 0.0f;
	for(int i = 0; i < N; i++)
		mean += values[i];
	mean /= (1.0f*N);

	float variance = 0.0f;
	for(int i = 0; i < N; i++)
		variance += (mean-values[i])*(mean-values[i]);
	variance /= (1.0f*N);

	std::cout << mean << std::endl;
	std::cout << variance << std::endl;

	BOOST_CHECK_CLOSE(mean, 0.5f*(a+b), 1.0f);
	BOOST_CHECK_CLOSE(variance, (b-a)*(b-a)/12.0f, 5.0f);
}

BOOST_AUTO_TEST_CASE( AssignmentCompileTest )
{
	NormalRandomGenerator<2,float> a,b;
	NormalRandomGenerator<2,float> c(a);
	NormalRandomGenerator<2,float> c2(a.distribution());

	a = b;
	a = b.distribution();

	UniformRandomGenerator<float> u,v;
	UniformRandomGenerator<float> w(u);
	UniformRandomGenerator<float> w2(u.distribution());

	u = v;
	u = v.distribution();

}

BOOST_AUTO_TEST_CASE( RandomNormalTest )
{
	Eigen::Matrix2f sigma;

	sigma << 2.0f,1.0f,
			 1.0f,3.0f;

	NormalRandomGenerator<2,float> nrnd2(sigma);

	const int N=100000;
	Eigen::Matrix2f sigma2 = Eigen::Matrix2f::Zero();

	for(int i=0; i<N; ++i)
	{
		Eigen::Vector2f x = nrnd2();
		sigma2 += x * x.transpose();
	}
	sigma2 *= 1.0f/N;

	std::cout << sigma2 << std::endl;

	BOOST_CHECK_CLOSE(sigma2(0,0),sigma(0,0),1.0f);
	BOOST_CHECK_CLOSE(sigma2(0,1),sigma(0,1),1.0f);
	BOOST_CHECK_CLOSE(sigma2(1,0),sigma(1,0),1.0f);
	BOOST_CHECK_CLOSE(sigma2(1,1),sigma(1,1),1.0f);

}

BOOST_AUTO_TEST_CASE( RandomBenchmark )
{
	const int N=10000000;

	Time start;

	float sum=0;
	float sumd=0;

	std::cout << std::endl;
	std::cout << "Execution times: " << std::endl;

	start=Time::now();
	for(int i=0; i<N; ++i)
		sum += MIRA_RANDOM.uniform(0.0f,1.0f);
	std::cout << "MIRA_RANDOM.uniform:              " << (Time::now() - start) << std::endl;


	UniformRandomGenerator<float> urnd(0.0f,1.0f);
	start=Time::now();
	for(int i=0; i<N; ++i)
		sum += urnd();
	std::cout << "UniformRandomGenerator<float>:    " << (Time::now() - start) << std::endl;


	UniformRandomGenerator<double> urnd2(0.0,1.0);
	start=Time::now();
	for(int i=0; i<N; ++i)
		sum += urnd2();
	std::cout << "UniformRandomGenerator<double>:   " << (Time::now() - start) << std::endl;


	start=Time::now();
	for(int i=0; i<N; ++i)
		sum += MIRA_RANDOM.normal(1.0f);
	std::cout << "MIRA_RANDOM.normal:               " << (Time::now() - start) << std::endl;


	NormalRandomGenerator<1,float> nrnd(1.0f);
	start=Time::now();
	for(int i=0; i<N; ++i)
		sum += nrnd();
	std::cout << "NormalRandomGenerator<1, float>:  " << (Time::now() - start) << std::endl;


	NormalRandomGenerator<1,double> nrndd(1.0f);
	start=Time::now();
	for(int i=0; i<N; ++i)
		sumd += nrndd();
	std::cout << "NormalRandomGenerator<1, double>: " << (Time::now() - start) << std::endl;


	std::cout << std::endl;
	std::cout << "Multivariate normal:" << std::endl;
	Eigen::Vector2f sum2;
	NormalRandomGenerator<2,float> nrnd2;
	start=Time::now();
	for(int i=0; i<N; ++i)
		sum2 += nrnd2();
	std::cout << "NormalRandomGenerator<2, float>:  " << (Time::now() - start) << std::endl;

	Eigen::Vector2d sum2d;
	NormalRandomGenerator<2,double> nrnd2d;
	start=Time::now();
	for(int i=0; i<N; ++i)
		sum2d += nrnd2d();
	std::cout << "NormalRandomGenerator<2, double>: " << (Time::now() - start) << std::endl;

	Eigen::Vector3f sum3;
	NormalRandomGenerator<3,float> nrnd3;
	start=Time::now();
	for(int i=0; i<N; ++i)
		sum3 += nrnd3();
	std::cout << "NormalRandomGenerator<3, float>:  " << (Time::now() - start) << std::endl;

	Eigen::Vector3d sum3d;
	NormalRandomGenerator<3,double> nrnd3d;
	start=Time::now();
	for(int i=0; i<N; ++i)
		sum3d += nrnd3d();
	std::cout << "NormalRandomGenerator<3, double>: " << (Time::now() - start) << std::endl;

	std::cout << std::endl;
	std::cout << sum << std::endl;
	std::cout << sumd << std::endl;
	std::cout << sum2 << std::endl;
	std::cout << sum3 << std::endl;
	std::cout << sum2d << std::endl;
	std::cout << sum3d << std::endl;

}
