/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file JSONTest.C
 *    Test case for JSON.h.
 *
 * @author Tim Langner
 * @date   2012/04/02
 */

#include <boost/test/unit_test.hpp>

#include <json/JSON.h>
#include <serialization/BinarySerializer.h>
#include <serialization/Print.h>

using namespace mira;

BOOST_AUTO_TEST_CASE( JSONSerializeTest )
{
	json::Value v1 = json::Object();
	v1.get_obj()["Float"] = json::Value(3.0f);
	v1.get_obj()["String"] = json::Value("Hello");
	v1.get_obj()["Int"] = json::Value(2000);

	BinaryBufferOstream::buffer_type s;
	BinaryBufferSerializer bs(&s);
	bs.serialize(v1);

	json::Value v2;
	BinaryBufferDeserializer ds(&s);
	ds.deserialize(v2);

	BOOST_REQUIRE(v2.type() == json_spirit::obj_type);
	BOOST_CHECK_EQUAL(v2.get_obj()["Float"].get_real(), 3.0f);
	BOOST_CHECK_EQUAL(v2.get_obj()["String"].get_str(), std::string("Hello"));
	BOOST_CHECK_EQUAL(v2.get_obj()["Int"].get_int(), 2000);
}

BOOST_AUTO_TEST_CASE( JSONWriteReadTest )
{
	json::Value v1 = json::Object();
	v1.get_obj()["Float"] = json::Value(3.011f);
	v1.get_obj()["String"] = json::Value("Hello");
	v1.get_obj()["Int"] = json::Value(2000);

	std::string s;
	json::Value v2;

	// test write() then read() from written string returns same object
	s = json::write(v1, false);
	json::read(s, v2);

	BOOST_REQUIRE(v2.type() == json_spirit::obj_type);
	BOOST_CHECK_CLOSE(v2.get_obj()["Float"].get_real(), 3.011f, 0.00001f);
	BOOST_CHECK_EQUAL(v2.get_obj()["String"].get_str(), std::string("Hello"));
	BOOST_CHECK_EQUAL(v2.get_obj()["Int"].get_int(), 2000);

	// test write() then read() from written string returns same object,
	// regardless of write formatting
	s = json::write(v1, true);
	json::read(s, v2);

	BOOST_REQUIRE(v2.type() == json_spirit::obj_type);
	BOOST_CHECK_CLOSE(v2.get_obj()["Float"].get_real(), 3.011f, 0.00001f);
	BOOST_CHECK_EQUAL(v2.get_obj()["String"].get_str(), std::string("Hello"));
	BOOST_CHECK_EQUAL(v2.get_obj()["Int"].get_int(), 2000);

	// test precision parameter
	s = json::write(v1, false, 1);
	json::read(s, v2);

	BOOST_REQUIRE(v2.type() == json_spirit::obj_type);
	BOOST_CHECK_CLOSE(v2.get_obj()["Float"].get_real(), 3.f, 0.00001f);

	s = json::write(v1, false, 2);
	json::read(s, v2);

	BOOST_REQUIRE(v2.type() == json_spirit::obj_type);
	BOOST_CHECK_CLOSE(v2.get_obj()["Float"].get_real(), 3.01f, 0.00001f);
	
	s = json::write(v1, false, 3);
	json::read(s, v2);

	BOOST_REQUIRE(v2.type() == json_spirit::obj_type);
	BOOST_CHECK_CLOSE(v2.get_obj()["Float"].get_real(), 3.011f, 0.00001f);

	s = json::write(v1, false, 4);
	json::read(s, v2);

	BOOST_REQUIRE(v2.type() == json_spirit::obj_type);
	BOOST_CHECK_CLOSE(v2.get_obj()["Float"].get_real(), 3.011f, 0.00001f);
}

BOOST_AUTO_TEST_CASE( JSONWriteTest )
{
	json::Value v1 = json::Object();
	v1.get_obj()["Array"] = std::vector<json::Value>({json::Value(1),json::Value(2),json::Value(3)});
	v1.get_obj()["Double"] = json::Value(3.011);
	v1.get_obj()["Float"] = json::Value(3.011f);

	std::string s;

	// test unformatted output
	s = json::write(v1);
	std::cout << "unformatted : " << s << std::endl;
	BOOST_CHECK_EQUAL(s, std::string("{\"Array\":[1,2,3],\"Double\":3.011,\"Float\":3.011}"));

	// test formatted output
	s = json::write(v1, true);
	std::cout << "formatted :\n" << s << std::endl;
	BOOST_CHECK_EQUAL(s, std::string("{\n    \"Array\" : [ 1, 2, 3 ],\n    \"Double\" : 3.011,\n    \"Float\" : 3.011\n}"));

	// test increasing precision
	s = json::write(v1, true, 1);
	BOOST_CHECK_EQUAL(s, std::string("{\n    \"Array\" : [ 1, 2, 3 ],\n    \"Double\" : 3.0,\n    \"Float\" : 3.0\n}"));

	s = json::write(v1, true, 2);
	BOOST_CHECK_EQUAL(s, std::string("{\n    \"Array\" : [ 1, 2, 3 ],\n    \"Double\" : 3.01,\n    \"Float\" : 3.01\n}"));

	s = json::write(v1, true, 9);
	BOOST_CHECK_EQUAL(s, std::string("{\n    \"Array\" : [ 1, 2, 3 ],\n    \"Double\" : 3.011000000,\n    \"Float\" : 3.010999918\n}"));
}

BOOST_AUTO_TEST_CASE( JSONDefaultPrecisionTest )
{
	struct ExposeDefaultPrecision : public json::JSONDefaultPrecision
	{
		operator unsigned int() const { return MIRA_JSON_DEFAULT_PRECISION; }
	};

	BOOST_CHECK_EQUAL(json::JSONDefaultPrecision::get(), (unsigned int)ExposeDefaultPrecision());

	double d = 1.25048828125; // 1.250'488'281'250  = 2^0 + 2^-2 + 2^-11, stored as double without conversion error,
	                          // so we can be sure of trailing zeros after 11 digits

	BOOST_CHECK_EQUAL(json::write(json::Value(d)), std::string("1.250")); // default precision is 3
	BOOST_CHECK_EQUAL(json::write(json::Value(d), false, 4), std::string("1.2505"));
	BOOST_CHECK_EQUAL(json::write(json::Value(d), false, 5), std::string("1.25049"));
	BOOST_CHECK_EQUAL(json::write(json::Value(d), true, 5), std::string("1.25049")); // formatted does not matter here

	json::JSONDefaultPrecision::set(8);
	BOOST_CHECK_EQUAL(json::JSONDefaultPrecision::get(), 8);

	BOOST_CHECK_EQUAL(json::write(json::Value(d)), std::string("1.25048828"));  // default precision is 8
	BOOST_CHECK_EQUAL(json::write(json::Value(d), false, 4), std::string("1.2505")); // explicit precision parameter overrides default
	BOOST_CHECK_EQUAL(json::write(json::Value(d), false, 5), std::string("1.25049"));
	BOOST_CHECK_EQUAL(json::write(json::Value(d), true, 5), std::string("1.25049"));

	json::JSONDefaultPrecision::set(16);
	BOOST_CHECK_EQUAL(json::JSONDefaultPrecision::get(), 16);

	BOOST_CHECK_EQUAL(json::write(json::Value(d)), std::string("1.2504882812500000"));

	json::JSONDefaultPrecision::reset();  // reset to "default default"

	BOOST_CHECK_EQUAL(json::JSONDefaultPrecision::get(), (unsigned int)ExposeDefaultPrecision());
	BOOST_CHECK_EQUAL(json::write(json::Value(d)), std::string("1.250")); // default precision is 3
}

BOOST_AUTO_TEST_CASE( JSONSerializationPrecisionTest )
{
	json::JSONDefaultPrecision::set(3);

	double d = 1.25048828125; // 1.250'488'281'250  = 2^0 + 2^-2 + 2^-11, stored as double without conversion error,
	                          // so we can be sure of trailing zeros after 11 digits

	json::Value value(d);

	Buffer<uint8> buf;
 	BinaryBufferSerializer s(&buf);

	s.serialize(value);

 	BinaryBufferDeserializer ds(&buf);
	json::Value value2;
	ds.deserialize(value2);

	BOOST_CHECK_EQUAL(json::write(value2), std::string("1.250"));
	BOOST_CHECK_EQUAL(json::write(value2, false, 0), std::string("1.2504882812500000"));
}

BOOST_AUTO_TEST_CASE( JSONNonFiniteTest )
{
	json::Value v1 = json::Object();
	v1.get_obj()["nan"] = json::Value(std::numeric_limits<float>::quiet_NaN());
	v1.get_obj()["inf"] = json::Value(std::numeric_limits<float>::infinity());
	v1.get_obj()["-inf"] = json::Value(-std::numeric_limits<float>::infinity());

	std::string s;
	json::Value v2;
	float fi, fi2;

	// test write() then read() from written string returns same object
	s = json::write(v1, false);
	json::read(s, v2);

	BOOST_REQUIRE(v2.type() == json_spirit::obj_type);
	BOOST_CHECK(std::isnan(v2.get_obj()["nan"].get_real()));

	fi = v2.get_obj()["inf"].get_real();
	BOOST_CHECK(std::isinf(fi));
	BOOST_CHECK(fi > 0.f);

	fi2 = v2.get_obj()["-inf"].get_real();
	BOOST_CHECK(std::isinf(fi2));
	BOOST_CHECK(fi2 < 0.f);

	// test write() then read() from written string returns same object,
	// regardless of write formatting
	s = json::write(v1, true);
	json::read(s, v2);

	BOOST_REQUIRE(v2.type() == json_spirit::obj_type);
	BOOST_CHECK(std::isnan(v2.get_obj()["nan"].get_real()));

	fi = v2.get_obj()["inf"].get_real();
	BOOST_CHECK(std::isinf(fi));
	BOOST_CHECK(fi > 0.f);

	fi2 = v2.get_obj()["-inf"].get_real();
	BOOST_CHECK(std::isinf(fi2));
	BOOST_CHECK(fi2 < 0.f);
}

BOOST_AUTO_TEST_CASE( UIntTest )
{
	std::stringstream ss, ss2;

	int32 i32 = -1;
	ss << json::write(json::Value(json::cast(i32)));
	ss2 << i32;
	std::cout << "int32(-1): " << ss.str() << std::endl;
	BOOST_CHECK_EQUAL(ss.str(), ss2.str());
	ss.str("");
	ss2.str("");

	uint32 ui32 = -1;
	ss << json::write(json::Value(json::cast(ui32)));
	ss2 << ui32;
	std::cout << "uint32(-1): " << ss.str() << std::endl;
	BOOST_CHECK_EQUAL(ss.str(), ss2.str());
	ss.str("");
	ss2.str("");

	int64 i64 = -1;
	ss << json::write(json::Value(json::cast(i64)));
	ss2 << i64;
	std::cout << "int64(-1): " << ss.str() << std::endl;
	BOOST_CHECK_EQUAL(ss.str(), ss2.str());
	ss.str("");
	ss2.str("");

	uint64 ui64 = -1;
	ss << json::write(json::Value(json::cast(ui64)));
	ss2 << ui64;
	std::cout << "uint64(-1): " << ss.str() << std::endl;
	BOOST_CHECK_EQUAL(ss.str(), ss2.str());
	ss.str("");
	ss2.str("");

}

void checkElement(const json::Value& iValue, const std::string& element,
                  bool exists, const json::Value& ref, const std::string& referror = "")
{
#if BOOST_VERSION >= 106200
	BOOST_TEST_CONTEXT(std::string("checkElement for ") + element) {
#endif
	json::Value v;
	bool b;
	json::QueryValueResult r;

	if (exists) {
		v = json::getElement(iValue, element);
		BOOST_CHECK(v == ref);
	} else
		BOOST_CHECK_THROW(json::getElement(iValue, element), XInvalidConfig);

	b = json::hasElement(iValue, element);
	BOOST_CHECK(b == exists);

	r = json::getElementIfExists(iValue, element);
	BOOST_CHECK(r.exists == exists);
	if (exists)
		BOOST_CHECK(r.value == ref);
	BOOST_CHECK(r.error == referror);
#if BOOST_VERSION >= 106200
	}
#endif
}

void checkNumberElement(const json::Value& iValue, const std::string& element,
                        bool exists, double ref, const std::string& referror = "")
{
#if BOOST_VERSION >= 106200
	BOOST_TEST_CONTEXT(std::string("checkNumberElement for ") + element) {
#endif
	double d;
	bool b;
	json::QueryNumberResult r;

	if (exists) {
		d = json::getNumberElement(iValue, element);
		BOOST_CHECK(d == ref);
	} else
		BOOST_CHECK_THROW(json::getNumberElement(iValue, element), XInvalidConfig);

	b = json::hasNumberElement(iValue, element);
	BOOST_CHECK(b == exists);

	r = json::getNumberElementIfExists(iValue, element);
	BOOST_CHECK(r.exists == exists);
	if (exists)
		BOOST_CHECK(r.number == ref);
	BOOST_CHECK(r.error == referror);
#if BOOST_VERSION >= 106200
	}
#endif
}

BOOST_AUTO_TEST_CASE( GetElementTest )
{
	json::Value f1 = json::Value(1.f);
	json::Value f2 = json::Value(2.f);
	json::Value f25 = json::Value(2.5f);
	json::Value s = json::Value("s");
	json::Value i2 = json::Value(2);
	json::Value i3 = json::Value(3);
	json::Value m = json::Value("[1,2,3;4,5,6;7,8,9]");

	json::Value o = json::Object();
	o.get_obj()["F1"] = f1;
	o.get_obj()["S"] = s;

	json::Array a = json::Array();
	a.push_back(i2);
	a.push_back(i3);
	a.push_back(s);
	o.get_obj()["A"] = a;

	json::Array a2 = json::Array();
	json::Array a3 = json::Array();
	json::Array a4 = json::Array();
	a3.push_back(f1);
	a3.push_back(f2);
	a4.push_back(f25);
	a4.push_back(i3);
	a2.push_back(a3);
	a2.push_back(a4);
	o.get_obj()["A2"] = a2;

	o.get_obj()["M"] = m;

	json::Value o1 = json::Object();
	o1.get_obj()["F2"] = f2;
	o1.get_obj()["F25"] = f25;
	o.get_obj()["O"] = o1;

	json::write(o, std::cout);
	std::cout << std::endl;

	checkElement(a, "[0]", true, i2);
	checkElement(o, "F1", true, f1);
	checkElement(o, "S", true, s);
	checkElement(o, "A[0]", true, i2);
	checkElement(o, "A[1]", true, i3);
	checkElement(o, "A[2]", true, s);
	checkElement(o, "A2[1].[0]", true, f25);
	checkElement(o, "M", true, m);
	checkElement(o, "O.F2", true, f2);
	checkElement(o, "O.F25", true, f25);

	checkNumberElement(m, "[2][2]", true, 9.);
	checkNumberElement(o, "F1", true, 1.);
	checkNumberElement(o, "S", false, 0., "json::getNumberElement(): Data must be of real, int or bool type");
	checkNumberElement(o, "A[0]", true, 2.);
	checkNumberElement(o, "A[1]", true, 3.);
	checkNumberElement(o, "A[2]", false, 0., "json::getNumberElement(): Data must be of real, int or bool type");
	checkNumberElement(o, "A2[1].[0]", true, 2.5);
	checkNumberElement(o, "M[0]", false, 0., "json::accessElement(): Data must be a json::Array if index is specified");
	checkNumberElement(o, "M[0][0]", true, 1.);
	checkNumberElement(o, "M[1][2]", true, 6.);
	checkNumberElement(o, "M[2][0]", true, 7.);
	checkNumberElement(o, "M[1][3]", false, 0., "json::getNumberElement(): Matrix element index is out of range");
	checkNumberElement(o, "M[3][0]", false, 0., "json::getNumberElement(): Matrix element index is out of range");
	checkNumberElement(o, "O.F2", true, 2.);
	checkNumberElement(o, "O.F25", true, 2.5);
}
