/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SQLiteQuery.C
 *    Implementation of SQLiteQuery.h
 *
 * @author Tim Langner
 * @date   2010/10/25
 */

#include <database/SQLiteQuery.h>

#include <sqlite3.h>

#include <database/SQLiteDB.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

SQLiteQuery::SQLiteQuery() :
	mStatement(NULL),
	mMore(false)
{
}

SQLiteQuery::SQLiteQuery(sqlite3* db, sqlite3_stmt* statement, bool more) :
	mDB(db),
	mStatement(statement),
	mMore(more)
{
	assert(mDB != NULL);
	assert(mStatement != NULL);
	mColumnCount = sqlite3_column_count(mStatement);
}

SQLiteQuery::SQLiteQuery(SQLiteQuery&& other) :
	mDB(other.mDB),
	mStatement(other.mStatement),
	mMore(other.mMore),
	mColumnCount(other.mColumnCount)
{
	other.mDB = NULL;
	other.mStatement = NULL;
	other.mMore = false;
	other.mColumnCount = 0;
}

SQLiteQuery& SQLiteQuery::operator=(SQLiteQuery&& other)
{
	if (&other == this)
		return *this;

	std::swap(mDB, other.mDB);
	std::swap(mStatement, other.mStatement); // other will finalize it in its destructor
	std::swap(mMore, other.mMore);
	std::swap(mColumnCount, other.mColumnCount);

	return *this;
}

SQLiteQuery::~SQLiteQuery()
{
	if ( mStatement )
		sqlite3_finalize(mStatement);
}

int SQLiteQuery::getDataType(std::size_t column)
{
	if(column>=(std::size_t)mColumnCount)
		MIRA_THROW(XAccessViolation, "Column " << column << " does not exist");
	return sqlite3_column_type(mStatement, column);
}

bool SQLiteQuery::isNull(std::size_t column)
{
	return getDataType(column) == SQLITE_NULL;
}

std::string SQLiteQuery::getName(std::size_t column)
{
	if(column>=(std::size_t)mColumnCount)
		MIRA_THROW(XAccessViolation, "Column " << column << " does not exist");
	return sqlite3_column_name(mStatement, column);
}

const char* SQLiteQuery::getTxtValue(std::size_t column)
{
	if ( isNull(column) )
		return NULL;
	return (const char*)sqlite3_column_text(mStatement, column);
}

void SQLiteQuery::next()
{
	if ( eof() )
		return;

	int ret = sqlite3_step(mStatement);

	if ( ret == SQLITE_ERROR || ret == SQLITE_BUSY || ret == SQLITE_MISUSE )
	{
		const char* msg = sqlite3_errmsg(mDB);
		MIRA_THROW(XSQLite, "Query next() failed. Error: " << msg);
	}
	if ( ret == SQLITE_DONE )
		mMore = false;
}

///////////////////////////////////////////////////////////////////////////////

SQLiteCachedQuery::SQLiteCachedQuery() : mCurrent(0)
{
}

SQLiteCachedQuery::SQLiteCachedQuery(SQLiteQuery& query) : mCurrent(0)
{
	mHeader.resize(query.getColumnCount());
	mCache.clear();
	// store all header names in cache
	for (int c=0; c<query.getColumnCount(); ++c)
		mHeader[c] = query.getName(c);
	int index = 0;
	// get all rows from query
	while(!query.eof())
	{
		// for each column store its type (for null types) and its value in cache 
		mCache.push_back(std::vector<std::pair<int, std::string>>(query.getColumnCount()));
		for (int c=0; c<query.getColumnCount(); ++c)
		{
			std::pair<int, std::string> entry;
			entry.first = query.getDataType(c);
			if (!query.isNull(c))
				entry.second = query.getValue<std::string>(c);
			mCache[index][c] = entry;
		}
		index++;
		query.next();
	}
}

int SQLiteCachedQuery::getDataType(std::size_t column)
{
	validate(column);
	return mCache[mCurrent][column].first; 
}

bool SQLiteCachedQuery::isNull(std::size_t column)
{
	return getDataType(column) == SQLITE_NULL;
}

///////////////////////////////////////////////////////////////////////////////

}
