/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file IOService.C
 *    Implementation of IOService.h.
 *
 * @author Tim Langner
 * @date   2010/07/28
 */

#include <communication/IOService.h>

#include <thread/Thread.h>
#include <iostream>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class IOService::Pimpl
{
public:
	boost::asio::io_service service;
	boost::asio::io_service::work work;

	Pimpl() : work(service) {}
	~Pimpl() {
		stop();
	}

	void stop() {
		service.stop();
	}

	static void workerThread(boost::shared_ptr<Pimpl> pimpl)
	{
		// keeping the pimpl as shared pointer here, makes sure that it will
		// exist as long as the thread is running, even if the IOService does
		// not use it any longer
		pimpl->service.run();
	}
};



///////////////////////////////////////////////////////////////////////////////

IOService::IOService() :
	mPimpl(new Pimpl),
	mService(mPimpl->service)
{
}

IOService::IOService(boost::asio::io_service& service) :
	mService(service)
{
}

void IOService::join()
{
	mPimpl->stop(); // make sure to stop service, otherwise threads may block

	// Interrupt all threads first
	for (size_t i=0; i<mThreads.size(); ++i)
		mThreads[i]->interrupt();

	// Now join all threads
	for (size_t i=0; i<mThreads.size(); ++i) {
		mThreads[i]->join();
		mThreads[i].reset();
	}

	mThreads.clear();
}

void IOService::runThreads(size_t threadCount, bool runInThisThread)
{
	while( threadCount > mThreads.size() )
	{
		mThreads.push_back(ThreadPtr(
			new boost::thread(boost::bind(Pimpl::workerThread, mPimpl)) ));
	}

	if ( runInThisThread )
	{
		mService.run();
		join();
	}
}

///////////////////////////////////////////////////////////////////////////////

} // namespace
