/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ResolveVariable.h
 *    Functions for replacing/resolving variables in strings.
 *
 * @author Tim Langner
 * @date   2011/11/18
 */

#ifndef _MIRA_RESOLVEVARIABLE_H_
#define _MIRA_RESOLVEVARIABLE_H_

#include <string>
#ifndef Q_MOC_RUN
#include <boost/function.hpp>
#endif

#include <platform/Types.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Resolves all variables in a string: It replaces
 *  -# $VARIABLE
 *  -# ${VARIABLE}
 *  -# %VARIABLE%
 * by calling the given functor for each found variable.
 * @param[in] s The string to be resolved
 * @param functor Function that is called for each found variable, returning a
 *        value for the variable
 * @param ignoreErrors If true exceptions thrown by the functor call are
 * 		  catched and ignored
 * @return The resolved string
 */
MIRA_BASE_EXPORT std::string resolveVariables(std::string s,
                            boost::function<std::string (const std::string&)> functor,
                            bool ignoreErrors = true);

/**
 * Resolves all occurrences of ${PATTERN variable} in a string by calling
 * the given functor for each found pattern - variable combination.
 * The functor must resolve the variable according to the pattern
 * (e.g. s="${ENV variable}" calls functor with (ENV, variable))
 * @param s The string to be resolved
 * @param functor Function that is called for each found pattern-variable combination
 * @param ignoreErrors If true exceptions thrown by the functor call are
 * 		  catched and ignored
 * @return The resolved string
 */
MIRA_BASE_EXPORT std::string resolveSpecialVariables(std::string s,
                            boost::function<std::string (const std::string&,
                                                         const std::string&)> functor,
                            bool ignoreErrors = true);

/**
 * Like resolveSpecialVariables above but only resolves a special pattern.
 * Resolves all occurrences of ${PATTERN variable} in a string by calling
 * the given functor for each found variable.
 * The functor must resolve the variable.
 * @param pattern The pattern (e.g. pattern = ENV finds all ${ENV variable})
 * @param s The string to be resolved
 * @param functor Function that is called for each found variable
 * @param ignoreErrors If true exceptions thrown by the functor call are
 * 		  catched and ignored
 * @return The resolved string
 */
MIRA_BASE_EXPORT std::string resolveSpecialVariables(const std::string& pattern, 
                            std::string s,
                            boost::function<std::string (const std::string&)> functor,
                            bool ignoreErrors = true);

///////////////////////////////////////////////////////////////////////////////

}

#endif 
