/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PropertyHint.h
 *    Provides property hints and attributes.
 *
 * @author Erik Einhorn
 * @date   2011/01/04
 */

#ifndef _MIRA_PROPERTYHINT_H_
#define _MIRA_PROPERTYHINT_H_

#include <assert.h>

#include <list>
#include <string>
#include <utility>

#include <utils/ToString.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 *
 * A property hint gives optional instructions to the property editor, i.e.
 * minimum and maximum value and which kind of editor to use.
 * Each property hint may consist of a list of hints and attributes that are
 * stored as Attribute-Value pairs.
 *
 * Hints can be concatenated easily using the | operator, e.g.
 * \code
 *     PropertyHints::limits(1,10) | PropertyHint("MyHint", "MyValue") | PropertyHints::step(1)
 * \endcode
 *
 * @note This class must be inline to avoid performance penalties in
 *       serializers that are performance critical and that do NOT take care
 *       of these property hints. For the same reason the attributes are stored
 *       using a pointer, to avoid unnecessary instantiation of containers
 *       if no property hint is specified. Using these optimizations, property
 *       hints do not pose any computational overhead.
 */
class PropertyHint
{
public:

	typedef std::list<std::pair<std::string, std::string>> AttributeValueList;

public:

	PropertyHint()  : mAttributes(NULL) {}

	/// Constructs a single hint from the given attribute value pair
	PropertyHint(const std::string& attribute, const std::string& value) {
		mAttributes = new AttributeValueList;
		mAttributes->push_back(std::make_pair(attribute, value));
	}

	/// move constructor
	PropertyHint(PropertyHint&& other) : mAttributes(NULL) {
		std::swap(mAttributes, other.mAttributes);
	}

	~PropertyHint() {
		delete mAttributes;
	}

public:

	/// Returns the attributes/value list containing the property hints
	AttributeValueList toList() const {
		return mAttributes ? *mAttributes : AttributeValueList();
	}

	/// Sets the property hints from the specified list of attribute/value pairs.
	void fromList(const AttributeValueList& list) {

		if(list.empty()) {
			delete mAttributes;
			mAttributes = NULL;
			return;
		}

		if(!mAttributes)
			mAttributes = new AttributeValueList;

		*mAttributes = list;
	}

public:

	/// move assignment operator
	PropertyHint& operator=(PropertyHint&& other) {
		std::swap(mAttributes, other.mAttributes);
		return *this;
	}


public:

	/**
	 * Creates an explicit copy as replacement for the copy constructor.
	 * The copy constructor was made private do avoid any performance penalties
	 * within the serialization framework whenever property hints are used.
	 */
	PropertyHint clone() const
	{
		PropertyHint copy;
		if(mAttributes!=NULL) {
			copy.mAttributes = new AttributeValueList;
			*copy.mAttributes = *mAttributes;
		}
		return copy;
	}

public:

	/// Returns true if the specified attribute exists
	bool has(const std::string& attribute) const
	{
		if(mAttributes!=NULL) {
			for(auto it=mAttributes->begin(); it!=mAttributes->end(); ++it)
			{
				if(it->first==attribute)
					return true;
			}
		}
		return false;
	}

	/**
	 * Returns the specified value for the given attribute. If no such
	 * attribute is set in this hint, then the defaultValue is returned.
	 */
	template <typename T>
	T get(const std::string& attribute, const T& defaultValue = T()) const
	{
		if(mAttributes!=NULL) {
			for(auto it=mAttributes->begin(); it!=mAttributes->end(); ++it)
			{
				if(it->first==attribute)
					return fromString<T>(it->second);
			}
		}
		return defaultValue;
	}

public:

	/**
	 * Concatenates two hints.
	 * NOTE: to increase performance this operator takes the content of the
	 * second hint and moves it into the first hint. Therefore, after calling
	 * this operator the second hint will be empty.
	 */
	friend PropertyHint operator|(PropertyHint&& hint, PropertyHint&& otherHint)
	{
		if(otherHint.mAttributes==NULL)
			return std::move(hint);

		if(hint.mAttributes==NULL) {
			std::swap(hint.mAttributes, otherHint.mAttributes);
			return std::move(hint);
		}

		assert(hint.mAttributes);
		assert(otherHint.mAttributes);

		// move attributes from otherHint to hint
		hint.mAttributes->splice(hint.mAttributes->begin(), *otherHint.mAttributes,
				otherHint.mAttributes->begin(), otherHint.mAttributes->end());
		return std::move(hint);
	}

private:
	/// no copying constructor
	PropertyHint(PropertyHint& other);
	/// no assignment operator
	PropertyHint& operator=(PropertyHint& other);

	AttributeValueList* mAttributes;
};

/**
 * @ingroup SerializationModule
 * Namespace to put all PropertyHint creation functions.
 * Some common creation functions are defined in PropertyHint.h
 *
 * @note If you define new property hint creator functions, please make sure
 *       to make them inline to avoid performance penalties in serializers that
 *       are performance critical and that do NOT take care of these property
 *       hints (e.g. the binary serializers).
 */
namespace PropertyHints {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 * Sets the attribute "minimum" to the specified value.
 * In property editors this is the minimum value the user can select.
 */
template <typename T>
inline PropertyHint minimum(const T& min) {
	return PropertyHint("minimum",toString(min));
}

/**
 * @ingroup SerializationModule
 * Sets the attribute "maximum" to the specified value.
 * In property editors this is the maximum value the user can select.
 */
template <typename T>
inline PropertyHint maximum(const T& max) {
	return PropertyHint("maximum",toString(max));
}

/**
 * @ingroup SerializationModule
 * Sets the attribute "step" to the specified value.
 * In property editors this specifies the single step. When the user
 * is interacting with the editor the value will be incremented/decremented
 * by this amount.
 */
template <typename T>
inline PropertyHint step(const T& step) {
	return PropertyHint("step",toString(step));
}

/**
 * @ingroup SerializationModule
 * Sets both attributes "minimum" and "maximum" to the specified values.
 * See above.
 */
template <typename T>
inline PropertyHint limits(const T& min, const T& max) {
	return minimum<T>(min) | maximum<T>(max);
}

/**
 * @ingroup SerializationModule
 * Sets the attribute "precision". A property editor can
 * use this hint to display the specified number of decimals
 * of floating point values.
 */
inline PropertyHint precision(int p) {
	return PropertyHint("precision", toString(p));
}

/**
 * @ingroup SerializationModule
 * Sets the attribute "type" to the specified value. The attribute may be
 * used by property editors to select the appropriate editor element
 * (e.g. spin box or sliders, etc.)
 */
inline PropertyHint type(const std::string& t) {
	return PropertyHint("type", t);
}

/**
 * @ingroup SerializationModule
 * Sets the attribute "enumeration". The specified values must be
 * separated by semicolons and specify the strings of the enumeration
 * values. A property editor that supports this hint may show a drop
 * down box with the strings.
 */
inline PropertyHint enumeration(const std::string& values) {
	return PropertyHint("enumeration", values);
}

/**
 * @ingroup SerializationModule
 * Sets the attribute "type" to the value "slider" and sets the
 * "minimum", "maximum" and "step" attributes. A property editor that
 * supports this hint may show a slider control instead of a spin box
 */
template <typename T>
inline PropertyHint slider(const T& min, const T& max, const T& s) {
	return type("slider") | limits<T>(min,max) | step<T>(s);
}

/**
 * @ingroup SerializationModule
 * Sets the attribute "type" to the value "spinbox" and sets the
 * "minimum", "maximum" and "step" attributes. A property editor that
 * supports this hint may show a spinbox control.
 */
template <typename T>
inline PropertyHint spin(const T& min, const T& max, const T& s) {
	return type("spin") | limits<T>(min,max) | step<T>(s);
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_PROPERTYHINT_H_ */
