/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file JSONSerializer.h
 *    Serializer and Deserializer for JSON format.
 *
 * @author Tim Langner
 * @date   2010/11/01
 */

#ifndef _MIRA_JSONSERIALIZER_H_
#define _MIRA_JSONSERIALIZER_H_

#include <iostream>
#include <list>
#include <algorithm>
#include <stack>

#include <serialization/Serializer.h>
#include <serialization/Deserializer.h>

#include <json/JSON.h>

#include <serialization/adapters/std/StlCollections.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 *
 * Serializer for serializing objects in JSON format.
 *
 * Example usage:
 * \code
 * #include <serialization/adapters/std/vector>
 * // the data to be serialized
 * std::vector<int> myValue;
 * ...
 *
 * // create a JSONSerializer that writes into a JSON value
 * JSONSerializer serializer;
 *
 * // serialize the data and return the JSON value containing the data
 * json::Value v = serializer.serialize(myValue);
 *
 * // write the JSON value to the console
 * json::write(v, std::cout, true);
 * \endcode
 *
 * For the deserialization of JSON content see
 * @ref mira::JSONDeserializer "JSONDeserializer".
 *
 * @see @ref SerializationPage, JSONDeserializer
 */
class JSONSerializer : public Serializer<JSONSerializer>
{
	typedef Serializer<JSONSerializer> Base;

public:

	/// Flags for JSON output format
	enum OutputFormat
	{
		/// Default format
		STANDARD               = 0x00,

		/// Serialize associative containers with string keys as JSON object (keys = member names)
		STRING_MAP_AS_OBJECT   = 0x01
	};

	MIRA_ENUM_TO_FLAGS_INCLASS(OutputFormat) // generate logical operators for enum values

	/**
	 * Construct a serializer.
	 * @param readOnly If true it also visits ro properties and creates
	 *                 JSON output that should NOT be used to deserialize an object from.
	 */
	JSONSerializer(bool readOnly = false, OutputFormat format = STANDARD) :
		mReadOnly(readOnly), mOutputFormat(format) {}

	/// serialize value and return its json representation
	template <typename T>
	json::Value serialize(const T& value)
	{
		mFirstComplex = true;
		mValue = json::Value();
		Base::serialize("Value", value);
		return mValue;
	}

	json::Value serialize(const json::Value& value)
	{
		return value;
	}

	typedef Base::VersionType VersionType;

	template <typename T>
	VersionType version(VersionType version, const T* caller = NULL)
	{
		return this->version<T>(version, false);
	}

	MIRA_DEPRECATED("Please call as version<MyType>(v) or version(v, this)",
	VersionType version(VersionType version))
	{
		return versionLegacy(version);
	}

	template <typename T>
	VersionType version(VersionType version, AcceptDesiredVersion, const T* caller = NULL)
	{
		return this->version<T>(version, true);
	}

private:

	template <typename T>
	VersionType version(VersionType version, bool acceptDesiredVersion)
	{
		version = this->template queryDesiredClassVersion<T>(version, acceptDesiredVersion);

		int vf = this->forcedSerializeVersion();
		if ((vf == 0) || (vf == 1))
			return versionLegacy(version);

		static const std::string v = std::string("@version[") + typeName<T>() + "]";
		mValue.get_obj()[v] = json::Value(json::cast(version));
		return version;
	}

	VersionType versionLegacy(VersionType version)
	{
		mValue.get_obj()["@version"] = json::Value(json::cast((int)version));
		return version;
	}

public:

	template<typename T>
	void atomic(T& member)
	{
		// member of a currently serialized object -> add member
		if ( mValue.type() == json_spirit::obj_type )
			mValue.get_obj()[getCurrentMemberMeta().id] = json::Value(json::cast(member));
		// item of a currently serialized array -> add item
		else if ( mValue.type() == json_spirit::array_type )
			mValue.get_array().push_back(json::Value(json::cast(member)));
		// no currently serialized value -> set member as new value
		else
			mValue = json::Value(json::cast(member));
	}

	void atomic(json::Value& member)
	{
		// member of a currently serialized object -> add member
		if ( mValue.type() == json_spirit::obj_type )
			mValue.get_obj()[getCurrentMemberMeta().id] = member;
		// item of a currently serialized array -> add item
		else if ( mValue.type() == json_spirit::array_type )
			mValue.get_array().push_back(member);
		// no currently serialized value -> set member as new value
		else
			mValue = member;
	}


	template<typename T>
	void object(T& member)
	{
		// in json the outer most object (the root object) has no name
		if ( mFirstComplex )
		{
			mValue = json::Object();
			mFirstComplex = false;
			if(!mPointerClass.empty()) {
				mValue.get_obj()["@class"] = json::Value(mPointerClass);
				mPointerClass.clear();
			}
			Base::object(member);
		}
		else
		{
			json::Value oldValue = mValue;
			mValue = json::Object();
			if(!mPointerClass.empty()) {
				mValue.get_obj()["@class"] = json::Value(mPointerClass);
				mPointerClass.clear();
			}
			Base::object(member);
			if ( oldValue.type() == json_spirit::obj_type )
				oldValue.get_obj()[getCurrentMemberMeta().id] = mValue;
			else if ( oldValue.type() == json_spirit::array_type )
					oldValue.get_array().push_back(mValue);
			mValue = oldValue;
		}
	}

	template<typename T>
	void collection(T& member)
	{
		collectionInternal(member);
	}

	template<typename mapped_type>
	void collection(std::map<std::string, mapped_type>& member)
	{
		if (mOutputFormat & STRING_MAP_AS_OBJECT)
			object(member);
		else
			collectionInternal(member);
	}

	void pointerReference(int referencedObjectID) {
		json::Value val = json::Object();
		val.get_obj()["@ref"] = json::Value(this->getHumanReadableFullID(referencedObjectID));
		if (mValue.type() == json_spirit::obj_type )
			mValue.get_obj()[getCurrentMemberMeta().id] = val;
		else if (mValue.type() == json_spirit::array_type )
			mValue.get_array().push_back(val);
	}

	void pointerWithClassType(const std::string& type) {
		mPointerClass = type;
	}

	void pointerNull() {
		if ( mValue.type() == json_spirit::obj_type )
			mValue.get_obj()[getCurrentMemberMeta().id] = json::Value();
		else if ( mValue.type() == json_spirit::array_type )
			mValue.get_array().push_back(json::Value());
		else
			mValue = json::Value();
	}

	template<typename T>
	void roproperty(const char* name, const T& member, const char* comment,
	                PropertyHint&& hint = PropertyHint(),
	                ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		if (!mReadOnly)
			return;
		// cast away constness, this is okay, since read only properties we
		// will never write to the data
		property(name, const_cast<T&>(member),
		         comment, std::move(hint), flags);
	}

	template<typename T>
	void roproperty(const char* name, const std::string& id, const T& member,
	                const char* comment, PropertyHint&& hint = PropertyHint(),
	                ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		if (!mReadOnly)
			return;
		// cast away constness, this is okay, since read only properties we
		// will never write to the data
		property(name, id, const_cast<T&>(member),
		         comment, std::move(hint), flags);
	}

	template<typename T>
	void roproperty(const char* name, Getter<T> getter, const char* comment,
	                PropertyHint&& hint = PropertyHint(),
	                ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		if (!mReadOnly)
			return;
		auto a = makeAccessor(getter, NullSetter<T>());
		property(name, a, comment, std::move(hint), flags);
	}

public:

	template<typename Container>
	void mapEntry(int id, const typename Container::value_type& p)
	{
		typedef typename Container::mapped_type type;

		if (mOutputFormat & STRING_MAP_AS_OBJECT) {
			type& nonconstv = const_cast<type&>(p.second);
			member(p.first.c_str(), nonconstv, "");
		} else {
			serialization::reflectReadMapPair<JSONSerializer, Container>(*this, "item", id, p);
		}
	}

private:
	
	template<typename T>
	void collectionInternal(T& member)
	{
		if ( mFirstComplex )
		{
			mValue = json::Array();
			mFirstComplex = false;
			Base::object(member);
		}
		else
		{
			json::Value oldValue = mValue;
			mValue = json::Array();
			Base::object(member);
			if ( oldValue.type() == json_spirit::obj_type )
				oldValue.get_obj()[getCurrentMemberMeta().id] = mValue;
			else
				if ( oldValue.type() == json_spirit::array_type )
					oldValue.get_array().push_back(mValue);
			mValue = oldValue;
		}
	}

private:
	bool mFirstComplex;
	bool mReadOnly;
	json::Value mValue;
	std::string mPointerClass;

	OutputFormat mOutputFormat;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 *
 * Deserializer for serializing objects from JSON format.
 *
 * Example usage:
 * \code
 * // load the json content
 * json::Value v;
 * json::read(..., v);
 *
 * // create the JSON deserializer
 * JSONDeserializer deserializer(v);
 *
 * // the object that will be filled with the content
 * std::vector<int> myValue;
 *
 * // deserialize the value
 * deserializer.deserialize(myValue);
 *
 * // myValue is now filled with the content that was stored in the JSON value
 * \endcode
 *
 * For the serialization of JSON content see
 * @ref mira::JSONSerializer "JSONSerializer".
 *
 * @see @ref SerializationPage, JSONSerializer
 */
class JSONDeserializer : public Deserializer<JSONDeserializer>
{
	typedef Deserializer<JSONDeserializer> Base;

public:

	/// Flags for JSON input format
	enum InputFormat
	{
		/// Default format
		STANDARD               = 0x00,

		/**
		 * Accept associative containers with string keys as JSON object (keys = member names),
		 * in ADDITION to the standard array format (i.e. the deserializer will understand BOTH formats)
		 */
		ACCEPT_STRING_MAP_AS_OBJECT   = 0x01
	};

	MIRA_ENUM_TO_FLAGS_INCLASS(InputFormat) // generate logical operators for enum values

	JSONDeserializer(const json::Value& value, InputFormat format = STANDARD) :
		mValue(&value), mStringMapAsObject(false), mInputFormat(format)
	{
	}

	template <typename T>
	void deserialize(T& value)
	{
		mIndex = 0;
		mFirstComplex = true;
		Base::deserialize("Value",value);
	}

	void deserialize(json::Value& value)
	{
		value = *mValue;
	}

	const json::Value* getMember(const json::Value* value, const std::string& name)
	{
		if (value->get_obj().count(name) == 0)
			MIRA_THROW(XMemberNotFound, "Member '" << name << "' is missing"); // hence we have to abort with an exception
		return &value->get_obj().at(name);
	}
	
	const json::Value* getItem(const json::Value* value, std::size_t index)
	{
		if (value->get_array().size() <= index)
			MIRA_THROW(XMemberNotFound, "Item[" << index << "] is missing"); // hence we have to abort with an exception
		return &value->get_array()[index];
	}

	typedef Base::VersionType VersionType;

	template <typename T>
	VersionType version(VersionType expectedVersion, const T* object = NULL)
	{
		int vf = this->forcedDeserializeVersion();
		if ((vf == 0) || (vf == 1))
			return versionLegacy(expectedVersion);

		static const std::string v = std::string("@version[") + typeName<T>() + "]";
		return getVersion<T>(expectedVersion, v);
	}

	MIRA_DEPRECATED("Please call as version<MyType>(v) or version(v, this)",
	VersionType version(VersionType expectedVersion))
	{
		return versionLegacy(expectedVersion);
	}

	template <typename T>
	VersionType version(VersionType expectedVersion, AcceptDesiredVersion, const T* caller = NULL)
	{
		return this->version<T>(expectedVersion);
	}

private:

	VersionType versionLegacy(VersionType expectedVersion)
	{
		return getVersion<void>(expectedVersion, "@version");
	}

	template <typename T>
	VersionType getVersion(VersionType expectedVersion, const std::string& versionElement)
	{
		if (mValue->get_obj().count(versionElement) == 0)
			return 0;
		VersionType version = mValue->get_obj().at(versionElement).get_value<json::TypeTrait<int>::type >();
		if (version > expectedVersion) {
			MIRA_LOG(WARNING) << "Trying to deserialize JSON data of a newer version (" << (int)version <<
				") of type " << typeName<T>() << " into an older version (" << (int)expectedVersion << ").";
		}
		return version;
	}

public:

	template<typename T>
	void atomic(T& member)
	{
		if ( mValue->type() == json_spirit::obj_type )
		{
			try
			{
				member = json::reverse_cast<T>(getMember(mValue, mCurrentMemberID)->get_value<typename json::TypeTrait<T>::type >());
			}
			catch(std::exception& ex)
			{
				MIRA_THROW(XMemberNotFound, "Error deserializing member '" << mCurrentMemberID
				           << "' (" << getCurrentMemberFullID()
				           << ") from object='" << json::write(*mValue)
				           << "' as " << typeName<T>() << ": " << ex.what());
			}
		}
		else if ( mValue->type() == json_spirit::array_type )
		{
			try
			{
				member = json::reverse_cast<T>(getItem(mValue, mIndex++)->get_value<typename json::TypeTrait<T>::type >());
			}
			catch(std::exception& ex)
			{
				MIRA_THROW(XMemberNotFound, "Error deserializing item[" << mIndex-1
				           << "] (" << getCurrentMemberFullID()
				           << ") from collection='" << json::write(*mValue)
				           << "' as " << typeName<T>() << ": " << ex.what());
			}
		}
		else
		{
			try
			{
				member = json::reverse_cast<T>(mValue->get_value<typename json::TypeTrait<T>::type>());
			}
			catch(std::exception& ex)
			{
				MIRA_THROW(XMemberNotFound, "Error deserializing atomic (" << getCurrentMemberFullID()
				           << ") with content='" << json::write(*mValue) << "' as " << typeName<T>() << ": " << ex.what());
			}
		}
	}

	void atomic(json::Value& member)
	{
		if ( mValue->type() == json_spirit::obj_type )
			member = *getMember(mValue, mCurrentMemberID);
		else if ( mValue->type() == json_spirit::array_type )
			member = *getItem(mValue, mIndex++);
		else
			member = *mValue;
	}

	template<typename T>
	void object(T& member)
	{
		if ( mFirstComplex )
		{
			mFirstComplex = false;
			Base::object(member);
			return;
		}
		const json::Value* oldValue = mValue;
		std::size_t oldIndex = mIndex;
		if ( mValue->type() == json_spirit::obj_type )
		{
			mValue = getMember(mValue, mCurrentMemberID);
		}
		else if ( mValue->type() == json_spirit::array_type )
		{
			mValue = getItem(mValue, oldIndex);
			mIndex = 0;
		}

		Base::object(member);

		mValue = oldValue;
		mIndex = oldIndex+1;
	}

	template<typename T>
	void collection(T& member)
	{
		collectionInternal(member);
	}

	template<typename mapped_type>
	void collection(std::map<std::string, mapped_type>& member)
	{
		if (!(mInputFormat & ACCEPT_STRING_MAP_AS_OBJECT)) {
			collectionInternal(member);
			return;
		}

		if (mStringMapAsObject) {
			json::Object::const_iterator oldCurrentStringMapMember = mCurrentStringMapMember;
			object(member);
			mCurrentStringMapMember = oldCurrentStringMapMember;
			return;
		}

		assert(mStringMapAsObject == false);

		// backup deserializer state
		const json::Value* oldValue = mValue;
		std::size_t oldIndex = mIndex;
		std::string memberID = mCurrentMemberID;
		bool firstComplex = mFirstComplex;

		ObjectNameToInstanceMap trackingNames = mObjectNameToInstance;
		ObjectsVector trackingInstances = mObjects;

		try {
			try {
				collectionInternal(member);
			}
			catch (XInvalidConfig& ex) {
				MIRA_LOG(DEBUG) << "Deserialize map<string, T> as array failed, try as object:" << std::endl;

				// reset deserializer state to state before calling collection(), and try again, calling object()
				mObjectNameToInstance = trackingNames;
				mObjects = trackingInstances;

				mValue = oldValue;
				mIndex = oldIndex;
				mCurrentMemberID = memberID;
				mFirstComplex = firstComplex;

				MIRA_LOG(DEBUG) << json::write(*mValue) << std::endl;

				mStringMapAsObject = true;
				object(member);
			}
		}
		// make sure we get to reset mStringMapAsObject as final statement on our way "up"
		catch(Exception& ex) {
			mStringMapAsObject = false;
			MIRA_RETHROW(ex, "While trying to deserialize map<string, T> as object.");
		}
		catch(...) {
			mStringMapAsObject = false;
			throw;
		}

		mStringMapAsObject = false;
	}

	const json::Array* getCollection()
	{
		const json::Array* array;
		try
		{
			array = &mValue->get_array();
		}
		catch(std::exception& ex)
		{
			MIRA_THROW(XInvalidConfig, "Error deserializing collection (" << getCurrentMemberFullID()
			           << ") from json::Value='" << json::write(*mValue) << "': " << ex.what());
		}
		return array;
	}

	template<typename T>
	void pointer(T* &pointer)
	{
		const json::Value* value;
		if ( mFirstComplex )
			value = mValue;
		else
		{
			if ( mValue->type() == json_spirit::obj_type )
				value = getMember(mValue, mCurrentMemberID);
			else if ( mValue->type() == json_spirit::array_type )
				value = getItem(mValue, mIndex);
		}

		std::size_t oldIndex = mIndex;

		// do we have a null pointer?
		if(value->is_null()) {
			pointer = NULL;
		}
		// do we have a reference?
		else {
			assert(value->type() == json_spirit::obj_type);
			auto it = value->get_obj().find("@ref");
			if(it!=value->get_obj().end()) {
				// we have a reference, so resolve it
				pointer = resolveReference<T>(it->second.get_value<std::string>());
			} else {
				// we have a "normal" pointer, so deserialize it
				Base::pointer(pointer);
			}
		}

		if (mValue->type() == json_spirit::array_type)
			mIndex = oldIndex+1;
	}

	std::string pointerClassType()
	{
		const json::Value* value;
		if ( mFirstComplex )
			value = mValue;
		else
		{
			if ( mValue->type() == json_spirit::obj_type )
			{
				try
				{
					value = getMember(mValue, mCurrentMemberID);
				}
				catch(XMemberNotFound& ex)
				{
					MIRA_RETHROW(ex, "Error getting pointer class type of member '"
					             << mCurrentMemberID << "' (" << getCurrentMemberFullID()
					             << ") from object='" << json::write(*mValue)<< "'");
				}
			}
			else if ( mValue->type() == json_spirit::array_type )
			{
				try
				{
					value = getItem(mValue, mIndex);
				}
				catch(XMemberNotFound& ex)
				{
					MIRA_RETHROW(ex, "Error getting pointer class type of item[" << mIndex
					             << "] (" << getCurrentMemberFullID()
					             << ") from collection='" << json::write(*mValue) << "'");
				}
			}
		}
		if(value->type() == json_spirit::obj_type)
		{
			auto it = value->get_obj().find("@class");
			if(it!=value->get_obj().end())
				return it->second.get_value<std::string>();
		}
		return "Unknown Class";
	}

	// hijack the invokeOverwrite method for adding a starting and closing tag
	// around the serialized data
	template<typename T>
	void invokeMemberOverwrite(T& member, const ReflectMemberMeta& meta)
	{
		std::string oldID = mCurrentMemberID;
		if(meta.id!=NULL)
			mCurrentMemberID = meta.id;
		try
		{
			Base::invokeMemberOverwrite(member, meta);
		}
		catch(...)
		{
			if(meta.id!=NULL)
				mCurrentMemberID = oldID;
			throw;
		}
		if(meta.id!=NULL)
			mCurrentMemberID = oldID;
	}

public:

	uint32 getStringMapElementCount()
	{
		if (mStringMapAsObject) {
			try {
				const json::Object* o = &mValue->get_obj();
				mCurrentStringMapMember = mValue->get_obj().begin();
				return (uint32)o->size();
			}
			catch(std::exception& ex)
			{
				MIRA_THROW(XInvalidConfig, "Error counting object elements from json::Value='"
				                            << json::write(*mValue) << "': " << ex.what());
			}
		} else {
			const json::Array* a = getCollection();
			return (uint32)a->size();
		}
	}

	template<typename Container>
	void mapEntry(int id, Container& c, typename Container::iterator& hint)
	{
		typedef typename Container::value_type value_type;
		typedef typename Container::mapped_type mapped_type;

		if (mStringMapAsObject) {
			std::string key = mCurrentStringMapMember->first;
			hint = c.insert(hint, value_type(key, mapped_type()));
			property(key.c_str(), hint->second, "");
			++mCurrentStringMapMember;
		} else {
			serialization::reflectWriteMapPair(*this, "item", "key", id, c, hint);
		}
	}

private:
	template<typename T>
	void collectionInternal(T& member)
	{
		if ( mFirstComplex )
		{
			mFirstComplex = false;
			Base::object(member);
			return;
		}
		const json::Value* oldValue = mValue;
		std::size_t oldIndex = mIndex;
		if ( mValue->type() == json_spirit::obj_type )
		{
			mValue = getMember(mValue, mCurrentMemberID);
			mIndex = 0;
		}
		else if ( mValue->type() == json_spirit::array_type )
		{
			mValue = getItem(mValue, oldIndex);
			mIndex = 0;
		}

		Base::object(member);

		mValue = oldValue;
		mIndex = oldIndex+1;
	}

private:
	bool mFirstComplex;
	std::size_t mIndex;
	const json::Value* mValue;
	std::string mCurrentMemberID;

	bool mStringMapAsObject;
	json::Object::const_iterator mCurrentStringMapMember;

	InputFormat mInputFormat;
};

namespace serialization { // our private namespace

///////////////////////////////////////////////////////////////////////////////

/**
 * Specialization for JSONSerializer which does not write the
 * item count explicitly. The deserializer will count the
 * item nodes in the parent node to recover the item count, which
 * is much more user friendly, since the user does not need
 * to provide the count himself.
 */
template<typename Collection>
struct ReflectCollectionCount<JSONSerializer, Collection>
{
	static void reflect(JSONSerializer& r, uint32& ioCount)
	{
		// do nothing
	}
};
/**
 * Specialization for JSONDeserializer which counts the
 * item nodes in the parent node to recover the item count, which
 * is much more user friendly, since the user does not need
 * to provide the count himself.
 */
template<typename Collection>
struct ReflectCollectionCount<JSONDeserializer, Collection>
{
	static void reflect(JSONDeserializer& r, uint32& ioCount)
	{
		const json::Array* a = r.getCollection();
		ioCount = (uint32)a->size();
	}
};

template<typename mapped_type>
struct ReflectCollectionCount<JSONDeserializer, std::map<std::string, mapped_type>>
{
	static void reflect(JSONDeserializer& r, uint32& ioCount)
	{
		ioCount = r.getStringMapElementCount();
	}
};

template<typename mapped_type>
struct ReflectReadMapItems<JSONSerializer, std::map<std::string, mapped_type>>
{
	typedef std::map<std::string, mapped_type> Container;
	typedef typename Container::value_type     value_type;

	static void reflect(JSONSerializer& r, Container& c)
	{
		// store each item
		int id=0;
		foreach(value_type& p, c)
		{
			r.mapEntry<Container>(id, p);
			++id;
		}
	}
};

template<typename mapped_type>
struct ReflectWriteMapItems<JSONDeserializer, std::map<std::string, mapped_type>>
{
	typedef std::map<std::string, mapped_type> Container;
	typedef typename Container::iterator iterator;

	static void reflect(JSONDeserializer& r, Container& c, uint32 count)
	{
		iterator hint = c.begin();

		// restore each item and insert into the map
		for(uint32 id=0; id<count; ++id)
			r.mapEntry<Container>(id, c, hint);
	}
};

///////////////////////////////////////////////////////////////////////////////

}

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
