/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file BinaryRPCBackend.h
 *    Provides binary client and server side requests and responses.
 *
 * @author Erik Einhorn
 * @date   2010/11/07
 */

#ifndef _MIRA_BINARYRPCBACKEND_H_
#define _MIRA_BINARYRPCBACKEND_H_

#ifndef Q_MOC_RUN
#include <boost/thread/future.hpp>
#endif
#include <serialization/BinarySerializer.h>
#include <rpc/RPCError.h>

#include <thread/Atomic.h>

namespace mira {

///@cond INTERNAL
template <typename R>
inline void binaryRPCreturnHelper(boost::promise<R>& promise, BinaryBufferDeserializer& in)
{
	R value;
	in.deserialize(value);
	promise.set_value(std::move(value));
}

template <>
inline void binaryRPCreturnHelper(boost::promise<void>& promise, BinaryBufferDeserializer& in)
{
	promise.set_value();
}

template <uint8 BinaryFormatVersion>
struct binaryRPCdeserializeHelper;

///@endcond INTERNAL



/**
 * @ingroup RPCModule
 * Provides binary client and server side requests and responses.
 *
 *
 *
 * The RPC Request contains the following serialized items:
 * \verbatim
  	  1 byte=27        // marks a binary request
  	  string callId
  	  string service
  	  RPCSignature
      Parameter 1
      ...
      Parameter N
 * \endverbatim
 *
 * The RPC Response contains the following serialized items:
 * \verbatim
  	  1 byte=27        // marks a binary response
  	  string callId
      ReturnStatus::ReturnStatus status

    depending on status:
      [status == RETURN_VOID]
       .

      [status == RETURN_RESULT]
       Return value

      [status == RETURN_EXCEPTION]
       string exceptionMessage
 * \endverbatim
 */
template <uint8 BinaryFormatVersion>
class BinaryRPCBackendTempl
{
public:
	// must be a non-printable character that is not used by JSON-RPC or other RPC protocols
	// it is used to identify the requests and responses as binary ones
	enum {
		BINARY_RPC_MARKER = 27 // Escape Character
	};

	typedef Buffer<uint8> ContainerType;

	//////////////////////////////////////////////////////////////////////////////

	/**
	 * @ingroup RPCModule
	 *
	 * Binary client-side response.
	 * The response must have been created by the RPCServer and it must be
	 * contained in the buffer that is passed in the constructor.
	 *
	 * @see @ref RPC_Backends
	 */
	class ClientResponse
	{
	public:
		enum ReturnStatus {
			RETURN_VOID=0,
			RETURN_RESULT,
			RETURN_EXCEPTION,
		};
	public:

		ClientResponse(ContainerType* buffer) : mIn(buffer) {}

		/**
		 * Read the response header (i.e. the call id of the RPC call)
		 * from the response. This method is called first by the RPCClient when a
		 * response is handled
		 */
		void getHeader(std::string& callId)
		{
			uint8 marker;
			mIn.deserialize(marker, false);
			if(marker!=BINARY_RPC_MARKER)
				MIRA_THROW(XRuntime, "The buffer does not contain a valid binary response");

			mIn.deserialize(callId);
		}

		/**
		 * Obtain return value from response and set it using promise.set_value()
		 * or set exception using promise.set_exception().
		 * See documentation of boost::promise and boost::future for details about
		 * futures and promises.
		 * This method is called after getHeader().
		 */
		template<typename R>
		void getReturn(boost::promise<R>& promise)
		{
			ReturnStatus status;
			binaryRPCdeserializeHelper<BinaryFormatVersion>::deserialize(status, mIn);
			switch ( status )
			{

				case RETURN_VOID :
				case RETURN_RESULT:
					binaryRPCreturnHelper(promise, mIn);
					break;
				case RETURN_EXCEPTION : {
					std::string error;
					mIn.deserialize(error);
					XRPC ex(error);
					CallStack stack;
					ThreadID thread;
					try {
						mIn.deserialize(stack);
						mIn.deserialize(thread);
						ex.addExternalStackInfo<XRPC>(stack, thread);

						// try to deserialize and add the original exception
						SerializableException* origEx;
						mIn.deserialize(origEx);
						ex.setOrigException(origEx);
					}
					catch(...) {
						ex.addInfo("There is no stack trace from the RPC method or it is unreadable.");
					}
					promise.set_exception(boost::copy_exception(ex));
					break;
				}
				default :
				{
					// we should never reach here
					MIRA_THROW(XLogical, "Unknown return status.");
				}
			}
		}


	private:
		BinaryBufferDeserializer mIn;
	};

	/**
	 * @ingroup RPCModule
	 *
	 * Binary client-side request.
	 * The request is written into the buffer that is passed in the constructor.

	 * @see @ref RPC_Backends
	 */
	class ClientRequest
	{
	public:
		ClientRequest(ContainerType* buffer) : mOut(buffer) {}

		/**
		 * Generates a unique call ID for a client request of this backend.
		 * As call id a random id is generated.
		 */
		std::string generateCallID() {
			static uint32 sID = 0;
			return toString(atomic::inc(&sID));
		}

		/**
		 * Write the request header consisting of the call ID, the service name
		 * and the signature of the requested method.
		 * This method is called first by the RPCClient when a new request is
		 * generated.
		 */
		void setHeader(const std::string& callId, const std::string& service,
		               const RPCSignature& signature)
		{
			uint8 marker = BINARY_RPC_MARKER;
			mOut.serialize(marker, false);

			mOut.serialize(callId);
			mOut.serialize(service);
			mOut.serialize(signature);
		}

		/**
		 * Write the value of the next parameter to the request.
		 * This method is called multiple times after the setHeader() method for
		 * each parameter that is passed to the RPCClient::call() method.
		 */
		template <typename P>
		void setParameter(const P& param) {
			mOut.serialize(param);
		}

	private:
		ConcreteBinarySerializer<BinaryBufferOstream, BinaryFormatVersion> mOut;
	};

	/**
	 * @ingroup RPCModule
	 *
	 * Binary server-side response.
	 * The response is written to the buffer that is passed in the constructor.

	 * @see @ref RPC_Backends
	 */
	class ServerResponse
	{
	public:

		ServerResponse(ContainerType* buffer) : mOut(buffer) {}

		/**
		 * Write the response header consisting of the ID of the call.
		 * This method is called first by the RPCServer to create a response
		 * for a finished RPC call.
		 * It will be followed by a call of either returnException(), returnResult()
		 * or returnVoid().
		 */
		void setHeader(const std::string& callId)
		{
			uint8 marker = BINARY_RPC_MARKER;
			mOut.serialize(marker, false);

			mOut.serialize(callId);
		}

		/**
		 * Write exception as result of an RPC call.
		 * This method is called after setHeader() in case of an error or
		 * exception while processing the call.
		 */
		void returnException(RPCError reason, const std::string& message) {
			mOut.serialize(ClientResponse::RETURN_EXCEPTION);
			mOut.serialize(message);
		}

		/**
		 * Write exception as result of an RPC call.
		 * This method is called after setHeader() in case of an error or
		 * exception while processing the call.
		 */
		void returnException(RPCError reason, const std::string& message,
		                     const CallStack& callstack)
		{
			mOut.serialize(ClientResponse::RETURN_EXCEPTION);
			mOut.serialize(message);
			mOut.serialize(callstack);
			mOut.serialize(getCurrentThreadID());
		}

		/**
		 * Write exception as result of an RPC call.
		 * This method is called after setHeader() in case of an error or
		 * exception while processing the call.
		 */
		void returnException(RPCError reason, SerializableException& ex)
		{
			mOut.serialize(ClientResponse::RETURN_EXCEPTION);
			mOut.serialize(std::string(ex.what()));
			mOut.serialize(ex.callStack());
			mOut.serialize(getCurrentThreadID());
			mOut.serialize(&ex);
		}

		/**
		 * Write result of an RPC call.
		 * This method is called after setHeader() in case of a successful
		 * RPC call that returns a value.
		 */
		template<typename R>
		void returnResult(const R& res) {
			mOut.serialize(ClientResponse::RETURN_RESULT);
			mOut.serialize(res);
		}

		/**
		 * Write successful end of an RPC call.
		 * This method is called after setHeader() in case of a successful
		 * RPC call that does NOT return a value.
		 */
		void returnVoid() {
			mOut.serialize(ClientResponse::RETURN_VOID);
		}

	private:
		ConcreteBinarySerializer<BinaryBufferOstream, BinaryFormatVersion> mOut;
	};

	/**
	 * @ingroup RPCModule
	 *
	 * Binary server-side request.
	 * The request must have been created by the RPCClient and it must be
	 * contained in the buffer that is passed in the constructor.
	 *
	 * @see @ref RPC_Backends
	 */
	class ServerRequest
	{
	public:

		ServerRequest(ContainerType* buffer) : mIn(buffer) {}

	public:
		/**
		 * Read the request header including the call ID, the service name
		 * and the signature of the method.
		 * The call ID and the service name must be returned via the by-reference
		 * parameters. The signature should be stored internally to be able to
		 * handle the methods below.
		 * This method is called first when a new request is handled by the RPCServer.
		 */
		void getHeader(std::string& oCallId, std::string& oService)
		{
			uint8 marker;
			mIn.deserialize(marker, false);
			if(marker!=BINARY_RPC_MARKER)
				MIRA_THROW(XRuntime, "The buffer does not contain a valid binary response");

			mIn.deserialize(oCallId);
			mIn.deserialize(oService);
			mIn.deserialize(mSignature);
		}

		/**
		 * Check, if the passed signature is compatible with the signature that was
		 * read from the request header in the previous getHeader() call above.
		 */
		bool checkSignature(const RPCSignature& signature)
		{
			return signature == mSignature;
		}

		/**
		 * Return the signature, that was read from the request header in the previous
		 * getHeader() call above.
		 */
		const RPCSignature& getSignature()
		{
			return mSignature;
		}

		/**
		 * Read and deserializes the next parameter from the request.
		 * This method is called multiple times for each parameter of the RPC method.
		 */
		template <typename P>
		void getParameter(P& oParam)
		{
			// no need to do type check here, since it is done by BinaryBufferDeserializer
			mIn.deserialize(oParam);
		}

	private:
		BinaryBufferDeserializer mIn;
		RPCSignature mSignature;
	};

};

typedef BinaryRPCBackendTempl<0> BinaryRPCBackendLegacy;
typedef BinaryRPCBackendTempl<2> BinaryRPCBackend;

///@cond INTERNAL
template <uint8 BinaryFormatVersion>
struct binaryRPCdeserializeHelper
{

	static void deserialize(typename BinaryRPCBackendTempl<BinaryFormatVersion>::ClientResponse::ReturnStatus& status,
                            BinaryBufferDeserializer& deserializer)
	{
		deserializer.deserialize(status);
	}
};

template <>
struct binaryRPCdeserializeHelper<0>
{
	static void deserialize(BinaryRPCBackendTempl<0>::ClientResponse::ReturnStatus& status,
                            BinaryBufferDeserializer& deserializer)
	{
		// need to read the typename separately and suppress the type check
		// (because the type's name is NOT the same anymore)
		std::string fulltypename;
		deserializer.deserialize(fulltypename, false);

		assert(fulltypename == "mira::BinaryRPCBackend::ClientResponse::ReturnStatus");

		deserializer.deserialize(status, false);
	}
};

///@endcond INTERNAL

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_BINARYRPCBACKEND_H_ */
