/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ManifestAgent.h
 *    ManifestAgent manages loading of manifest files.
 *
 * @author Ronny Stricker
 * @date   2011/02/24
 */

#ifndef _MIRA_MANIFESTAGENT_H_
#define _MIRA_MANIFESTAGENT_H_

#include <vector>

#include <platform/Platform.h>
#include <factory/LibraryRegistry.h>
#include <factory/VacantClass.h>
#include <utils/Path.h>

#include <error/LoggingCore.h>
#include <utils/PathFinder.h>
#include <utils/Foreach.h>
#include <serialization/adapters/boost/shared_ptr.hpp>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * @brief Simple class encapsulation parent and class information stored in
 * a "real" class object. This class is used for loading manifest files.
 */
struct MIRA_BASE_EXPORT ManifestClassInfo {
	boost::shared_ptr<VacantClass>	mClass;
	std::vector< std::string >	mParents;

	bool operator ==( ManifestClassInfo const& other ) {
		return other.mClass == mClass;
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member( "Class", mClass, "" );
		r.member( "Parents", mParents, "" );
	}
};

//////////////////////////////////////////////////////////////////////////////

/**
 * @brief Class for managing manifest files.
 * The class should be used to load and store manifest files.
 * Please note that you should use one instance of the ManifestAgent to write
 * one manifest file. However, you can load as many manifest files as you want
 * with a single instance. Don't forget to call finalize after all manifest
 * files have been loaded.
 */
class MIRA_BASE_EXPORT ManifestAgent {
public:

	ManifestAgent() {}

	/**
	 * @brief Set the library name which will be added to the ManifestClassInfo
	 * of every class (added with addClasses())
	 */
	ManifestAgent( std::string const& libraryName );

	/**
	 * @brief Set the library name and calls addClasses() for the given classes.
	 */
	ManifestAgent( std::string const& libraryName,
	               std::vector< ClassProxy > const& classes );

	/**
	 * @brief Set version of library
	 */
	void setLibraryVersion( int major, int minor, int patch );

	/**
	 * @brief Add the given classes to the manifest agent
	 * The functions generates a ManifestClassInfo for every class and stores
	 * the information in the mClasses vector.
	 */
	void addClasses( std::vector< ClassProxy > const& classes );

	/**
	 * @brief Load a single manifest file and announce the contained classes
	 * to the class factory.
	 * The parents are not directly announced. Therefore, finalize() have to
	 * be called if one or more files are loaded.
	 * The ManifestAgent assumes that the library and manifest files resides
	 * at the same location. Therefore, the parent path of filePath is appended
	 * to the library name to load the library later on.
	 */
	void loadFile( Path const& filePath );

	/**
	 * @brief Serialize the mClasses vector to the given file.
	 */
	void saveToFile( Path const& filePath );

	/**
	 * @brief Register the parent - child relationships of the previously
	 * loaded files.
	 */
	void finalize();

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member( "LibraryName", mLibraryName, "" );
		r.member( "LibraryVersion", mVersion, "" );
		r.member( "Classes", mClasses, "" );
	}

protected:
	std::vector< ManifestClassInfo > mClasses;
	std::string mLibraryName;
	LibraryVersion mVersion;
};



#ifdef MIRA_WINDOWS
#  define MIRA_LIB_FOLDER "bin"
#else
#  define MIRA_LIB_FOLDER "lib"
#endif

/**
 * Loads all manifest files that match the given file pattern within all
 * paths that are specified in the MIRA_PATH environment variable.
 */
inline void loadManifests(const Path& pattern = MIRA_LIB_FOLDER"/*.manifest")
{
	std::vector<Path> manifests = findProjectFiles(pattern);
	ManifestAgent manifestAgent;
	foreach(const auto& p, manifests)
	{
		try
		{
			manifestAgent.loadFile(p);
		} catch(XIO& ex)
		{
			// catch IO exceptions only (e.g. file not found, etc)
			// all other exceptions concerning loading manifests are critical !!!
			MIRA_LOG(ERROR) << "Failed to load manifest '" << p << "': " << ex.message();
		} catch(Exception& ex) {
			MIRA_RETHROW(ex, "in file '" << p.string() << "'");
		}
	}
	manifestAgent.finalize();
}

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_MANIFESTAGENT_H_ */

