/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LoggingCore.h
 *    Core class of the logging library.
 *
 * @author Tim Langner
 * @date   2010/07/06
 */

#ifndef _MIRA_LOGGINGCORE_H_
#define _MIRA_LOGGINGCORE_H_

#include <map>

#ifndef Q_MOC_RUN
#include <boost/thread/mutex.hpp>
#include <boost/function.hpp>
#endif

#include <platform/Types.h>
#include <thread/ThreadID.h>
#include <utils/Singleton.h>
#include <utils/Time.h>
#include <utils/Foreach.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * Severity levels to graduate between different log outputs
 * @ingroup LoggingModule
 */
enum SeverityLevel
{
	CRITICAL = 0,
	ERROR = 1,
	WARNING = 2,
	NOTICE = 3,
	DEBUG = 4,
	TRACE = 5,
};

/**
 * String conversion for the enum severity types
 */
const std::string severityLevelStr[]= {
	"CRITICAL",
	"ERROR   ",
	"WARNING ",
	"NOTICE  ",
	"DEBUG   ",
	"TRACE   "
};

/**
 * Converts the specified string into a numeric severity level.
 * @throws XInvalidParameter, if the string does not name a valid level.
 * @ingroup LoggingModule
 */
MIRA_BASE_EXPORT SeverityLevel stringToSeverityLevel(const std::string& levelString);

//////////////////////////////////////////////////////////////////////////////

/**
 * Holds all the information about a log entry
 * @ingroup LoggingModule
 */
struct LogRecord
{
	SeverityLevel level;
	Time time;
	Duration uptime;
	std::string message;
	std::string details;
	int line;
	std::string file;
	std::string function;
	ThreadID threadID;
};

/**
 * Holds information about a log entry function.
 * Can be generated by the LogCore::parseFunction(LogRecord&) function
 * @ingroup LoggingModule
 */
struct LogRecordFunctionInfo
{
	std::string nameSpace;
	std::string className;
	std::string functionName;
};

/**
 * Non-intrusive reflector for LogRecord. Needed because we write LogRecords to
 * a channel.
 */
template<typename Reflector>
void reflect(Reflector& r, LogRecord& record)
{
	r.member("level", record.level, "Log level");
	r.member("time", record.time, "Log time");
	r.member("uptime", record.uptime, "Log uptime");
	r.member("message", record.message, "Log message");
	r.member("line", record.line, "Line in file where log call occurred");
	r.member("file", record.file, "File where log call occurred");
	r.member("function", record.function, "Function where log call occurred");
	r.member("threadID", record.threadID, "ID of thread where log call occurred");
}

//////////////////////////////////////////////////////////////////////////////

/**
 * Abstract base class for sink formatters
 * @ingroup LoggingModule
 */
class LogFormatterBase
{
public:

	/// Destructor
	virtual ~LogFormatterBase() {}

	/**
	 * Creates a formatted string out of a log entry.
	 * @param record The log entry.
	 * @return The formatted string
	 */
	virtual std::string format(const LogRecord& record) = 0;
};

/// A shared pointer for a log formatter
typedef boost::shared_ptr<LogFormatterBase> LogFormatterPtr;

//////////////////////////////////////////////////////////////////////////////

/**
 * Abstract base class for log filters.
 * Sinks can be configured to log only messages that match the filter.
 * e.g. Logging to cout will only made if severity level is CRITICAL
 * but logging to file will be made always.
 *
 * @ingroup LoggingModule
 */
class LogFilterBase
{
public:

	/// Destructor
	virtual ~LogFilterBase() {}

	/**
	 * Filters a log entry.
	 * @param record The log entry.
	 * @return true if passed, false if filtered out
	 */
	virtual bool filter(const LogRecord& record) = 0;
};

/// A shared pointer for a log filter
typedef boost::shared_ptr<LogFilterBase> LogFilterPtr;

//////////////////////////////////////////////////////////////////////////////

/**
 * Abstract base class for all log sinks.
 *
 * @ingroup LoggingModule
 */
class MIRA_BASE_EXPORT LogSink
{
public:

	/// Destructor
	virtual ~LogSink() {}

	/**
	 * Is called by LogCore singleton whenever a new log record is written
	 * @param record The log entry.
	 */
	void aboutToConsume(const LogRecord& record);

	/**
	 * Overwrite this in derived class. Is called
	 * for each log entry that passes the filter (if any)
	 * @param record The log entry.
	 */
	virtual void consume(const LogRecord& record) = 0;

	/**
	 * Resets the log filter. No filter will be used to filter
	 * log entries.
	 */
	void resetFilter()
	{
		mFilter.reset();
	}

	/**
	 * Set the filter. The filter must be copy constructible
	 * since we are creating a copy and inherited from LogFilterBase.
	 * @param filter The filter object
	 */
	template <typename T>
	LogFilterPtr setFilter(const T& filter)
	{
		mFilter.reset(new T(filter));
		return mFilter;
	}

	/**
	 * Resets the formatter. No formatter will be used to format
	 * log entries.
	 */
	void resetFormatter()
	{
		mFormatter.reset();
	}

	/**
	 * Set the formatter. The formatter must be copy constructible
	 * since we are creating a copy and inherited from LogFormatterBase.
	 * @param formatter The formatter object
	 */
	template <typename T>
	LogFormatterPtr setFormatter(const T& formatter)
	{
		mFormatter.reset(new T(formatter));
		return mFormatter;
	}

protected:
	LogFormatterPtr mFormatter;
	LogFilterPtr mFilter;
};

/// A shared pointer for a log sink
typedef boost::shared_ptr<LogSink> LogSinkPtr;

//////////////////////////////////////////////////////////////////////////////

/**
 * Single instance of the core logging class.
 * This class takes log records and distributes them between all registered sinks.
 *
 * @ingroup LoggingModule
 */
class MIRA_BASE_EXPORT LogCore : public LazySingleton<LogCore>
{
public:

	/// Signature of a callback function for log severity level changes
	typedef boost::function<void(SeverityLevel)> SeverityLevelChangedCallback;

	/**
	 * The Constructor.
	 */
	LogCore();

	/**
	 * Parses the log record and extracts namespace, class name and function.
	 * @param record The record that will be parsed
	 * @return information about the function, class and namespace
	 */
	LogRecordFunctionInfo parseFunction(const LogRecord& record);

	/**
	 * Sets the application wide severity level.
	 * @param level The new level.
	 */
	void setSeverityLevel(SeverityLevel level)
	{
		boost::mutex::scoped_lock lock(mCallbackMutex);
		mSeverityLevel = level;
		foreach(auto i, mSeverityChangedCallbacks)
			i.second(level);
	}

	/**
	 * Get the application wide severity level.
	 * @return The level.
	 */
	SeverityLevel getSeverityLevel() const
	{
		return mSeverityLevel;
	}

	/**
	 * Register a callback function. The callback is called after the
	 * log severity level is changed by a call to \ref setSeverityLevel.
	 * @return Id number that can be used to unregister the callback
	 */
	uint32 registerCallback(SeverityLevelChangedCallback callback)
	{
		boost::mutex::scoped_lock lock(mCallbackMutex);
		mSeverityChangedCallbacks[mCallbackCounter] = callback;
		return mCallbackCounter++;
	}

	/**
	 * Unregister a callback function.
	 */
	void unregisterCallback(uint32 id)
	{
		boost::mutex::scoped_lock lock(mCallbackMutex);
		mSeverityChangedCallbacks.erase(id);
	}

	/**
	 * Get the up-time of the core.
	 * @return The up-time.
	 */
	Duration getUptime() const
	{
		return Time::now()-mStartTime;
	}

	/**
	 * Register a new sink. The sinks consume method is called upon a new log entry.
	 * Note: The sink must be copy constructible and inherited from LogSink.
	 * @param sink The sink to add.
	 * @return A pointer to the registered sink.
	 */
	template <typename T>
	LogSinkPtr registerSink(const T& sink)
	{
		boost::mutex::scoped_lock lock(mSinkMutex);
		LogSinkPtr ptr(new T(sink));
		mSinks.push_back(ptr);
		return ptr;
	}

	/**
	 * Unregisters a sink.
	 * @param sink The sink that will be unregistered.
	 */
	void unregisterSink(LogSinkPtr sink)
	{
		boost::mutex::scoped_lock lock(mSinkMutex);
		auto i = mSinks.begin();
		for (; i!= mSinks.end(); ++i)
			if ((*i) == sink)
				break;
		if (i != mSinks.end())
			mSinks.erase(i);
	}

	/**
	 * Writes a log record and distribute it between all the registered sinks.
	 * Never call this method directly. Use the macros MIRA_LOG, MIRA_LOG_ATTR
	 * or MIRA_LOG_EXCEPTION instead.
	 * @param record The log record.
	 */
	void log(LogRecord& record)
	{
		boost::mutex::scoped_lock lock(mSinkMutex);
		record.uptime = getUptime();
		for(size_t i=0; i<mSinks.size(); ++i)
			mSinks[i]->aboutToConsume(record);
	}

private:
	boost::mutex mSinkMutex;
	boost::mutex mCallbackMutex;
	Time mStartTime;
	SeverityLevel mSeverityLevel;
	std::vector<LogSinkPtr> mSinks;
	uint32 mCallbackCounter;
	std::map<uint32, SeverityLevelChangedCallback> mSeverityChangedCallbacks;
};

//////////////////////////////////////////////////////////////////////////////

/**
 * Macro for easier access to the logging core instance
 *
 * @ingroup LoggingModule
 */
#define MIRA_LOGGER (mira::LogCore::instance())

/**
 * Helper class that is created to make one logging process atomic and thread safe.
 * The stream is used to collect log data and the logger destructor will notify the core
 * the core about the new log record.
 * Never use this class directly. Use the macros MIRA_LOG and MIRA_LOG_ATTR instead.
 *
 * @ingroup LoggingModule
 */
class Logger : boost::noncopyable
{
public:
	/**
	 * The constructor taking the details of a log entry.
	 */
	Logger(SeverityLevel level,
	       const Time& time,
	       const std::string& file, int line,
	       const std::string& function, ThreadID threadID)
	{
		mRecord.level = level;
		mRecord.time = time;
		mRecord.file = file;
		mRecord.line = line;
		mRecord.function = function;
		mRecord.threadID = threadID;
	}

	/// Destructor
	~Logger()
	{
		mStream.flush();
		mRecord.message = mStream.str();
		MIRA_LOGGER.log(mRecord);
	}
	/**
	 * Get the underlying stream.
	 * @return Stream
	 */
	std::ostringstream& stream()
	{
		return mStream;
	}

protected:

	LogRecord mRecord;
	std::ostringstream mStream;
};

//////////////////////////////////////////////////////////////////////////////

/**
 * Compile time defined maximum log level
 * In release build the maximum log level should be NOTICE to eliminate some
 * of the logging and gain performance.
 * Effectively, MIRA_SEVERITY_MAX_LEVEL imposes a static plateau on the
 * dynamically allowed range of logging levels: Any logging level above the
 * static plateau is simply eliminated from the code.
 *
 * @ingroup LoggingModule
 */
#ifndef MIRA_SEVERITY_MAX_LEVEL
#  ifdef NDEBUG
#    define MIRA_SEVERITY_MAX_LEVEL mira::NOTICE
#  else
#    define MIRA_SEVERITY_MAX_LEVEL mira::TRACE
#  endif
#endif

/**
 * @ingroup LoggingModule
 * Use this macro to log data.
 * e.g.
 * @code
 * MIRA_LOG(DEBUG) << "Hello, " << "World";
 * @endcode
 * !!!ATTENTION!!!
 * This code combines two tests.
 * If you pass a compile time constant to MIRA_LOG,
 * the first test is against two constants
 * and any optimizer will pick that up statically and
 * discard the dead branch entirely from generated code.
 * The second test examines the runtime logging level.
 * If the specified level is above the runtime logging level
 * the branch will never be called.
 * e.g.
 * Assuming MIRA_SEVERITY_MAX_LEVEL is ERROR:
 * @code
 * MIRA_LOG(DEBUG) << callAnImportantFunctionThatNeedsToBeCalledUnderAllCircumstances(42);
 * @endcode
 * will be expanded by the preprocessor to
 * @code
 * if (DEBUG > ERROR);
 * else if ( DEBUG > LogCore::instance()->severityLevel());
 * 		else Logger(...).stream() << callAnImportantFunctionThatNeedsToBeCalledUnderAllCircumstances(42);
 * @endcode
 * So the so important function will never be called
 * when the runtime logging level is below the specified.
 * 
 * If you combine a MIRA_LOG with an if case it is strongly recommended to add {} 
 * around the MIRA_LOG statement:
 * \code
 * if (myCondition)
 * {
 * 		MIRA_LOG(DEBUG) << "My condition is true";
 * } 
 * \endcode
 * Otherwise you will get compiler warnings about an ambiguous else.
 * !!!ATTENTION!!!
 */
#define MIRA_LOG(level)                                                       \
if (level > MIRA_SEVERITY_MAX_LEVEL) {}                                       \
else if (level > MIRA_LOGGER.getSeverityLevel()) {}                           \
	else mira::Logger(level, mira::Time::now(), __FILE__, __LINE__, MIRA_FUNCTION, mira::getCurrentThreadID()).stream()

/**
 * @ingroup LoggingModule
 * Use this macro to log data.
 * In contrast to MIRA_LOG() the log messages will be passed to the log sinks
 * independent from the current severity level and build target.
 * One can make use of LogCustomizableFilter to modify the severity levels for
 * different log sinks.
 */
#define MIRA_LOG_ALWAYS(level)                                                \
	mira::Logger(level, mira::Time::now(), __FILE__, __LINE__, MIRA_FUNCTION, mira::getCurrentThreadID()).stream()

/**
 * @ingroup LoggingModule
 * Use this macro to log data when you have collected the log data yourself.
 */
#define MIRA_LOG_ATTR(level, time, file, line, function, threadID)            \
if (level > MIRA_SEVERITY_MAX_LEVEL) {}                                       \
else if (level > MIRA_LOGGER.getSeverityLevel()) {}                           \
	else mira::Logger(level, time, file, line, function, threadID).stream()

#define MIRA_TRACE MIRA_LOG(mira::TRACE) << "in : " << __FILE__  << "(" << __LINE__ << ") in " << MIRA_FUNCTION << " "

//////////////////////////////////////////////////////////////////////////////

}

#endif
