/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file main.C
 *
 * @author Erik Einhorn
 * @date   2011/09/16
 */

#include <QTranslator>
#include <QLocale>
#include <QLibraryInfo>
#include <QApplication>
#include <QMessageBox>

#include <Wizard.h>

#include <boost/program_options.hpp>

#include <error/Logging.h>
#include <error/LogConsoleSink.h>

#include <iostream>

using namespace mira;
using namespace std;

///////////////////////////////////////////////////////////////////////////////

class Application: public QApplication
{
public:
	Application(int& argc, char** argv): QApplication(argc,argv) {}

	/// Overwrites QApplication::notify in order to catch exceptions.
	bool notify(QObject *rec, QEvent *ev)
	{
		try {
			return QApplication::notify(rec, ev);
		}
		catch (std::exception& e) {
			QMessageBox::critical(NULL, "Exception", e.what(), QMessageBox::Ok);
		}
		catch (...) {
			QMessageBox::critical(NULL, "Exception", "Unknown Exception", QMessageBox::Ok);
		}
		this->exit(-1);
		return false;
	}
};

///////////////////////////////////////////////////////////////////////////////

int main(int argc, char *argv[])
{
	// fire up the logging mechanism
	LogConsoleSink sink;
	MIRA_LOGGER.registerSink(sink);

	/////////////////////////////////
	/// evaluate command line options

	namespace po = boost::program_options;
	po::options_description desc("Options");
	desc.add_options()
	      ("help,h", "Print help messages")
	      ("external,e", po::value<std::vector<std::string> >()->multitoken(),
	    		  "Creates a new external project. Please provide PROJECTNAME PROJECTPATH.");

	po::variables_map vm;
	try
	{
		po::store(po::parse_command_line(argc, argv, desc), vm);

		if ( vm.count("help")  )
		{
			std::cout << desc << std::endl;
			return 0;
		}
		else if ( vm.count("external") ) {
			std::vector<std::string> arguments = vm["external"].as<std::vector<std::string> >();
			std::cout << "arguments: " << arguments.size() << std::endl;
			if ( arguments.size() != 2 ) {
				cout<<"ERROR: Option -e needs specification of project name and path."<<endl;
				std::cout << desc << std::endl;
				return -1;
			}
			//we shall create an external project
			cout << "Creating external project with name :" << arguments[0]
				 << " in path " << arguments[1] << endl;

			Backend be;
			be.createExternalProject(arguments[0], arguments[1]);

			return 0;
		}
	}
	catch(po::error& e)
	{
	  std::cerr << "ERROR: " << e.what() << std::endl << std::endl;
	  std::cerr << desc << std::endl;
	  return -1;
	}

	/////////////////////////////////
	/// execute gui

	Application app(argc, argv);

	QString translatorFileName = QLatin1String("qt_");
	translatorFileName += QLocale::system().name();
	QTranslator *translator = new QTranslator(&app);
	if (translator->load(translatorFileName,
	                     QLibraryInfo::location(QLibraryInfo::TranslationsPath)))
		app.installTranslator(translator);

	Wizard wizard;
	wizard.resize(550,700);
	wizard.show();
	return app.exec();
}
