/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeEditor.C
 *    Implementation of TaeEditor.h.
 *
 * @author Tim Langner
 * @date   2011/12/25
 */

#include <TapeEditor.h>

#include <QGraphicsScene>
#include <QMouseEvent>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QAction>
#include <QProgressDialog>
#include <QMessageBox>
#include <QFormLayout>
#include <QLabel>
#include <QDoubleSpinBox>

#include <math/Saturate.h>
#include <math/Random.h>
#include <serialization/Serialization.h>
#include <fw/TapeFileDialog.h>

#include <TapeDataRenderer.h>
#include <TapeCommand.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

const int TapeEditor::sResolution[] = {1, 10, 25, 50, 75, 100,
                                       250, 500, 750, 1000,
                                       2500, 5000, 7500, 10000,
                                       25000, 50000, 75000, 100000,
                                       125000, 250000, 500000, 750000, 1000000};

const int TapeEditor::sTimeStep[] = {100, 1000, 2000, 5000, 10000, 10000,
                                     20000, 50000, 100000, 100000,
                                     200000, 500000, 1000000, 1000000,
                                     2000000, 5000000, 10000000, 10000000,
                                     20000000, 20000000, 60000000, 60000000, 100000000};

const int TapeEditor::sSubTimeSteps[] = {10, 10, 4, 5, 10, 10,
                                         4, 5, 10, 10,
                                         4, 5, 10, 10,
                                         4, 5, 10, 10,
                                         4, 4, 10, 10, 10};

TapeEditor::TapeEditor(QWidget* parent) :
	QWidget(parent)
{
	mResolution = 13;
	QVBoxLayout* layout = new QVBoxLayout(this);
	layout->setMargin(0);
	layout->setSpacing(0);

	mOpenAct = new QAction("Open",this);
	mOpenAct->setIcon(QIcon(":/icons/FileOpen.png"));
	mOpenAct->setToolTip("Opens one or multiple tapes");
	connect(mOpenAct, SIGNAL(triggered()), this, SLOT(open()) );

	mSaveAct = new QAction("Save",this);
	mSaveAct->setIcon(QIcon(":/icons/FileSave.png"));
	mSaveAct->setEnabled(false);
	mSaveAct->setToolTip("Saves the current content to a tape");
	connect(mSaveAct, SIGNAL(triggered()), this, SLOT(save()) );

	mEraseMessagesAct = new QAction("Erase", this);
	mEraseMessagesAct->setIcon(QIcon(":/icons/Erase.png"));
	mEraseMessagesAct->setEnabled(false);
	mEraseMessagesAct->setToolTip("Erase messages in selected channels in the current interval");
	connect(mEraseMessagesAct, SIGNAL(triggered()), this, SLOT(eraseSelectedMessages()));

	mEraseTimeESAct = new QAction("Erase and move remaining towards start", this);
	mEraseTimeESAct->setIcon(QIcon(":/icons/EraseTimeES.ico"));
	mEraseTimeESAct->setEnabled(false);
	mEraseTimeESAct->setToolTip("Erase the current selected time interval (erasing from end towards start, moving remaining messages towards start)");
	connect(mEraseTimeESAct, SIGNAL(triggered()), this, SLOT(eraseSelectedTimeEndToStart()));

	mEraseTimeSEAct = new QAction("Erase and move remaining towards end",this);
	mEraseTimeSEAct->setIcon(QIcon(":/icons/EraseTimeSE.ico"));
	mEraseTimeSEAct->setEnabled(false);
	mEraseTimeSEAct->setToolTip("Erase the current selected time interval (erasing from start towards end, moving remaining messages towards end)");
	connect(mEraseTimeSEAct, SIGNAL(triggered()), this, SLOT(eraseSelectedTimeStartToEnd()));

	mTrimGapsAct = new QAction("Trim gaps in the tape",this);
	//mTrimAct->setIcon(QIcon(":/icons/EraseTimeSE.ico"));
	mTrimGapsAct->setEnabled(false);
	mTrimGapsAct->setToolTip("Removes gaps in the tape to create continous data");
	connect(mTrimGapsAct, SIGNAL(triggered()), this, SLOT(trimGaps()));


	mUndoAct = new QAction("Undo", this);
	mUndoAct->setIcon(QIcon(":/icons/Undo.png"));
	mUndoAct->setEnabled(false);
	mUndoAct->setToolTip("Undo the last action");
	connect(mUndoAct, SIGNAL(triggered()), this, SLOT(undo()));

	mRedoAct = new QAction("Redo", this);
	mRedoAct->setIcon(QIcon(":/icons/Redo.png"));
	mRedoAct->setEnabled(false);
	mRedoAct->setToolTip("Redo the last action");
	connect(mRedoAct, SIGNAL(triggered()), this, SLOT(redo()));

	QSplitter* splitter = new QSplitter(Qt::Vertical, this);
	mThumbView = new TapeThumbView(this);
	splitter->addWidget(mThumbView);
	mTapeView = new TapeView(this);
	splitter->addWidget(mTapeView);
	layout->addWidget(splitter);
	connect(getDataView(), SIGNAL(contentsMoving(int, int)), mThumbView, SLOT(updateThumb()));
	clear();
}

TapeEditor::~TapeEditor()
{
	clear();
}

void TapeEditor::clear()
{
	mVisitor = TapeVisitor();
	foreach(Tape* t, mTapes)
		delete t;
	mTapes.clear();
	mTapeFiles.clear();
	mChannels.clear();
	foreach(TapeCommand* c, mUndoList)
		delete c;
	mUndoList.clear();
	foreach(TapeCommand* c, mRedoList)
		delete c;
	mRedoList.clear();

	mFirstMessage = 0;
	mLastMessage = 0;
	mRecordingTime = Time::now();
}

void TapeEditor::updateVisitor()
{
	mFirstMessage = std::numeric_limits<int64>::max();
	mLastMessage = std::numeric_limits<int64>::min();

	const TapeVisitor::ChannelMap& channels = mVisitor.getChannels();
	typedef std::pair<std::string, const Tape::ChannelInfo*> ChannelPair;
	foreach(const ChannelPair& channel, channels)
	{
		addChannel(channel.first, channel.second->type,
		           channel.second->meta,
		           channel.second->metaDB );
		mMetaDB.merge(channel.second->metaDB);
	}
	TapeVisitor::iterator it = mVisitor.begin();
	for(; it != mVisitor.end(); ++it)
	{
		TapeChannelInfo& info = mChannels[it.getChannelInfo()->name];
		int64 t = it.getTimeOffset().totalMicroseconds();
		info.data[t] = TapeChannelMessage(it);
		if (t < mFirstMessage)
			mFirstMessage = t;
		if (t > mLastMessage)
			mLastMessage = t;
	}
	Time recStart = mVisitor.getStartTime();
	if (recStart.isValid())
		mRecordingTime = recStart;
	if (mFirstMessage > 0)
		mFirstMessage = 0;
	setResolution(mResolution);
	mTapeView->updateContents();
	mThumbView->updateContents();
	mThumbView->updateThumb();
}

void TapeEditor::setResolution(int res)
{
	mResolution = saturate(res, 0, (int)(sizeof(sResolution)/sizeof(int))-1);
	mTapeView->centerContents();
	//mTapeView->updateContents();
	mThumbView->updateThumb();
}

void TapeEditor::updateContents(bool refreshAll)
{
	mFirstMessage = std::numeric_limits<int64>::max();
	mLastMessage = std::numeric_limits<int64>::min();
	uint64 nrMessages = 0;
	foreach(TapeChannelInfoMap::value_type& channel, mChannels)
	{
		if (channel.second.data.size() == 0)
			continue;
		nrMessages += channel.second.data.size();
		if (channel.second.data.begin()->first < mFirstMessage)
			mFirstMessage = channel.second.data.begin()->first;
		if (channel.second.data.rbegin()->first > mLastMessage)
			mLastMessage = channel.second.data.rbegin()->first;
	}
	// we have no messages in our channels at all so use 0 as start and end time
	if (nrMessages == 0)
	{
		mFirstMessage = 0;
		mLastMessage = 0;
	}
	if (mFirstMessage > 0)
		mFirstMessage = 0;
	mTapeView->updateContents();
	if (refreshAll)
		mThumbView->updateContents();
}

int64 TapeEditor::getFirstMessageTime() const
{
	return mFirstMessage;
}

int64 TapeEditor::getLastMessageTime() const
{
	return mLastMessage;
}

int64 TapeEditor::getTotalTime() const
{
	return mLastMessage - mFirstMessage;
}

QString TapeEditor::getStringFromTime(int64 t, bool resolutionDependent)
{
	QString txt;
	int64 at = std::abs(t);
	int s = at / 1000000;
	at = at % 1000000;
	int m = s / 60;
	s = s % 60;

	txt = (t<0 ? QString("-") : QString("")) +
	      QString("%1:%2.%3").arg(m, 2, 10, QChar('0')).arg(s, 2, 10, QChar('0'));
	if (!resolutionDependent || (getResolution() == 1))
		txt = txt.arg(at, 6, 10, QChar('0'));
	else
		txt = txt.arg(at/1000, 3, 10, QChar('0'));

	return txt;
}

void TapeEditor::setSelectionStart(int64 start)
{
	start = saturate(start, mFirstMessage, mLastMessage);
	if (start > mSelectionEnd)
		setSelectionEnd(start);
	mSelectionStart = start;
	updateActionStates();
}

void TapeEditor::setSelectionEnd(int64 end)
{
	end = saturate(end, mFirstMessage, mLastMessage);
	if (end < mSelectionStart)
		setSelectionStart(end);
	mSelectionEnd = end;
	updateActionStates();
}

void TapeEditor::clearSelectedChannels()
{
	foreach(TapeChannelInfoMap::value_type& channel, mChannels)
		channel.second.selected = false;
}

QColor generateColor()
{
	static const QColor mix(127, 127, 127);
	int red = MIRA_RANDOM.uniform(0, 255);
	int green = MIRA_RANDOM.uniform(0, 255);
	int blue = MIRA_RANDOM.uniform(0, 255);

	// mix the color
	red = (red + mix.red()) / 2;
	green = (green + mix.green()) / 2;
	blue = (blue + mix.blue()) / 2;

	return QColor(red, green, blue);
}


void TapeEditor::addChannel(const std::string& name,
                            const std::string& type,
                            TypeMetaPtr meta,
                            const MetaTypeDatabase& db)
{
	if (mChannels.count(name) > 0)
		MIRA_THROW(XInvalidConfig, "Channel with name '" << name <<
		          "' already exists");
	if (name.empty())
		MIRA_THROW(XInvalidConfig, "Channel name is empty");
	if (type.empty())
		MIRA_THROW(XInvalidConfig, "Type is empty");

	TapeChannelInfo info;
	info.name = name;
	info.type = type;
	info.meta = meta;
	info.metadb = db;
	info.color = generateColor();
	info.renderer.reset(new TapeDataRenderer(this));
	mChannels[name] = info;
}

void TapeEditor::executeCommand(TapeCommand* command)
{
	foreach(TapeCommand* c, mRedoList)
		delete c;
	mRedoList.clear();
	mUndoList.push_back(command);
	command->exec();
	if (mUndoList.size() > 5)
	{
		delete mUndoList.front();
		mUndoList.pop_front();
	}
	mRedoAct->setEnabled(false);
	mUndoAct->setEnabled(true);
}

void TapeEditor::openFiles(const QStringList& files)
{
	clear();

	if(files.empty())
		return;

	QProgressDialog progress("Loading tapes ...", QString(), 0, files.size(), this);
	progress.setWindowModality(Qt::WindowModal);

	for(int32 i=0; i<files.size(); ++i)
	{
		progress.setValue(i);
		Tape* t = new Tape();
		t->open(files[i].toStdString(), Tape::READ);
		mTapes.push_back(t);
		mVisitor.visit(t);
	}
	mTapeFiles = files;
	progress.setValue(files.size());

	updateVisitor();
	mSaveAct->setEnabled(true);
	updateActionStates();

	// use original start time by default
	mTapeStartTime = mVisitor.getStartTime();
}

void TapeEditor::open()
{
	TapeFileDialog dialog(this);
	dialog.setNameFilter("Tape files (*.tape)");
	dialog.setFileMode(QFileDialog::ExistingFiles);
	dialog.setAcceptMode(QFileDialog::AcceptOpen);
	dialog.setUseOriginalTimestamp(true);
	if (!dialog.exec())
		return;
	openFiles(dialog.selectedFiles());

	if (!dialog.useOriginalTimestamp())
		mTapeStartTime = Time::now();
}

void TapeEditor::save()
{
	QString fileName = QFileDialog::getSaveFileName(this, tr("Filename"), "", "tapes (*.tape)");
	if ( fileName.isEmpty() )
		return;

	foreach(auto const & f, mTapeFiles)
	{
		if (QFileInfo(f).canonicalFilePath() == QFileInfo(fileName).canonicalFilePath())
		{
			QMessageBox::warning(this, "Tape Editor", "Cannot save to a tape file that is currently opened in Tape Editor. Saving was Aborted.");
			return;
		}
	}

	Tape tape;
	tape.open(fileName.toLocal8Bit().data(), Tape::WRITE);
	tape.alterStartTime(mTapeStartTime);

	typedef std::multimap<int64, std::pair<TapeChannelInfo*, TapeChannelMessage>> DataMap;
	DataMap allData;
	foreach(TapeChannelInfoMap::value_type& channel, mChannels)
		foreach(TapeChannelInfo::DataMap::value_type& item, channel.second.data)
			allData.insert(std::make_pair(item.first, std::make_pair(&channel.second, item.second)));

	QProgressDialog progress("Saving tape ...", "Abort", 0, allData.size(), this);
	progress.setWindowModality(Qt::WindowModal);

	std::size_t i=0;
	while(!allData.empty())
	{
		auto it = allData.begin();
		DataMap::value_type& item = *it;

		if(i%10 == 0) {
			progress.setValue(i);
			if (progress.wasCanceled()) {
				tape.close();
				QMessageBox::warning(this, "Tape Editor", "Saving was Aborted. The written tape will be truncated.");
				return;
			}
		}
		++i;

		Time t = tape.getStartTime() + Duration::microseconds(item.first);
		tape.write(item.second.first->name, item.second.first->type, t,
		           item.second.second.getFrameID(), item.second.second.getSequenceID(),
		           item.second.second.getData(), (item.second.second.getCompressed() ? -1 : 0),
		           item.second.first->meta,
		           item.second.first->metadb);

		// erase iterator. This will free the memory that is used by the loaded data
		allData.erase(it);
	}
	tape.close();
}

void TapeEditor::zoomIn(int numSteps)
{
	if (numSteps > 0)
		setResolution(mResolution - numSteps);
}

void TapeEditor::zoomOut(int numSteps)
{
	if (numSteps > 0)
		setResolution(mResolution + numSteps);
}

void TapeEditor::zoomReset()
{
	setResolution(13);
}

void TapeEditor::eraseSelectedMessages()
{
	if (mSelectionEnd == mSelectionStart)
		return;
	executeCommand(new EraseMessagesCommand(mSelectionStart, mSelectionEnd, this));
}

void TapeEditor::eraseSelectedTimeEndToStart()
{
	int64 erasedDuration = mSelectionEnd - mSelectionStart;
	if (erasedDuration == 0)
		return;
	executeCommand(new EraseTimeEndToStartCommand(mSelectionStart, mSelectionEnd, this));
}

void TapeEditor::eraseSelectedTimeStartToEnd()
{
	int64 erasedDuration = mSelectionEnd - mSelectionStart;
	if (erasedDuration == 0)
		return;
	executeCommand(new EraseTimeStartToEndCommand(mSelectionStart, mSelectionEnd, this));
}

void TapeEditor::trimGaps()
{
	std::string firstSelected;
	foreach(TapeChannelInfoMap::value_type& channel, mChannels)
		if(channel.second.selected) {
			firstSelected = channel.first;
			break;
		}
	if(firstSelected.empty()) {
		QMessageBox::information(this, "Cannot proceed", "You need to select a master channel, that is used for detecting the gaps");
		return;
	}

	QDialog d(this);
	QFormLayout *layout = new QFormLayout;
	layout->addRow("Master channel:", new QLabel(QString::fromStdString(firstSelected), &d));

	QDoubleSpinBox* maxGap = new QDoubleSpinBox(&d);
	maxGap->setMinimum(0.0);
	maxGap->setMaximum(1.0e100);
	maxGap->setValue(1.0);
	layout->addRow("Max. gap (in s)", maxGap);

	QDialogButtonBox* buttonBox = new QDialogButtonBox(QDialogButtonBox::Ok
	                                      | QDialogButtonBox::Cancel);

	connect(buttonBox, SIGNAL(accepted()), &d, SLOT(accept()));
	connect(buttonBox, SIGNAL(rejected()), &d, SLOT(reject()));
	layout->addRow(buttonBox);

	d.setLayout(layout);
	if(d.exec() == QDialog::Accepted)
		executeCommand(new TrimGapsCommand(firstSelected, maxGap->value()*1000000, this));
}

void TapeEditor::undo()
{
	if (mUndoList.size() == 0)
		return;
	TapeCommand* command = mUndoList.back();
	mUndoList.pop_back();
	command->undo();
	mRedoList.push_back(command);
	mRedoAct->setEnabled(true);
	mUndoAct->setEnabled(mUndoList.size() > 0);
}

void TapeEditor::redo()
{
	if (mRedoList.size() == 0)
		return;
	TapeCommand* command = mRedoList.back();
	mRedoList.pop_back();
	command->exec();
	mUndoList.push_back(command);
	mUndoAct->setEnabled(true);
	mRedoAct->setEnabled(mRedoList.size() > 0);
}

void TapeEditor::wheelEvent(QWheelEvent* event)
{
	int numDegrees = event->delta() / 8;
	int numSteps = numDegrees / 15;  // see QWheelEvent documentation
	if (numSteps > 0)
		zoomIn(numSteps);
	else
		zoomOut(-numSteps);
}

void TapeEditor::updateActionStates()
{
	mTrimGapsAct->setEnabled(true);

	bool validSelection = (mSelectionEnd > mSelectionStart);
	mEraseMessagesAct->setEnabled(validSelection);
	mEraseTimeESAct->setEnabled(validSelection);
	mEraseTimeSEAct->setEnabled(validSelection);

	mUndoAct->setEnabled(mUndoList.size() > 0);
	mRedoAct->setEnabled(mRedoList.size() > 0);
}

///////////////////////////////////////////////////////////////////////////////

} // namespace
