/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PackageTreeModel.C
 *    $Implementation of PackageTreeModel.h$.
 *
 * @author Ronny Stricker
 * @date   2011/07/26
 */

#include <QBrush>
#include <QFont>
#include <QPainter>
#include <QBitmap>

#include "core/Package.h"
#include "gui/IconProvider.h"
#include "gui/PackageTreeModel.h"

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

PackageTreeModel::PackageTreeModel( PackageGroup const* rootPackage, Database* database,
		Flags flags, QObject *parent) : mRoot( rootPackage ), mDatabase( database ), mFlags( flags )
{
	assert( mDatabase );
}

PackageTreeModel::~PackageTreeModel()
{
}

int PackageTreeModel::columnCount(const QModelIndex &parent) const
{
	return 7;
}

QVariant PackageTreeModel::data(const QModelIndex &index, int role) const
{
	if (!index.isValid())
		return QVariant();

	// every Package is checkable, however the package groups are not.
	// choose appropriate item set
	PackageGroup* group = static_cast<PackageGroup*>(index.internalPointer());
	Package* package = dynamic_cast<Package*>(group);
	Dependency* dep = dynamic_cast<Dependency*>(package);
	bool checkable = package ? true : false;

	// store persistent model index
	mTrackedIndexes[group][index.column()] = index;

	if (role == Qt::DecorationRole) {
		switch ( index.column() ) {
			case 0: return IconProvider::instance().getIcon(mDatabase->isInstalled(group),
						group->mType, dep ? dep->mDepFlag : Dependency::NONE );
					break;
			case 1: return QVariant(); break;
			case 2: return QVariant(); break;
			case 3: return QVariant(); break;
		}
	} else
	if (role == Qt::EditRole) {
		if ( index.column() == 2 && package) {
			QStringList tReturn;
			vector<string> tRepos = package->sortedRepos( mDatabase );
			foreach( std::string repo, tRepos ) {
				tReturn << QString::fromStdString( mDatabase->getRepoNameFromUrl( repo ) );
			}
			return tReturn;
		}
		return QVariant();
	} else
	if (role == Qt::DisplayRole) {
		bool optional = dep && (dep->mDepFlag & Dependency::FACULTATIVE);
		bool runtime = dep && (dep->mDepFlag & Dependency::RUNTIME);
		switch ( index.column() ) {
			case 0: return QVariant(); break;
			case 1:	return (group->mName
					+ (optional ? " [OPT]" : "" ) + (runtime ? " [RUN]" : "" )).c_str(); break;
			case 2: if ( package ) {
						if ( !package->mCurrentRepo.empty() )
							return mDatabase->getRepoNameFromUrl( package->mCurrentRepo ).c_str();

						if ( package->mRepos.size() )
							return mDatabase->getRepoNameFromUrl( package->sortedRepos(mDatabase)[0] ).c_str();
						return "";
					}
					break;
				return package ? mDatabase->getRepoNameFromUrl( package->mCurrentRepo ).c_str() : ""; break;
			case 3: return package ? Package::getDevelStateString( *package ).c_str() : ""; break;
			case 4: return package ? package->mVersion.str().c_str() : ""; break;
			case 5: return package ? package->mRepoSubPath.c_str() : ""; break;
			case 6: return group->getShortDescription().c_str(); break;
			default: return ""; break;
		}
	} else if ( role == Qt::CheckStateRole ) {
		if ( index.column() == 0 && checkable )
			return ( ( mDatabase->isInstalled( group ) && !(mDatabase->getFlag( group ) & Database::UNINSTALL ) ) ||
				(mDatabase->getFlag( group ) & Database::INSTALL ) ) ? Qt::Checked : Qt::Unchecked;
	} else if ( role == Qt::BackgroundRole ) {
		if ( mDatabase->willChange( group ) )
			return QBrush( QColor( 255, 248, 235 ) );
		else if ( mDatabase->isInstalled( group ) )
			return QBrush( QColor( 235, 245, 255 ) );
		else
			return QBrush( QColor( 255, 255, 255 ) );
	} else if ( role == Qt::FontRole ) {
		if ( index.column() == 1 )
			return QFont( "Arial", 10, 75 );
		else
			return QFont( "Arial", 10, 50 );
	} else if ( role == Qt::TextColorRole ) {
		if ( dep ) {
			if ( dep->mDepFlag & Dependency::VERSION_INCOMP )
				return QColor(Qt::red);
			else
				return QColor(Qt::darkGreen);
		}
		else
			return QColor(Qt::black);
	}

	return QVariant();
}

bool PackageTreeModel::setData(QModelIndex const& index, const QVariant &value, int role)
{
	if ( index.isValid() ) {
		PackageGroup* item = static_cast<PackageGroup*>(index.internalPointer());
		Package* package = dynamic_cast<Package*>( item );
		if (index.column() == 0 && role == Qt::CheckStateRole ) {
			if ( mFlags & EDITABLE ) {
				if ( package ) {
					if ( value.toBool() == true) {
						if ( mDatabase->isInstalled( package ) )
							mDatabase->setFlag( package, Database::NONE );
						else
							mDatabase->setFlag( package, Database::INSTALL );
					}
					else {
						if ( mDatabase->isInstalled( package ) )
							mDatabase->setFlag( package, Database::UNINSTALL );
						else
							mDatabase->setFlag( package, Database::DONOTINSTALL );
					}
					emit dataChanged(index, index);
				}
			}
			else {
				emit installChangeRequested( index );
			}
		}
		else if ( (mFlags & EDITABLE) && role == Qt::EditRole
				&& index.column() == 2 ) {
			package->mCurrentRepo = mDatabase->getRepoUrlFromName(
					value.toString().toStdString());
			emit dataChanged(index, index);
		}
	}
	return false;
}

Qt::ItemFlags PackageTreeModel::flags(const QModelIndex &index) const
{
	if (!index.isValid())
		return Qt::ItemFlags();

	Qt::ItemFlags tReturn
		= Qt::ItemFlags(Qt::ItemIsEnabled) | Qt::ItemFlags(Qt::ItemIsSelectable);

	if ( index.column() == 0 ) {
		PackageGroup* item = static_cast<PackageGroup*>(index.internalPointer());
		if ( item->getClass().getIdentifier() == Package::CLASS().getIdentifier() ) {
			tReturn |= Qt::ItemIsUserCheckable;
//			if ( mFlags & EDITABLE ) {
//				tReturn |= Qt::ItemIsEditable;
//			}
		}
	}
	if ( (index.column() == 2) && (mFlags & EDITABLE) ) {
		tReturn |= Qt::ItemIsEditable;
	}
	return tReturn;
}

QVariant PackageTreeModel::headerData(int section, Qt::Orientation orientation,
                                      int role) const
{
	if (orientation == Qt::Horizontal && role == Qt::DisplayRole) {
		switch ( section ) {
			case 0: return QVariant("");break;
			case 1: return QVariant("Name");break;
			case 2: return QVariant("Source");break;
			case 3: return QVariant("DevelState");break;
			case 4: return QVariant("Version");break;
			case 5: return QVariant("RepoSubPath");break;
			case 6: return QVariant("Description");break;
		}
	}
	return QVariant();
}

QModelIndex PackageTreeModel::index(int row, int column, const QModelIndex &parent) const
{
	if (!hasIndex(row, column, parent))
		return QModelIndex();

	PackageGroup const* parentItem;

	if (!parent.isValid())
		parentItem = mRoot;
	else
		parentItem = static_cast<PackageGroup*>(parent.internalPointer());

	if ( int( parentItem->mSubPackages.size() ) < row )
		return QModelIndex();
	else {
		QModelIndex tIndex = createIndex(row, column, parentItem->mSubPackages[row]);
		return tIndex;
	}
}

QModelIndex PackageTreeModel::parent(const QModelIndex &index) const
{
	if (!index.isValid())
		return QModelIndex();

	PackageGroup *childItem = static_cast<PackageGroup*>(index.internalPointer());
	PackageGroup* parentItem = childItem->mParentPackage;

	if ( parentItem == mRoot )
		return QModelIndex();

	// determine row
	int row = 0;
	if ( parentItem ) {
		foreach( PackageGroup* child, parentItem->mSubPackages ) {
			if ( child == childItem )
				break;
			++row;
		}
	}

	return createIndex( row, 0, parentItem );
}

int PackageTreeModel::rowCount(const QModelIndex &parent) const
{
	PackageGroup const* parentItem;
	if (parent.column() > 0)
		return 0;

	if (!parent.isValid()) {
		parentItem = mRoot;
	}
	else {
		parentItem = static_cast<PackageGroup*>( parent.internalPointer() );
	}

	return parentItem->mSubPackages.size();
}

///////////////////////////////////////////////////////////////////////////////

}
