/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PackageListWidget.C
 *    $short description$.
 *
 * @author Ronny Stricker
 * @date   2011/07/29
 */

#include "gui/PackageListWidget.h"

#include <QWidget>
#include <QSortFilterProxyModel>
#include <QStyledItemDelegate>
#include <QMenu>
#include <QWebFrame>
#include <QFile>
#include <QTextStream>

#include <utils/PathFinder.h>

#include "core/Database.h"
#include "core/Repository.h"
#include "core/MIRAPackage.h"
#include "gui/PackageTreeModel.h"
#include "gui/TagTableModel.h"
#include "gui/PackageFilterModel.h"
#include "gui/PromptProviderGui.h"
#include "gui/dialogs/DependencyDialog.h"

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

PackageListWidget::PackageListWidget( MIRAPackage* checkout )
	: mCheckoutCore( checkout ), mProviderGui( this )
{
	Ui::PackageListWidget::setupUi(this);

	mPackageModel = new PackageTreeModel( mCheckoutCore->mDB.getRootPackage(),
			&mCheckoutCore->mDB, PackageTreeModel::NOFLAGS, this );

	proxyModel = new PackageFilterModel(packageView);
	proxyModel->setSortCaseSensitivity(Qt::CaseInsensitive);

	mTagFilterModel = new TagFilterModel(tagView);

	packageView->setIconSize( QSize(24,24) );

	mSpTags->setStretchFactor(0,1);
	mSpTags->setStretchFactor(1,3);

	mSpPackages->setStretchFactor(0,3);
	mSpPackages->setStretchFactor(1,1);

	std::vector<Path> styleFiles = findProjectFiles( "Package.xhtml", true );

	if ( styleFiles.size() == 0 )
		MIRA_THROW( XIO, "cannot find style file (Package.xhtml)!");

	QFile file( styleFiles.front().string().c_str() );
	if ( !file.open(QIODevice::ReadOnly | QIODevice::Text) ) {
		MIRA_THROW( XIO, "cannot open style file ("
				+ styleFiles.front().string() + " )!" );
	}

	QTextStream in(&file);
	while (!in.atEnd()) {
		mDescriptionMaster += in.readLine();
	}

	connect( packageView->header(), SIGNAL( sectionCountChanged( int, int ) ),
	         this, SLOT( setHeaderResizing( int,int ) ) );

	mCheckoutCore->mDB.setRepositoryPromptProvider( &mProviderGui );

	proxyModel->setSourceModel( mPackageModel );
	packageView->setModel( proxyModel );
	packageView->header()->setSectionHidden(2,true);

	mTagModel = new TagTableModel( &mCheckoutCore->mDB, this );
	mTagFilterModel->setSourceModel( mTagModel );

	mTagFilterModel->sort(0);
	tagView->setModel( mTagFilterModel );

	connect( packageView->selectionModel(),
	         SIGNAL( currentChanged( QModelIndex const&, QModelIndex const& ) ),
	         this, SLOT( updatePackageDetails( QModelIndex const& ) ) );

	connect( packageView->header(), SIGNAL(sectionClicked ( int )),
	         this, SLOT(headerClicked( int)));

	packageView->header()->setContextMenuPolicy( Qt::CustomContextMenu );
	connect( packageView->header(), SIGNAL(customContextMenuRequested ( const QPoint &) ),
	         this, SLOT(headerContextMenu(const QPoint &)));

	connect( packageView, SIGNAL(pressed( QModelIndex const& ) ), this,
	         SLOT( showMappedContextMenu( QModelIndex const& ) ) );

	connect( mPackageModel, SIGNAL(installChangeRequested( QModelIndex const& ) ), this,
	         SLOT( showContextMenu( QModelIndex const& ) ) );

	connect( tagView->selectionModel(),
	         SIGNAL( selectionChanged( QItemSelection const&, QItemSelection const& ) ),
	         this, SLOT( updateTagSelection() ) );

	connect( mSliderRare, SIGNAL( valueChanged(int) ),
	         this, SLOT( updateTagSelection() ) );

	connect( webView, SIGNAL( linkClicked( QUrl const& ) ),
	         this, SLOT( openLink( QUrl const& ) ) );

	connect( searchFilterEdit, SIGNAL( textChanged( QString const& ) ),
	         this, SLOT( updateFilterModel( QString const& ) ) );

	connect( mCxFilterNameSelect, SIGNAL( stateChanged( int ) ),
	         this, SLOT( updateFilterMode() ) );
	connect( mCxFilterDescriptionSelect, SIGNAL( stateChanged( int ) ),
	         this, SLOT( updateFilterMode() ) );
	connect( mCxFilterAuthorSelect, SIGNAL( stateChanged( int ) ),
	         this, SLOT( updateFilterMode() ) );

	connect( buttonApply, SIGNAL( clicked( ) ),
	         this, SIGNAL( applyChangesRequested() ) );

	connect( buttonUpdate, SIGNAL( clicked() ),
	         this, SIGNAL( packageUpdateRequested() ) );

	updateFilterMode();
}

///////////////////////////////////////////////////////////////////////////////

void PackageListWidget::reflect(XMLSerializer& r)
{
	QByteArray data;
	std::string str;
	bool checked;

	data = mSpTags->saveState();
	str = QString(data.toBase64()).toStdString();
	r.member("SplitterTags", str, "");

	data = mSpPackages->saveState();
	str = QString(data.toBase64()).toStdString();
	r.member("SplitterPackages", str, "");

	checked = mCxFilterNameSelect->isChecked();
	r.member("FilterName", checked, "");
	checked = mCxFilterDescriptionSelect->isChecked();
	r.member("FilterDescription", checked, "");
	checked = mCxFilterAuthorSelect->isChecked();
	r.member("FilterAuthor", checked, "");
}

void PackageListWidget::reflect(XMLDeserializer& r)
{
	std::string str;
	QByteArray data;
	bool checked;

	r.member("SplitterTags", str, "");
	data = QByteArray::fromBase64(QByteArray(str.c_str()));
	mSpTags->restoreState(data);

	r.member("SplitterPackages", str, "");
	data = QByteArray::fromBase64(QByteArray(str.c_str()));
	mSpPackages->restoreState(data);

	r.member("FilterName", checked, "");
	mCxFilterNameSelect->setChecked(checked);
	r.member("FilterDescription", checked, "");
	mCxFilterDescriptionSelect->setChecked(checked);

	checked = mCxFilterAuthorSelect->isChecked();
	r.member("FilterAuthor", checked, "");
}

///////////////////////////////////////////////////////////////////////////////

void PackageListWidget::installPackage(QVariant const& source )
{
	if (packageView->currentIndex().isValid()) {
		PackageGroup* tGroup = static_cast<PackageGroup*>(
				proxyModel->mapToSource( packageView->currentIndex() ).internalPointer() );
		Package* tPackage = dynamic_cast<Package*>( tGroup );
		assert( tPackage );
		tPackage->mCurrentRepo = mCheckoutCore->mDB.getRepoUrlFromName(
				source.toString().toStdString() );
		emit packageInstallRequested( tPackage );
	}
}

void PackageListWidget::uninstallPackage()
{
	if (packageView->currentIndex().isValid()) {
		PackageGroup* tGroup = static_cast<PackageGroup*>(
				proxyModel->mapToSource( packageView->currentIndex() ).internalPointer() );
		Package* tPackage = dynamic_cast<Package*>( tGroup );
		assert( tPackage );
		emit packageUninstallRequested( tPackage );
	}
}

void PackageListWidget::changePackageVersion(QVariant const& newVersion)
{
	if (packageView->currentIndex().isValid()) {
		PackageGroup* tGroup = static_cast<PackageGroup*>(
				proxyModel->mapToSource( packageView->currentIndex() ).internalPointer() );
		Package* tPackage = dynamic_cast<Package*>( tGroup );
		assert( tPackage );

		string tNewVersion = newVersion.toString().toStdString();
		if ( tPackage->mVersion.str() != tNewVersion ) {
			map<string,Package*> alternativeVersions = mCheckoutCore->mDB.alternativeVersions( tPackage );
			assert( alternativeVersions.find(tNewVersion) != alternativeVersions.end() );

			// Use the old MIRA_PATH for the new version.
			alternativeVersions[tNewVersion]->mLocalPath =
					mCheckoutCore->getPathForPackageUpdate(tPackage);

			emit packageUninstallRequested( tPackage );
			emit packageInstallRequested( alternativeVersions[tNewVersion] );
		}
	}
}

void PackageListWidget::update()
{
	mTagModel->beginReset();
	mTagModel->endReset();

	updatePackageDetails( packageView->selectionModel()->currentIndex() );
}

void PackageListWidget::updatePackageDetails( QModelIndex const& index )
{
	if (!index.isValid())
		return;

	PackageGroup* tPackageGroup = static_cast<PackageGroup*>(
			proxyModel->mapToSource( index ).internalPointer() );

	QString htmlText;

	if ( tPackageGroup->getClass().getIdentifier() == PackageGroup::CLASS().getIdentifier() ) {

	}
	else
	{
		htmlText = mDescriptionMaster;

		Package* tPackage = static_cast<Package*>( tPackageGroup );

		htmlText.replace( "#PACKAGE_NAME#" ,QString::fromStdString( tPackage->mName ) );

		QString repoSource;

		// disable install/uninstall Button if no associated repository could be found
		bool disableButton = false;

		bool isInstalled = mCheckoutCore->mDB.isInstalled( tPackage, true );

		if ( isInstalled  ) {
			repoSource += "<select id=\"source\" disabled=\"disabled\">";
			repoSource += QString("<option>");
			repoSource += QString::fromStdString(
					mCheckoutCore->mDB.getRepoNameFromUrl(tPackage->mCurrentRepo) )
						+ "</option>";
			repoSource += "</select>";
			if ( tPackage->mCurrentRepo.empty() )
				disableButton = true;
		}
		else {
			repoSource += "<select id=\"source\">";
			vector<string> tRepos = tPackage->sortedRepos( &mCheckoutCore->mDB );
			foreach( std::string const& tUrl, tRepos ) {
				if ( tPackage->mCurrentRepo == tUrl )
					repoSource += QString("<option selected=\"selected\">");
				else
					repoSource += QString("<option>");
				repoSource += QString::fromStdString(
						mCheckoutCore->mDB.getRepoNameFromUrl( tUrl ) )
						+ "</option>";
			}
			repoSource += "</select>";
			if ( tRepos.size() == 0 )
				disableButton = true;
		}
		htmlText.replace( "#REPOINFO#", repoSource );

		QString installForm;

		if ( isInstalled ) {
			installForm += "<button name=\"InstButton\" type=\"button\" value=\"uninstall\" ";
			installForm += disableButton ? "disabled=\"disabled\" " : " ";
			installForm += "onclick=\"MyObject.uninstallPackage()\">Uninstall</button>";
		}
		else {
			installForm += "<button name=\"InstButton\" type=\"button\" value=\"install\" ";
			installForm += disableButton ? "disabled=\"disabled\" " : " ";
			installForm += "onclick=\"MyObject.installPackage(document.getElementById('source').options["
						"document.getElementById('source').selectedIndex].text)\""
					">Install</button>";
		}
		htmlText.replace( "#INSTALLBUTTON#", installForm );

		htmlText.replace( "#AUTHOR#" ,QString::fromStdString( tPackage->mAuthor ) );

		QString versionString = QString::fromStdString( tPackage->mVersion.str() );

		if ( isInstalled ) {
			map<string,Package*> alternativePkgs = mCheckoutCore->mDB.alternativeVersions( tPackage );
			if ( alternativePkgs.size() > 0 ) {
				// create a sorted list of all versions
				list<Package::Version> alternativeVersions;
				alternativeVersions.push_back(tPackage->mVersion);
				foreach( auto const& tAltPkg, alternativePkgs ) {
					list<Package::Version>::iterator iter = alternativeVersions.begin();
					while ((iter != alternativeVersions.end()) && (*iter > tAltPkg.second->mVersion))
						++iter;
					alternativeVersions.insert(iter, tAltPkg.second->mVersion);
				}

				versionString = "<select id=\"versionSelect\">";
				foreach( auto const& v, alternativeVersions ) {
					if (v == tPackage->mVersion) {
						// current installed version
						versionString += QString("<option selected=\"selected\">");
						versionString += QString::fromStdString(v.str());
						versionString += QString("</option>");
					} else {
						// alternative version
						versionString += QString("<option>");
						versionString += QString::fromStdString(v.str());
						versionString += QString("</option>");
					}
				}
				versionString += "</select>";
				versionString += "<button name=\"ChangeVersionButton\" type=\"button\" value=\"changeVersion\" ";
				versionString += "onclick=\"MyObject.changePackageVersion(document.getElementById('versionSelect').options["
							"document.getElementById('versionSelect').selectedIndex].text)\""
						">Change Version</button>";
			}
		}

		htmlText.replace( "#VERSION#" , versionString );
		htmlText.replace( "#LOCALPATH#", QString::fromStdString( tPackage->mLocalPath.string() ) );

		QString tags;
		foreach( std::string& tTag, tPackage->mTags ) {
			QString tsTag = QString::fromStdString( tTag );
			tsTag.replace(" ","miraspaceisr");
			tags +=
				( tags.size() > 0 ? QString(", ") : QString("") ) +
				"<a href=\"TAG://" + tsTag + "\">"
				+ QString::fromStdString( tTag ) + "</a>";
		}

		htmlText.replace( "#TAGS#" , tags );
		htmlText.replace( "#DESCRIPTION#", QString::fromStdString( tPackage->mDescription ) );
		htmlText.replace( "#REPOSUBPATH#", QString::fromStdString( tPackage->mRepoSubPath ) );
		htmlText.replace( "#CHANGELOG#",   QString::fromStdString( tPackage->mChangeLog ) );

		QString dependencies;
		foreach( Package* tPackage, tPackage->mDependencies ) {
			Dependency* tDep = dynamic_cast<Dependency*>( tPackage );
			assert( tDep );
			QString tsDepLink = QString::fromStdString( tDep->mName );
			QString tsDep = tsDepLink;
			tsDepLink.replace(" ","miraspaceisr");
			if ( Package::Version() < tDep->mVersion )
				tsDep += " (" + QString::fromStdString( tDep->mVersion.str() ) + ")";
			if ( tDep->mDepFlag & Dependency::FACULTATIVE )
				tsDep += " [OPT]";
			if ( tDep->mDepFlag & Dependency::RUNTIME )
				tsDep += " [RUN]";
			dependencies +=
				( dependencies.size() > 0 ? QString(", ") : QString("") ) +
				"<a href=\"DEP://" + tsDepLink + "\">" +
				tsDep + "</a>";
		}
		htmlText.replace( "#DEPENDENCIES#" ,dependencies );
	}
	webView->setContent( htmlText.toLocal8Bit(), "application/xhtml+xml" );
	webView->page()->setLinkDelegationPolicy( QWebPage::DelegateAllLinks );
	webView->page()->mainFrame()->addToJavaScriptWindowObject("MyObject", this);
}

void PackageListWidget::updateFilterModel( QString const& filterText )
{
	proxyModel->setFilterExpression( QRegExp(filterText,Qt::CaseInsensitive ) );
}

void PackageListWidget::updateFilterMode()
{
	proxyModel->setActiveFilter(
			( mCxFilterNameSelect->checkState() == Qt::Checked ?
					PackageFilterModel::FilterName : PackageFilterModel::FilterNone )
			| ( mCxFilterDescriptionSelect->checkState() == Qt::Checked ?
					PackageFilterModel::FilterDescription : PackageFilterModel::FilterNone )
			| ( mCxFilterAuthorSelect->checkState() == Qt::Checked ?
					PackageFilterModel::FilterAuthor : PackageFilterModel::FilterNone )
					);
}

void PackageListWidget::updateTagSelection()
{
	std::vector< std::string > selectedTags;
	Q_FOREACH( QModelIndex const& index, tagView->selectionModel()->selectedIndexes() ) {
		selectedTags.push_back( index.data( Qt::DisplayRole ).toString().toStdString() );
	}

	proxyModel->setTagFilter( selectedTags );
	QString tFilter;
	foreach( std::string const& tag, proxyModel->getActiveTags() ) {
		tFilter += (tFilter.size() > 0 ? QString("|") : QString("")) + QString::fromStdString( tag );
	}

	mTagFilterModel->setRareTagFilter( mSliderRare->value() );
	mTagFilterModel->setFilterRegExp( QRegExp( tFilter ) );
	mTagFilterModel->sort(0);
}

void PackageListWidget::openLink( QUrl const& url )
{
	if ( url.scheme().contains("tag", Qt::CaseInsensitive ) ) {
		int i = 0;
		bool found = false;
		typedef map<string,int> mapType;
		mapType::iterator it = mCheckoutCore->mDB.tags.begin();

		QString searchString = url.authority();
		searchString.replace("miraspaceisr"," ");

		while ( it != mCheckoutCore->mDB.tags.end() && !found ) {

			if ( QString::fromStdString(it->first).toLower() == searchString ) {
				found = true;
			}
			else {
				++it;
				++i;
			}
		};
		if ( found ) {
			QModelIndex index = mTagFilterModel->mapFromSource( mTagModel->index( i, 0 ) );
			tagView->selectionModel()->select(
				index,
				QItemSelectionModel::SelectCurrent );
			tagView->scrollTo( index );
			updateTagSelection();
		}
	}
	else if ( url.scheme().contains( "dep", Qt::CaseInsensitive ) ) {
		int i = 0;
		bool found = false;

		QString searchString = url.authority();
		searchString.replace("miraspaceisr"," ");

		do {
			if ( QString::fromStdString( mCheckoutCore->mDB.getRootPackage()->mSubPackages[i]->mName ).toLower() == searchString ) {
				found = true;
			}
			else {
				++i;
			}
		} while ( (i < int( mCheckoutCore->mDB.getRootPackage()->mSubPackages.size() ) ) && !found );
		if ( found ) {
			QModelIndex index = proxyModel->mapFromSource( mPackageModel->index(i,0) );
			packageView->selectionModel()->setCurrentIndex(
				index,
				QItemSelectionModel::SelectCurrent | QItemSelectionModel::Rows  );
			packageView->scrollTo( index );
			updatePackageDetails( index );
		}
	}
}

void PackageListWidget::headerClicked( int column)
{
	if ( column == 0 )
		proxyModel->setSortRole( Qt::CheckStateRole );
	else
		proxyModel->setSortRole( Qt::DisplayRole );
}

void PackageListWidget::setHeaderResizing( int oldsections, int sections )
{
	if ( sections > 0 ) {
#if QT_VERSION > QT_VERSION_CHECK(5, 12, 0)
		for ( int i=0; i< (sections-1);++i)
			packageView->header()->setSectionResizeMode(i,QHeaderView::ResizeToContents);

		packageView->header()->setSectionResizeMode(sections-1,QHeaderView::Stretch);
#else
		for ( int i=0; i< (sections-1);++i)
			packageView->header()->setResizeMode(i,QHeaderView::ResizeToContents);

		packageView->header()->setResizeMode(sections-1,QHeaderView::Stretch);
#endif
	}
}

void PackageListWidget::showContextMenu( QModelIndex const& index )
{
	if ( !index.isValid() || index.column() != 0 )
		return;

	PackageGroup* tGroup = static_cast<PackageGroup*>( index.internalPointer() );
	Package* tPackage = dynamic_cast<Package*>( tGroup );

	if ( proxyModel->mapToSource( packageView->selectionModel()->currentIndex() ) != index ) {
		QItemSelection selection( index, index );
		QItemSelection proxySelection = proxyModel->mapSelectionFromSource( selection );

		packageView->selectionModel()->clearSelection();
		// for some reason the following line fails so set a proper selection
		// the current item will be set correctly, however the selection is wrong sometimes?!
		packageView->selectionModel()->setCurrentIndex( proxyModel->mapFromSource( index ), QItemSelectionModel::Current );
	}

	if ( tPackage ) {
		// display the context menu
		QMenu tMenu;

		if ( mCheckoutCore->mDB.isInstalled( tPackage, true )
				&& tPackage->mRepos.size() > 0 ) {
			// package is installed -> show uninstall options

			map<string,Package*> alternativePkgs = mCheckoutCore->mDB.alternativeVersions( tPackage );

			if ( alternativePkgs.size() > 0 ) {
				// create a sorted list of the alternative versions
				list<Package::Version> alternativeVersions;
				foreach( auto const& tAltPkg, alternativePkgs ) {
					list<Package::Version>::iterator iter = alternativeVersions.begin();
					while ((iter != alternativeVersions.end()) && (*iter > tAltPkg.second->mVersion))
						++iter;
					alternativeVersions.insert(iter, tAltPkg.second->mVersion);
				}

				// add the alternative versions to a new sub menu
				QMenu* tSubMenu = new QMenu("Change Version", &tMenu );
				foreach( auto v, alternativeVersions )
					tSubMenu->addAction( QString::fromStdString( v.str() ) );

				tMenu.addMenu( tSubMenu );
			}
			else {
				// use a dummy action as some kind of header, to avoid usage
				// errors by double clicking...

				// generate some kind of header for the menu
				QAction* header = new QAction( "Really uninstall?", &tMenu );
				header->setEnabled( false );

				tMenu.addAction( header );
				tMenu.addSeparator();
			}

			// now we can add the uninstall option
			QAction* uninstall = tMenu.addAction( "uninstall");

			// execute the menu
			QAction* tAction = tMenu.exec( QCursor::pos() );
			if ( tAction == uninstall ) {
				uninstallPackage();
			}
			else if ( tAction != NULL ) {
				// Use the old MIRA_PATH for the new version.
				alternativePkgs[ tAction->text().toStdString() ]->mLocalPath =
						mCheckoutCore->getPathForPackageUpdate(tPackage);

				// update selected
				uninstallPackage();
				emit packageInstallRequested( alternativePkgs[ tAction->text().toStdString() ] );
			}
		}
		else {
			// package is not installed -> show install options

			// generate some kind of header for the menu
			QAction* header = new QAction( "Install from:", &tMenu );
			header->setEnabled( false );

			tMenu.addAction( header );
			tMenu.addSeparator();

			// now we can add the options
			foreach( Url const& tUrl, tPackage->mRepos ) {
				string tRepo = mCheckoutCore->mDB.getRepoNameFromUrl( tUrl );
				tMenu.addAction( QString::fromStdString( tRepo ) );
			}

			// execute the menu
			QAction* tAction = tMenu.exec( QCursor::pos() );
			if ( tAction != 0 ) {
				installPackage( tAction->text() );
			}
		}
	}
}

void PackageListWidget::showMappedContextMenu( QModelIndex const& index )
{
	if ( !index.isValid() || index.column() != 0 )
		return;

	QModelIndex mappedIndex = proxyModel->mapToSource( index );
	showContextMenu( mappedIndex );
}

void PackageListWidget::headerContextMenu(QPoint const& point)
{
	QMenu menu(this);

	QString text = packageView->header()->isSectionHidden(2) ?
			"show repository" : "hide repository";
	menu.addAction( text, this, SLOT(toggleRepoVisibility()) );
	menu.exec( packageView->header()->mapToGlobal( point ) );
}

void PackageListWidget::toggleRepoVisibility()
{
	packageView->header()->setSectionHidden(2,!packageView->header()->isSectionHidden(2));
}

///////////////////////////////////////////////////////////////////////////////

}
