/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CmdLineOptions.C
 *    Description.
 *
 * @author Ronny Stricker
 * @date   2012/02/29
 */

#include <app/CmdLineOptions.h>

#include <core/Repository.h>

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

void usage( vector<string> const& repos )
{
	std::cout <<
			"mirapackage <command> [parameters]\n\n"
			"Simplifies checking out branches and their dependencies, by\n"
			"examining the available branches on the specified repositories.\n"
			"The dependencies are obtained from the *.package files within the\n"
			"branches. The checkout location is determined using the mountdir.xml\n"
			"files in the branches. A mountdir.xml file contains the path in the\n"
			"source tree, that the branch corresponds to. Subdirectories of the\n"
			"branch are checked out relative to that path.\n"
			"\n"
			"By default the following repositories are added:\n";
	foreach( string path, repos ) {
		cout << "   " << path << std::endl;
	}
	cout << "\n"
			"To add further repositories, type:\n"
			"  > mirapackage --addrepo [name] [url of branch within repository]\n"
			<< endl;
}

bool listPackages( string const& listOption, MIRAPackage& core )
{
	if ( listOption != "all" && listOption != "installed" &&
			listOption != "available" && listOption != "export") {
		return false;
	}

	cout << "list of " << listOption << " packages:" << endl;
	foreach( PackageGroup* tGroup, core.mDB.getRootPackage()->mSubPackages ) {
		if ( listOption == "all" ||
				( listOption == "installed" && core.mDB.isInstalled( tGroup ) ) ||
				( listOption == "available" && !core.mDB.isInstalled( tGroup ) ) )
		cout << "  " << ((core.mDB.isInstalled( tGroup ) && listOption == "all") ? "[installed] " : "" )
			<< tGroup->mName << " - "
			<< tGroup->mDescription
			<< endl;
	}
	return true;
}

bool exportPackages( string const& fileName, MIRAPackage& core )
{
	core.exportDB( fileName );
	return true;
}

bool importPackages( string const& fileName, MIRAPackage& core )
{
	// try to get working mira path, if not already selected
	if ( core.mInstallRoot.empty() )
		core.resolveMIRAPath();

	core.importDB( fileName );

	core.doIt();
	core.save();
	return true;
}

bool showPackageDetails( string const& packageName, MIRAPackage& core )
{
	vector<Package*> tPackages = core.mDB.getRootPackage()->
			findPackages( packageName, true );

	cout << "The following packages have been found:\n\n";
	foreach( Package* tPackage, tPackages ) {
		cout << "----------------------\n";
		cout << "Name: " << tPackage->mName << "\n";
		cout << "Version: " << tPackage->mVersion.str() << "\n";
		cout << "Description: " << tPackage->mDescription << "\n";
		cout << "Author: " << tPackage->mAuthor << "\n";
		cout << "Dependencies: ";
		foreach( Package* dep, tPackage->mDependencies ) {
			cout << dep->mName << " ";
		}
		cout << "\n";
		cout << "Repository: " << core.mDB.getRepoNameFromUrl(tPackage->mCurrentRepo) << "\n";
		cout << "RepoSubPath: " << tPackage->mRepoSubPath << "\n";
		cout << "FileName: " << tPackage->mFileName << "\n";
		cout << "LocalPath: " << tPackage->mLocalPath.string() << "\n";
		cout << "\n\n";
	}
	return true;
}

bool installPackages( std::vector<std::string> packageNames, MIRAPackage& core )
{
	// try to get working mira path, if not already selected
	if ( core.mInstallRoot.empty() )
		core.resolveMIRAPath();

	foreach( string const& tPackage, packageNames ) {
		// find all packages with that name
		vector<Package*> packages = core.mDB.
				getRootPackage()->findPackages( tPackage,true );
		if ( packages.size() ) {
			// try to install the "best" package
			Package* tPackage = core.mDB.stdSource( packages );
			if ( tPackage )
				core.addPackageToCheckoutPlan( tPackage );
			else {
				MIRA_LOG(NOTICE) << "Unable to find standard package for '"
				                << tPackage << "'";
			}
		}
		else {
			MIRA_THROW(XLogical, "Cannot find package with name \""+tPackage+"\"!");
		}
	}
	core.doIt();
	core.save();
	return true;
}

bool deinstallPackages( vector<string> packageNames, MIRAPackage& core )
{
	// if "AllPackages" is given as an option, the program tries
	// to deinstall all packages
	if ( packageNames.size() == 1 && packageNames[0] == "AllPackages" ) {
		// add all known packages
		packageNames.clear();
		foreach( Package* tPackage, core.mDB.mInstalledPackages ) {
			packageNames.push_back( tPackage->mName );
			cout << " deinstall " << tPackage->mName << endl;
		}
	}
	foreach( string const& tPackage, packageNames ) {
		// find all packages with that name
		vector<Package*> packages = core.mDB.
				getRootPackage()->findPackages( tPackage,true );
		if ( packages.size() ) {
			// remove all packages with that name
			foreach( Package* package, packages ) {
				try {
					core.addPackageForRemoval( package );
				}
				catch( Exception& ex ) {
					cout << ex.message() << endl;
				}
			}
		}
		else {
			cout << "ERROR: cannot find package with name \""
					<< tPackage << "\"!" <<  endl;
		}
	}
	core.doIt();
	core.save();
	return true;
}

bool listDependencies( string const& packageName, MIRAPackage& core )
{
	vector<Package*> tPackages = core.mDB.getRootPackage()->
			findPackages( packageName, true );
	Package* tPackage = core.mDB.stdSource( tPackages );
	if ( tPackage ) {
		vector<string> tDependencies;
		foreach( Package* dep, tPackage->mDependencies ) {
			tDependencies.push_back( dep->mName );
		}
		listInstallSequence( tDependencies, core );
	}
	return true;
}

bool showDependencyTree( string const& packageName, MIRAPackage& core, const string& prefix )
{
	vector<Package*> tPackages = core.mDB.getRootPackage()->
			findPackages( packageName, true );
	Package* tPackage = core.mDB.stdSource( tPackages );
	if ( tPackage ) {
		foreach( Package* dep, tPackage->mDependencies ) {
			cout << prefix << dep->mName << endl;
			if ( dep->mName != "MIRABase" && dep->mName != "MIRAFramework" ) {
				showDependencyTree( dep->mName, core, prefix + "  " );
			}
		}
	}
	return true;
}

bool showDependencyGraph( vector<string> const& packageNames, MIRAPackage& core, list<string>& processed )
{
	bool res = false;
	foreach( string const& packageName, packageNames )
		res |= showDependencyGraph( packageName, core, processed );
	return res;
}

bool showDependencyGraph( string const& packageName, MIRAPackage& core, list<string>& processed )
{
	vector<Package*> tPackages = core.mDB.getRootPackage()->
			findPackages( packageName, true );
	Package* tPackage = core.mDB.stdSource( tPackages );
	if ( tPackage ) {
		processed.push_back( tPackage->mName );
		tPackage->selectStdRepo( &core.mDB );
		cout << "\t" << tPackage->mName << " [shape=box,label=\"" << tPackage->mName;
		cout << "\\n" << tPackage->mVersion.str();
		RepositoryPtr repo = core.mDB.getRepoFromUrl(tPackage->mCurrentRepo);
		if ( repo )
			cout << "\\n" << repo->name;
		else
			cout << "\\n(" << tPackage->mCurrentRepo << ")";
		cout << "\"";
		if ( tPackage->mDevelState == Package::DEPRECATED )
			cout << ",style=dashed,color=grey";
		else if ( tPackage->mDevelState == Package::EXPERIMENTAL )
			cout << ",style=dashed";
		else if ( tPackage->mDevelState == Package::RELEASE )
			cout << ",style=bold";
		cout << "]" << endl;
		foreach( Package* dep, tPackage->mDependencies ) {
			if ( dep->mName != "MIRABase" && dep->mName != "MIRAFramework" ) {
				cout << "\t" << tPackage->mName << " -> " << dep->mName;
				Dependency* dep2 = dynamic_cast<Dependency*>(dep);
				if ( dep2 && dep2->mDepFlag & (Dependency::FACULTATIVE | Dependency::RUNTIME) ) {
					cout << " [style=dashed";
					if ( dep2->mDepFlag & Dependency::FACULTATIVE )
						cout << ",color=grey";
					cout << "]";
				}
				cout << endl;
				if ( find(processed.begin(), processed.end(), dep->mName) == processed.end() ) {
					showDependencyGraph( dep->mName, core, processed );
				}
			}
		}
	} else {
		processed.push_back( packageName );
		cout << "\t" << packageName << " [shape=box,label=\"" << packageName << "\",style=dotted]" << endl;
	}
	return true;
}

bool listDependents( string const& packageName, MIRAPackage& core )
{
	vector<Package*> tPackages = core.mDB.getRootPackage()->
			findPackages( packageName, true );
	Package* tPackage = core.mDB.stdSource( tPackages );
	if ( tPackage ) {
		set<string> tDependents;
		tDependents.insert( packageName );
		unsigned long oldSz = 0, sz = 1;
		while (oldSz != sz) {
			set<Package*> tAll = core.mDB.getRootPackage()->getPackages(true);
			foreach( Package* p, tAll ) {
				foreach( Package* dep, p->mDependencies ) {
					foreach( string depd, tDependents ) {
						if ( dep->mName == depd ) {
							tDependents.insert( p->mName );
							break;
						}
					}
				}
			}
			oldSz = sz;
			sz = tDependents.size();
		}
		tDependents.erase( packageName );
		vector<string> tDependentList( tDependents.begin(), tDependents.end() );
		listInstallSequence( tDependentList, core );
	}
	return true;
}

bool showDependentTree( string const& packageName, MIRAPackage& core, const string& prefix )
{
	vector<Package*> tPackages = core.mDB.getRootPackage()->
			findPackages( packageName, true );
	Package* tPackage = core.mDB.stdSource( tPackages );
	if ( tPackage ) {
		set<string> tDependents;
		set<Package*> tAll = core.mDB.getRootPackage()->getPackages(true);
		foreach( Package* p, tAll ) {
			foreach( Package* dep, p->mDependencies ) {
				if ( dep->mName == packageName ) {
					tDependents.insert( p->mName );
					break;
				}
			}
		}
		foreach( string const& name, tDependents ) {
			cout << prefix << name << endl;
			showDependentTree( name, core, prefix + "  " );
		}
	}
	return true;
}

bool listInstallSequence( vector<string> packageNames, MIRAPackage& core )
{
	vector<Package*> tPackages;
	foreach( string const& package, packageNames ) {
		vector<Package*> tPackageCandidates = core.mDB.getRootPackage()->
			findPackages( package, true );
		Package* tPackage = core.mDB.stdSource( tPackageCandidates );
		if ( tPackage )
			tPackages.push_back( tPackage );
	}
	vector<Package*> tInstallSequence = core.getInstallSequence( tPackages );
	foreach( Package* tPackage, tInstallSequence ) {
		cout << " " << tPackage->mName;
	}
	cout << endl;
	return true;
}

///////////////////////////////////////////////////////////////////////////////

} // namespace
