/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Database.h
 *
 * @author Erik Einhorn, Ronny Stricker
 * @date   2011/02/20
 */

#include <QObject>

#include <utils/Path.h>
#include <utils/EnumToFlags.h>

#include <factory/Factory.h>
#include <serialization/XMLSerializer.h>
#include <serialization/adapters/std/list>
#include <serialization/adapters/std/set>
#include <serialization/adapters/std/map>
#include <serialization/adapters/std/vector>
#include <serialization/adapters/boost/shared_ptr.hpp>

#include <core/Url.h>
#include <core/Package.h>
#include <core/PackageGroup.h>
#include <core/Dependency.h>
#include <core/Repository.h>

#ifndef _MIRA_DATABASE_H_
#define _MIRA_DATABASE_H_

namespace mira {

///////////////////////////////////////////////////////////////////////////////

// Some forward declarations
class Repository;
class PromptProvider;

///////////////////////////////////////////////////////////////////////////////

/**
 * @brief The package database.
 * The database contains all packages (the localy installed ones as
 * well as the web repositories. Furthermore, the database keeps track
 * of the repositories and is responsible for building a tag index.
 * Last but not least the database handles the list of installed packages
 * and the checkout plan. The checkout plan contains all actions which
 * should be performed in the future (e.g. install or uninstall).
 */
class Database : public QObject
{
	Q_OBJECT

public:
	/**
	 * @brief Reflect function.
	 * store repository and package list.
	 */
	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Repos", repos, "");
		r.member("Packages", rootPackage, "");
		// pass the prompt provider to the restored packages
		setRepositoryPromptProvider( mPromptProvider );
	}

public:
	enum QueryType {
		QUERY_NAME = 0x1,
		QUERY_DESCRIPTION = 0x2,
		QUERY_TAGS = 0x4
	};

	enum Action {
		NONE = 0x1,
		DONOTINSTALL = 0x2,	/// The user realy dont want to install this package so don't bother him about this package any more
		INSTALL = 0x4, 		/// The package sould be installed
		UNINSTALL = 0x8, 	/// The package should be uninstalled
		DEPENDENCY = 0x10 	/// package is installed / uninstalled as a dependency
	};

	MIRA_ENUM_TO_FLAGS_INCLASS( QueryType );
	MIRA_ENUM_TO_FLAGS_INCLASS( Action );

	Database();

	virtual ~Database();

	/**
	 * @brief Try to load the database stored in the given XML document.
	 */
	void load(const XMLDom& content);

	/**
	 * @brief Try to store the database in the given XML document.
	 */
	void store(XMLDom& content) const;

	/**
	 * @brief Create Database with all repositories and a list of installed
	 * packages.
	 * The database will have a depth of 1 -> all packages are added to the
	 * root package. The (mira path part) of the local install path of every
	 * package is mapped to unique ids: e.g. the local path of a package
	 * installed to /local/mira/subpath is mapped to 0/subpath.
	 * @param ioDB database the packages should be added to.
	 */
	void exportInstalledPackages( Database& ioDB, std::map<Path,std::string>& oPathMap ) const;

	/**
	 * @brief Import Repositories.
	 * The repositories will be added (if not already covered by the existing
	 * repositories. See getRepoFromUrl().
	 * @param path Path of the MIRA Package Export (mpe) file.
	 */
	void importRepositories(Path const& path);

	/**
	 * @brief Remove and destroy all packages.
	 * Since the root package cannot be deleted, all children of the root
	 * package are deleted instead.
	 */
	void clearPackages();

	/**
	 * @brief Clean up the package list.
	 * Remove local path from packages which are no longer installed.
	 * @param rootPackage root package to start with cleanup -> rootpackage if NULL
	 */
	void cleanUp( PackageGroup* rootPackage = NULL );

	/**
	 * @brief Reset information about installed packages and forget about the checkout plan.
	 */
	void resetPackageFlags();

	/**
	 * @brief Return the dependency tree for the given package.
	 * The returned package and all of its children are copies from the original
	 * packages. Therefore, you have to keep track of the returned package on
	 * your own. If fullDependencies are set to true, the function will return
	 * all recursive dependencies. Otherwise, the function follows the
	 * default sources and source types of the packages only (e.g. preferes
	 * source packages over binary ones).
	 * @param[in] s The package for which the dependency graph sould be returned.
	 * @param[in] fullDependencies return full dependencies or only default ones.
	 * @return The dependency tree (you are responsible for deletion)
	 */
	Dependency* getInstallDependencies(Package const& s, bool fullDependencies) const;

	/**
	 * @brief Remove and destroy all repositories.
	 */
	void clearRepositories();

	/**
	 * @brief Add given repository.
	 * Database will take ownership of the given repository and
	 * sets the promptProvider.
	 */
	void addRepository( RepositoryPtr repo );

	/**
	* @brief Add given package to package root.
	* @see PackageGroup::add() for further details.
	*/
	void addPackage( PackageGroup* package );

	/**
	 * @brief Remove package with the given pointer.
	 * @see PackageGroup::removeChild() for further details.
	 * @param package
	 */
	void removePackage( PackageGroup* package );

	/**
	 * @brief Return true if package(group) is installed.
	 * If a package is passed, the function checks if the package is (or will
	 * be installed). If you pass a packageGroup, the function will do the
	 * same for all the children and return true if one of them is installed.
	 * @param[in] package The package for which the check should be performed.
	 * @param[in] regardCheckoutPlan Set to true if the function should return
	 * true if a package is scheduled for install.
	 */
	bool isInstalled( PackageGroup const* package, bool regardCheckoutPlan = false ) const;

	/**
	 * @brief Check if a package with the given name is installed.
	 * Obtains all packages with the given name from the root package
	 * and calls isInstalled(PackageGroup*) for every one.
	 * @see isInstalled( PackageGroup*, bool ) for further details.
	 */	 
	bool isInstalled( std::string const& name, bool regardCheckoutPlan = false ) const;

	/**
	 * @brief Return true if changes will be applied to the package.
	 * True if package will be installed or unistalled if checkout plan is applied.
	 * @param[in] package Pointer to the package.
	 */
	bool willChange( PackageGroup const* package ) const;

	/**
	 * @brief Return true if changes will be applied to the package.
	 * True if a PRIMARY package with the given name will be installed
	 * or unistalled if checkout plan is applied.
	 * @param[in] name Name of the Package
	 */
	bool willChange( std::string const& name ) const;

	/**
	 * @brief Get flag from install plan for given package.
	 * If the package cannot be found in the install plan, the function
	 * returns NONE.
	 * @param[in] package Desired package.
	 * @return Flag for given package.
	 */
	Action getFlag( PackageGroup const* package ) const;

	/**
	 * @brief Get flag for the first PRIMARY package with the given name.
	 * Calls getFlag( PackageGroup* ) for every primary package with the given name
	 * and returns if a flag for one package exists.
	 */
	Action getFlag( std::string const& name ) const;

	/**
	 * @brief Returns true if the package is already added to the checkout plan.
	 */
	bool hasFlag( PackageGroup const* package ) const;

	/**
	 * @brief Get identical package for given one.
	 * @see PackageGroup::getIdentical( PackageGroup* ) for further information
	 */
	PackageGroup* getIdentical( PackageGroup const* other,
			Package::Type ignoredFlags =  Package::UNSPECIFIED ) const;

	/**
	 * @brief Get vector of identical package for given one.
	 * @see PackageGroup::getAllIdentical( PackageGroup*, vector<PackageGroup>& ) for further information
	 */
	void getAllIdentical( PackageGroup const* other, std::vector<PackageGroup*>& identical,
			Package::Type ignoredFlags =  Package::UNSPECIFIED) const;

	/**
	 * @brief Set the install flag for the given package.
	 * Adds or overwrites the install flag for the given package in the
	 * install plan.
	 */
	void setFlag( Package* package, Action flag );

	/**
	 * @brief Add package to the list of installed packages.
	 */
	void setInstalled( Package* package )
	{
		mInstalledPackages.insert( package );
	}

	/**
	 * @brief Cycle through all packages and build tag index.
	 * Tags from packages with the same name are not added twice.
	 */
	void buildTagIndex();

	/**
	 * @brief Return a const reference to the root package.
	 */
	PackageGroup const* getRootPackage() const
	{
		return &rootPackage;
	}

	/**
	 * @brief Return the best std source from the given vector.
	 * returns primary packages with the highest development state and prefers Source
	 * packages over binary ones.
	 * If the version is set, the function requires the packages to be binary
	 * compatible with the specified version
	 */
	template <class PT>
	Package* stdSource( std::vector<PT*> const& list,
			Package::Version requiredVersion = Package::Version() ) const;

	/**
	 * @brief Return a map of packages with the same package type but different
	 * versions as the given package.
	 * @param package package for which the alternative versions should be searched.
	 * @return map containing of version string an package.
	 */
	std::map<std::string,Package*> alternativeVersions( Package const* package ) const;

	/**
	 * @brief Return pointer to the package with a newer version than the given
	 * package. The package type will be the same.
	 * If no such package exists, the pointer will be NULL;
	 * @param oldPackage the package for which a newer version should be searched.
	 */
	Package* newVersionAvailable( Package const* oldPackage ) const;

	/**
	 * @brief Mark vector of version according to the required version given.
	 * Set the flags of the packages according to the given version.
	 * @param url
	 * @return
	 */
	template <class PT>
	void markVersions( std::vector<PT*>& list, Package::Version const& version ) const;

	/**
	 * @brief Mark vector of version according to the required version given.
	 * Set the flags of the packages according to the given version.
	 * @param url
	 * @return
	 */
	template <class PT>
	void markVersion( PT* package, Package::Version const& version ) const;

	/**
	 * @brief Recursively select the std package for the given dependency tree.
	 * The functions traverses the given dependency tree and schedules the best
	 * matching packages for install. If one of the packages in the tree is
	 * already scheduled for install, it will be preferred.
	 * The packages which should be installed to fulfill the dependency tree
	 * are scheduled for install.
	 * @param depTree contains the whole dependency tree
	 */
	void recursiveSelectStdSource( PackageGroup* depTree );

	/**
	 * @brief Return the responsible repository for the given url.
	 * Returns NULL if no matching repository can be found.
	 */
	RepositoryPtr getRepoFromUrl(Url const& url) const;

	/**
	 * @brief Return name of the responsible repository for the given url.
	 * Returns a empty string if no matching repository can be found.
	 */
	std::string getRepoNameFromUrl(Url const& url) const;

	/**
	 * @brief Return repository with the given repository name.
	 * Returns NULL if no matching repository can be found.
	 */
	RepositoryPtr getRepoFromName(std::string const& name) const;

	/**
	 * @brief Return repository url for the given repository name.
	 * Returns a empty path if no matching repository can be found.
	 */
	Url getRepoUrlFromName(std::string const& name) const;

	/**
	 * @brief Try to find the responsible Repository for the given local path.
	 * Ask every repository if it is responsible.
	 * @see Repository::isResponsibleFor() for further details.
	 */
	RepositoryPtr getRepositoryForLocalFile( mira::Path const& localPath );

	/**
	 * @brief Set the active prompt provider.
	 * The prompt provider is propagated to the repositories as well.
	 */
	void setRepositoryPromptProvider( PromptProvider* prompt );

	/**
	 * @brief return the complete map of repositories.
	 */
	std::map<std::string, RepositoryPtr> getRepositoryMap() const
	{
		return repos;
	}

	/**
	 * @brief Return the prompt provider.
	 * @return Pointer to the prompt provider.
	 */
	PromptProvider* getPromptProvider() const
	{
		return mPromptProvider;
	}

signals:
	void flagChanged( PackageGroup* group );

protected:
	/**
	 * @brief Return the dependency tree for the given package.
	 * @see getDependentPackages() for further information.
	 * This is the implementation of the recursive dependency discovery.
	 * The functions throws exceptions if the fullDependencies are not desired
	 * and one of the required dependencies cannot be fulfilled. Otherwise,
	 * if full dependencies are not requested, the packages are marked with
	 * VERSION_INCOMP. Therefore, you should check afterwards if the (selected)
	 * dependencies are consistent.
	 */
	void gatherDependencies( Dependency* package,
	                         bool fullDependencies,
	                         int recursionLevel = 15 ) const;

public:
	/// constructed tag index, @see buildTagIndex()
	std::map<std::string,int> tags;

	/// map of repositories <name, repository>
	std::map<std::string, RepositoryPtr> repos;

	/// map of generic repositories (one for every repository typ)
	/// these repositories are used to uninstall packages for which the
	/// original repository is unknown
	std::map<std::string, RepositoryPtr> genericRepos;

	/// Typedef for all pending operations
	typedef std::map<Package*,Action> ActionPlan;

	/// checkout plan
	ActionPlan mActionPlan;

	/// list of already installed packages
	std::set<Package*> mInstalledPackages;

protected:
	PackageGroup rootPackage;		///< the root package
	PromptProvider* mPromptProvider;///< active prompt provider
};

///////////////////////////////////////////////////////////////////////////////

template <class PT>
inline Package* mira::Database::stdSource( std::vector<PT*> const& list,
		Package::Version requiredVersion ) const
{
	Package* tReturn = NULL;
	Package::Type tType = Package::ETC;
	int develState = -1;
	Package::Version foundVersion = Package::Version();
	foreach( PT* group, list ) {
		Package* pkg =  dynamic_cast<Package*>( group );
		// regard only primary packages
		if ( pkg && pkg->isPrimary() ) {
			// Skip packages, which are dependencies, which already have the
			// Dependency::VERSION_INCOMP flag.
			Dependency* dep = dynamic_cast<Dependency*>(pkg);
			if (!dep || (!(dep->mDepFlag & Dependency::VERSION_INCOMP))) {
				// prefer release packages
				if ( int(pkg->mDevelState) >= develState ) {
					// prefer package with highest version
					if ( (requiredVersion.isIndifferent() && pkg->mVersion >= foundVersion ) ||
							pkg->mVersion.isBinaryCompatible( requiredVersion ) ) {
						// prefer source packages
						if ( pkg->mType >= tType ) {
							tReturn = pkg;
							develState = pkg->mDevelState;
							foundVersion = pkg->mVersion;
						}
					}
				}
			}
		}
	}
	return tReturn;
}

template <class PT>
void mira::Database::markVersions( std::vector<PT*>& list, Package::Version const& version ) const
{
	foreach( PackageGroup* group, list ) {
		markVersion( group, version );
	}
}

template <class PT>
void mira::Database::markVersion( PT* package, Package::Version const& version ) const
{
	Dependency* dependency =  dynamic_cast<Dependency*>( package );
	if ( dependency ) {
		if ( dependency->mVersion.isBinaryCompatible( version ) ||
				version.isIndifferent() ) {
			dependency->mDepFlag |= Dependency::VERSION_OK;
		}
		else {
			dependency->mDepFlag |= Dependency::VERSION_INCOMP;
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

}

#endif /* _MIRA_DATABASE_H_ */
