/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ConfigurationEditor.C
 *    Allows to edit the current configuration and adding new units
 *
 * @author Tim Langner, Erik Einhorn, Christof Schröter
 * @date   2011/12/08
 */

#include <views/ConfigurationEditor.h>

// include Qt specific stuff
#include <QCheckBox>
#include <QComboBox>
#include <QLabel>
#include <QLineEdit>
#include <QPlainTextEdit>
#include <QPushButton>

#include <serialization/Serialization.h>
#include <xml/XMLDomPreprocessor.h>

#include <fw/MicroUnit.h>
#include <fw/UnitManager.h>

#include <widgets/SelectionListDialog.h>

namespace mira { 

///////////////////////////////////////////////////////////////////////////////

ConfigurationEditor::ConfigurationEditor() : mWidget(NULL)
{

}

QWidget* ConfigurationEditor::createPartControl()
{
	mWidget = new QWidget(this);

	QPushButton* addUnit = new QPushButton("Add Unit Config", mWidget);
	connect(addUnit, SIGNAL(clicked()), this, SLOT(addUnitConfig()));

	mCommentsBox = new QCheckBox("Add Param Descriptions", mWidget);

	QPushButton* format = new QPushButton("Reformat Config", mWidget);
	connect(format, SIGNAL(clicked()), this, SLOT(formatConfig()));

	mEdit = new QPlainTextEdit(mWidget);
	XMLDom xml;
	xml.root().add_comment("Add content here");
	mEdit->setPlainText(xml.saveToString().c_str());

	QTextCursor cursor(mEdit->document());
	cursor = mEdit->document()->find("</root>", cursor);
	if (!cursor.isNull()) {
		cursor.movePosition(QTextCursor::StartOfLine);
		cursor.insertText("\n");
		cursor.movePosition(QTextCursor::Up);
		mEdit->setTextCursor(cursor);
	}
	
	mServiceWidget = new QWidget(mWidget);
	mServiceBox = new QComboBox(mServiceWidget);

	mLoadButton = new QPushButton("Load (run) this config", mWidget);
	connect(mLoadButton, SIGNAL(clicked()), this, SLOT(loadConfig()));

	QVBoxLayout* leftlayout = new QVBoxLayout;
	leftlayout->addWidget(addUnit);
	leftlayout->addWidget(mCommentsBox);
	leftlayout->addSpacing(format->height());
	leftlayout->addWidget(format);
	leftlayout->addStretch();

	QHBoxLayout* serviceLayout = new QHBoxLayout(mServiceWidget);
	serviceLayout->addWidget(new QLabel("Target Framework:", mServiceWidget));
	serviceLayout->addWidget(mServiceBox, 1); // this takes the space

	QVBoxLayout* rightlayout = new QVBoxLayout;
	rightlayout->addWidget(mEdit);
	rightlayout->addWidget(mServiceWidget);
	rightlayout->addWidget(mLoadButton);
	
	QHBoxLayout* hlayout = new QHBoxLayout(mWidget);
	hlayout->addLayout(leftlayout);
	hlayout->addLayout(rightlayout);

	updateServicesList();
	mUpdateServicesTimer = startTimer(2000);

	return mWidget;
}

void ConfigurationEditor::timerEvent(QTimerEvent* event)
{
	if(event->timerId() == mUpdateServicesTimer)
		updateServicesList();
}

void ConfigurationEditor::updateServicesList()
{
	const std::string localFramework = MIRA_FW.getGlobalID();
	const QString localEntry = QString::fromStdString("local  : " + localFramework);
	RPCManager& rpcman = MIRA_FW.getRPCManager();
	std::list<std::string> services = rpcman.queryServicesForInterface("IConfigurationLoader");

	QString current = mServiceBox->currentText();

	mServiceBox->clear();
	mServices.clear();
	foreach(const std::string& s, services) {
		mServices.push_back(s);
		if (s == localFramework) {
			mServiceBox->addItem(localEntry);
			if (mWaitForLocal) { // prefer to show local framework when it is discovered
				current = localEntry;
				mWaitForLocal = false;
			}
		}
		else
			mServiceBox->addItem(QString::fromStdString("remote : " + s));
	}

	// keep selection, if still valid
//	mServiceBox->setCurrentText(current); // works in Qt5, but not Qt4
	int index = mServiceBox->findText(current);
	if (index > 0)
		mServiceBox->setCurrentIndex(index);

	mServiceWidget->setVisible(mServiceBox->count() > 1);
}

void ConfigurationEditor::addUnitConfig()
{
	SelectionListDialog<ClassProxy> d(this);
	d.setWindowTitle(tr("Choose a unit"));
	d.setHeaderLabels(QStringList() << "Unit");
	d.resize(600,400);

	// query all unit classes from the class factory:
	typedef std::map<std::string, ClassProxy > ClassMap;
	ClassMap unitClasses = MicroUnit::CLASS().getDerivedClasses( );

	foreach(ClassMap::value_type i, unitClasses)
	{
		if(i.second.isAbstract())
			continue;

		std::string name = i.first; // use class name

		QStringList labels;
		labels << name.c_str();
		d.addItem(labels, i.second);
	}

	if(d.exec()==0)
		return;

	auto selection = d.selectedItems();
	if(selection.empty())
		return; // nothing chosen

	// get the parameters of the unit by creating it and serializing it into xml
	const ClassProxy& unitClass = selection.begin()->second;
	MicroUnit* unit = unitClass.newInstance<MicroUnit>();
	XMLDom xml;
	XMLDom::iterator root = xml.root();
	XMLSerializer s(root);
	s.serialize("unit", unit, "");
	delete unit;
	root.find("unit").add_attribute("id", unitClass.getName());

	// remove <root> + </root> tags
	xml.root().replace(root.find("unit"));

	std::string content = xml.saveToString();

	// remove <?xml ... ?> header line
	content.erase(0, content.find("\n"));

	// remove comments
	if (!mCommentsBox->isChecked()) {
		size_t startComment = content.find("<!--");
		while (startComment != std::string::npos) {
			size_t endComment = content.find("-->\n", startComment + 4);
			if (endComment == std::string::npos)
				startComment = std::string::npos;
			else {
				content.erase(startComment, endComment - startComment + 4);
				startComment = content.find("<!--", startComment); // search next
			}
		}
	}
	
	mEdit->insertPlainText(content.c_str());
	formatConfig();
}

void ConfigurationEditor::formatConfig()
{
	std::string content = mEdit->toPlainText().toStdString();
	XMLDom xml;
	try {
		xml.loadFromString(content);
	}
	catch(...) { return; }

	mEdit->setPlainText(xml.saveToString().c_str());

	QTextCursor cursor(mEdit->document());
	cursor = mEdit->document()->find("<unit", cursor);
	while (!cursor.isNull()) {
		cursor.movePosition(QTextCursor::StartOfLine);
		cursor.insertText("\n");
		cursor.movePosition(QTextCursor::Down);
		cursor = mEdit->document()->find("<unit", cursor);
	}

	cursor = mEdit->document()->find("</root>", cursor);
	if (!cursor.isNull()) {
		cursor.movePosition(QTextCursor::StartOfLine);
		cursor.insertText("\n");
		cursor.movePosition(QTextCursor::Up);
		mEdit->setTextCursor(cursor);
	}
}

void ConfigurationEditor::loadConfig()
{
	std::string content = mEdit->toPlainText().toStdString();
	XMLDom xml;
	xml.loadFromString(content);

	const std::string& service = mServices[mServiceBox->currentIndex()];
	RPCManager& rpcman = MIRA_FW.getRPCManager();
	rpcman.call<void>(service, "loadConfig", xml);
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::ConfigurationEditor, mira::EditorPart );
