/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file KeepDistanceTask.h
 *    Task for keeping a specified distance to obstacles.
 *
 * @author Erik Einhorn
 * @date   2013/04/17
 */

#ifndef _MIRA_KEEPDISTANCETASK_H_
#define _MIRA_KEEPDISTANCETASK_H_

#include <navigation/Task.h>

namespace mira { namespace navigation {

///////////////////////////////////////////////////////////////////////////////

/**
 * Task for keeping a specified distance to obstacles. Obstacles are
 * not "just" avoided but the robot tries to keep away from them in the
 * specified distance.
 */
class MIRA_NAVIGATION_EXPORT KeepDistanceTask : public SubTask
{
	MIRA_OBJECT(KeepDistanceTask);
public:

	/// Creates a task with all distances 0.
	KeepDistanceTask() : minDistance(0.0f), desiredDistance(0.0f), criticalDistance(0.0f) {}

	/**
	 * @brief Creates a task with specified min/desired distances, criticalDistance = minDistance - 0.05.
	 * @param[in] minDist The minimum distance to keep from obstacles.
	 * @param[in] desiredDist The desired distance to keep from obstacles.
	 */
	KeepDistanceTask(float minDist, float desiredDist) :
		minDistance(minDist),
		desiredDistance(desiredDist),
		criticalDistance(minDist - 0.05f)
	{}

	/**
	 * @brief Creates a task with specified min/desired/critical distances.
	 * @param[in] minDist The minimum distance to keep from obstacles.
	 * @param[in] desiredDist The desired distance to keep from obstacles.
	 * @param[in] criticalDist The distance that triggers recovery.
	 */
	KeepDistanceTask(float minDist, float desiredDist, float criticalDistance) :
		minDistance(minDist),
		desiredDistance(desiredDist),
		criticalDistance(criticalDistance)
	{}

	/// The reflect method.
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, SubTask);
		r.property("MinDistance", minDistance, "The minimum distance to keep from obstacles.", 0.0f);
		r.property("DesiredDistance", desiredDistance, "The desired distance to keep from obstacles.", 0.0f);
		r.property("CriticalDistance", criticalDistance, "The critical distance that triggers recovery "
		           "(should be below MinDistance).", 0.0f);
	}

public:

	float minDistance;
	float desiredDistance;
	float criticalDistance;

};

///////////////////////////////////////////////////////////////////////////////

}}

#endif
