/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ILocalization.h
 *    Interface for localization services.
 *
 * @author Tim Langner
 * @date   2013/02/15
 */

#ifndef _MIRA_ILOCALIZATION_H_
#define _MIRA_ILOCALIZATION_H_

#include <transform/Pose.h>

namespace mira { namespace localization {

///////////////////////////////////////////////////////////////////////////////

/**
 * Interface for localization services
 */
class ILocalization
{
public:
	/// The reflect method.
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.interface("ILocalization");
		r.method("setInitPose", &ILocalization::setInitPose, this,
		         "Initializes localization at the given pose. "
		         "Pose is assumed to be in the reference frame.",
		         "pose", "the pose to set", PoseCov2());
		r.method("setInitPose", &ILocalization::setInitPoseFrame, this,
		         "Initializes localization at the given pose using the given frame.",
		         "pose", "the pose to set", PoseCov2(),
		         "poseFrameID", "the frame that the pose is defined in", "/GlobalFrame");
		r.method("setInitPosition",
		         (void(ILocalization::*)(const Point2f&, const Eigen::Matrix<float,2,2>&))
		         &ILocalization::setInitPosition, this,
		         "Initializes localization at the given position, keeping orientation unchanged. "
		         "Position is assumed to be in the reference frame.",
		         "position", "the position to set", Point2f(),
		         "cov", "x/y covariance", Eigen::DiagonalMatrix<float,2>(0.1, 0.1));
		r.method("setInitPosition",
		         (void(ILocalization::*)(const Point2f&, const Eigen::Matrix<float,2,2>&, const std::string&))
		         &ILocalization::setInitPosition, this,
		         "Initializes localization at the given position, keeping orientation unchanged, "
		         "using the given frame.",
		         "position", "the position to set", Point2f(),
		         "cov", "x/y covariance", Eigen::DiagonalMatrix<float,2>(0.1, 0.1),
		         "positionFrameID", "the frame that the position is defined in", "/GlobalFrame");
		r.method("setInitOrientation",
		         (void(ILocalization::*)(const Anglef&, float))&ILocalization::setInitOrientation, this,
		         "Initializes localization at the given orientation, keeping position unchanged. "
		         "Orientation is assumed to be in the reference frame.",
		         "orientation", "the orientation to set", Anglef(),
		         "cov", "phi covariance", 10.f);
		r.method("setInitOrientation",
		         (void(ILocalization::*)(const Anglef&, float, const std::string&))
		         &ILocalization::setInitOrientation, this,
		         "Initializes localization at the given orientation, keeping position unchanged, "
		         "using the given frame.",
		         "orientation", "the orientation to set", Anglef(),
		         "cov", "phi covariance", 10.f,
		         "orientationFrameID", "the frame that the orientation is defined in", "/GlobalFrame");
	}

	/// The destructor
	virtual ~ILocalization() {}

public:

	/**
	 * Initializes localization at the given pose.
	 * The frame of the pose is assumed to be the reference frame of the localization
	 * algorithm.
	 */
	virtual void setInitPose(const PoseCov2& pose) = 0;

	/**
	 * Initializes localization at the given pose.
	 * The pose will be transformed from the pose frame to the
	 * reference frame of the localization algorithm.
	 */
	virtual void setInitPoseFrame(const PoseCov2& pose, const std::string& poseFrameID) = 0;

	/**
	 * Initializes only position, keeping orientation unchanged.
	 * The frame of the position is assumed to be the reference frame of the localization
	 * algorithm.
	 */
	virtual void setInitPosition(const Point2f& position, const Eigen::Matrix<float,2,2>& cov) {
		MIRA_THROW(XNotImplemented, "Service does not implement setInitPosition(position, cov)."); }

	/**
	 * Initializes only position, keeping orientation unchanged.
	 * The position will be transformed from the position frame to the
	 * reference frame of the localization algorithm.
	 */
	virtual void setInitPosition(const Point2f& position, const Eigen::Matrix<float,2,2>& cov,
	                             const std::string& positionFrameID)  {
		MIRA_THROW(XNotImplemented, "Service does not implement setInitPosition(position, cov, positionFrameID)."); }

	/**
	 * Initializes only orientation, keeping position unchanged.
	 * The frame of the orientation is assumed to be the reference frame of the localization
	 * algorithm.
	 */
	virtual void setInitOrientation(const Anglef& orientation, float cov) {
		MIRA_THROW(XNotImplemented, "Service does not implement setInitPosition(orientation, cov)."); }

	/**
	 * Initializes only orientation, keeping position unchanged.
	 * The orientation will be transformed from the orientation frame to the
	 * reference frame of the localization algorithm.
	 */
	virtual void setInitOrientation(const Anglef& orientation, float cov,
	                                const std::string& orientationFrameID)	{
		MIRA_THROW(XNotImplemented, "Service does not implement setInitPosition(orientation, cov, orientationFrameID)."); }
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace

#endif
