###############################################################################
# Copyright (C) 2012 by
#   MetraLabs GmbH (MLAB), GERMANY
# and
#   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
# All rights reserved.
#
# Contact: info@mira-project.org
#
# Commercial Usage:
#   Licensees holding valid commercial licenses may use this file in
#   accordance with the commercial license agreement provided with the
#   software or, alternatively, in accordance with the terms contained in
#   a written agreement between you and MLAB or NICR.
#
# GNU General Public License Usage:
#   Alternatively, this file may be used under the terms of the GNU
#   General Public License version 3.0 as published by the Free Software
#   Foundation and appearing in the file LICENSE.GPL3 included in the
#   packaging of this file. Please review the following information to
#   ensure the GNU General Public License version 3.0 requirements will be
#   met: http://www.gnu.org/copyleft/gpl.html.
#   Alternatively you may (at your option) use any later version of the GNU
#   General Public License if such license has been publicly approved by
#   MLAB and NICR (or its successors, if any).
#
# IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
# INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
# THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
# "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
# BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
# FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
# ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
# PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
#
###############################################################################
#
# Provides MIRA_ADD_LIBRARY macro as alternative for the build-in
# ADD_LIBRARY macro. This macro perform the following steps:
#   1.) Add the library as a new target using ADD_LIBRARY
#   2.) Setting target properties VERSION, SOVERSION
#   3.) Define a symbol for the Windows DLL exports
#   4.) Handle Qt4 ui-Files, moc's, resources and translations
#   5.) Link the library using the build-it TARGET_LINK_LIBRARIES
#   6.) Install the library to lib
#   7.) Build a manifest, if the target is not MIRABase
#   8.) Install optional other files to somewhere
#   9.) Add documentation using MIRA_ADD_DOCUMENTATION
#  10.) Install the target and the manifest to the cmake/cpack component
#
# Note for Qt translations: 
#   The name of the ts-files must be specified by using QT_TRs. The files to
#   translate must be specified by using QT_TR_SOURCES. The build chain will
#   created a separate qrc-file for the qm-files and compile this qrc-file into
#   the executable. To avoid conflicts with other resources, an optional prefix
#   for the translation can be specified by using QT_TR_PREFIX.
#
# Usage:
#     MIRA_ADD_LIBRARY(TargetName
#                     [STATIC | SHARED | MODULE]
#                     [PACKAGE        <package_name>
#                     [VERSION        <library-version>]     <obsolete!>
#                     [APIVERSION     <api-version>]         <obsolete!>
#                     [COMPONENT      <component>]           <obsolete!>
#                     [EXPORT_SYMBOL  <export-symbol>]
#                     [SOURCE         <source1 ... sourceN>]
#                     [PCH            <precompiled header>]
#                     [QT_UIs         <ui1 ... uiN>]
#                     [QT_MOCs        <moc1 ... mocN>]
#                     [QT_QRCs        <qrc1 ... qrcN>]
#                     [QT_TRs         <file1 ... fileN>]
#                     [QT_TR_SOURCES  <file1 ... fileN>]
#                     [QT_TR_INCLUDES <dir1 ... dirN>]
#                     [QT_TR_PREFIX   <prefix>]
#                     [LINK_LIBS      <lib1 ... libN>]
#                     [INSTALL_PATH   <path>]
#                     [INSTALL_FILES  <src1 destDir1 ... srcN destDirN>]
#                     [DOC_DIRS       <dir1 ... dirN>]
#                     [DOC_DEPENDS    <dep1 ... depN>]
#                     [MANIFEST_IGNORE_CLASSES <class1 ... classN>]
#                     [DONT_INSTALL]
#                    )
#
# Author: Christian Martin
#
###############################################################################

include(CMakeParseArguments)
include(Manifest)

if (WIN32)
	set(LIBRARY_DEST_FOLDER bin)
else(WIN32) # LINUX
	set(LIBRARY_DEST_FOLDER lib)
endif(WIN32)

###############################################################################

macro(MIRA_ADD_LIBRARY target)

	###########################################################################
	# Parse the macro arguments

	CMAKE_PARSE_ARGUMENTS(arg
		# options:
		"STATIC;SHARED;MODULE;DONT_INSTALL"
		# one-value arguments:
		"PACKAGE;VERSION;APIVERSION;COMPONENT;EXPORT_SYMBOL;INSTALL_PATH;PCH;QT_TR_PREFIX"
		# multi-value arguments:
		"SOURCE;QT_UIs;QT_MOCs;QT_QRCs;QT_TRs;QT_TR_SOURCES;QT_TR_INCLUDES;LINK_LIBS;INSTALL_FILES;DOC_DIRS;DOC_DEPENDS;MANIFEST_IGNORE_CLASSES"
		${ARGN})

	###########################################################################
	# Check the macro arguments

	set(targetType "")
	set(targetTypeCnt 0)

	if (arg_STATIC)
		set(targetType "STATIC")
		math(EXPR targetTypeCnt ${targetTypeCnt}+1)
	endif()
	if (arg_SHARED)
		set(targetType "SHARED")
		math(EXPR targetTypeCnt ${targetTypeCnt}+1)
	endif()
	if (arg_MODULE)
		set(targetType "MODULE")
		math(EXPR targetTypeCnt ${targetTypeCnt}+1)
	endif()

	# Ensure, that exactly one target type is specified
	if (NOT ${targetTypeCnt} EQUAL 1)
		message(FATAL_ERROR "Target ${target}: "
		        "You need to specify exactly one of the options STATIC | SHARED | MODULE.")
	endif()

	# Get version and component information from PACKAGE parameter.
	if (arg_PACKAGE)
		# If PACKAGE parameter exists, get version information from package and
		# ignore VERSION, APIVERSION and COMPONENT.
		if (arg_VERSION OR arg_APIVERSION OR arg_COMPONENT)
			message(FATAL_ERROR "Target ${target}: You should not mix PACKAGE and VERSION, APIVERSION or COMPONENT.")
		endif()

		set(arg_VERSION    ${${arg_PACKAGE}_VERSION})
		set(arg_APIVERSION ${${arg_PACKAGE}_APIVERSION})
		set(arg_COMPONENT  ${arg_PACKAGE})

		# Store the name of the package, to which this target belongs, as a
		# (CACHE INTERNAL) variable named ${target}_PACKAGE.
		set(${target}_PACKAGE ${arg_PACKAGE} CACHE INTERNAL "" FORCE)

		# Append the current target to the global (CACHE INTERNAL) variable
		# ${package}_TARGETS.
		list(FIND ${arg_PACKAGE}_TARGETS ${target} found)
		if(${found} EQUAL -1)
			list(APPEND ${arg_PACKAGE}_TARGETS ${target})
			set(${arg_PACKAGE}_TARGETS "${${arg_PACKAGE}_TARGETS}" CACHE INTERNAL "" FORCE)
		endif()

	else()
		if (arg_VERSION OR arg_APIVERSION OR arg_COMPONENT)
			message("-- WARNING - Target ${target}: "
			        "The arguments VERSION, APIVERSION and COMPONENT are obsolete and will be removed in a future release. Please use the PACKAGE parameter.")
		endif()
	endif()

	# Ensure, that either both VERSION and APIVERSION are given or none.
	if ( NOT((arg_VERSION AND arg_APIVERSION) OR (NOT arg_VERSION AND NOT arg_APIVERSION)) )
		message(FATAL_ERROR "Target ${target}: "
		        "Either only APIVERSION or VERSION is defined! Always use both or none.")
	endif()

	# Ensure, that a export symbol is given
	if (NOT arg_EXPORT_SYMBOL)
		string(TOUPPER "${target}" capitalTarget)
		set(arg_EXPORT_SYMBOL "MIRA_${capitalTarget}_EXPORTS")
		#message("-- WARNING - Target ${target}: "
		#        "No export symbol given. Using default: ${arg_EXPORT_SYMBOL}")
	endif()

	# Ensure, that we have at least one source file
	if (NOT arg_SOURCE)
		message(FATAL_ERROR "Target ${target} has no source files.")
	endif()

	# Ensure, that INSTALL_FILES has a even number of elements
	set(installFilesLen 0)
	list(LENGTH arg_INSTALL_FILES installFilesLen)
	set(installFilesModulo 0)
	math(EXPR installFilesModulo ${installFilesLen}%2)
	if (NOT installFilesModulo EQUAL 0)
		message(FATAL_ERROR "Target ${target} has odd number of elements in INSTALL_FILES")
	endif()

	# Test if the obsolete INSTALL_PATH is given
	if (arg_INSTALL_PATH)
		message("-- WARNING - Target ${target}: "
		        "Obsolete option INSTALL_PATH is defined in MIRA_ADD_LIBRARY. "
		        "This option will be removed soon!")
	endif()

	###########################################################################

	# Qt4 ui-Files
	set(QT_UI_HDRS "")
	if (arg_QT_UIs)
		QT_WRAP_UI(QT_UI_HDRS ${arg_QT_UIs})
		include_directories(${CMAKE_CURRENT_BINARY_DIR})
	endif()

	# Qt moc
	set(QT_MOC_SRC "")
	if (arg_QT_MOCs)
		# Avoid problems related to the fact that moc do not expand macro properly
		# by defining BOOST_TT_HAS_OPERATOR_HPP_INCLUDED. For some more details
		# please read here: https://bugreports.qt-project.org/browse/QTBUG-22829
		QT_WRAP_CPP(QT_MOC_SRC ${arg_QT_MOCs} OPTIONS -DBOOST_TT_HAS_OPERATOR_HPP_INCLUDED)
	endif()

	# Qt resources
	set(QT_RCC_SRCS "")
	if (arg_QT_QRCs)
		QT_ADD_RESOURCES(QT_RCC_SRCS ${arg_QT_QRCs})
	endif()

	# Qt translations
	set(QT_QM_FILES "")
	if (arg_QT_TRs)
		if (NOT arg_QT_TR_SOURCES)
			message(FATAL_ERROR "Target ${target}: Missing arguments for QT_TR_SOURCES. Need at least one file to translate.")
		endif()
		MIRA_QT_CREATE_TRANSLATION(QT_QM_FILES QT_QM_QRC_FILE ${target}
				TRANSLATIONS ${arg_QT_TRs}
				FILES        ${arg_QT_TR_SOURCES}
				INCLUDES     ${arg_QT_TR_INCLUDES}
				OPTIONS      -locations none
				)
		# add the generated qrc file for the translations
		QT_ADD_RESOURCES(QT_RCC_SRCS ${QT_QM_QRC_FILE})
	endif()

	###########################################################################
	# compile, link and install

	# optionally generate a source file for the precompiled header
	if (arg_PCH)
		MIRA_PREPARE_PRECOMPILED_HEADER(${target} ${arg_PCH} arg_SOURCE)
	endif()

	list(APPEND arg_SOURCE ${MIRA_ROOT_DIR}/base/src/factory/RegisterLibraryConstructor.C)

	# add the target
	add_library(${target} ${targetType} ${arg_SOURCE} ${QT_UI_HDRS} ${QT_MOC_SRC} ${QT_RCC_SRCS})

	get_target_property(COMPILE_FLAGS ${target} COMPILE_FLAGS)		
	if (${COMPILE_FLAGS} STREQUAL "COMPILE_FLAGS-NOTFOUND")
		set(COMPILE_FLAGS "")
	endif()

	# define PACKAGE and PACKAGE_VERSION identifiers to be used in code
	if (arg_PACKAGE)
		set(COMPILE_FLAGS "${COMPILE_FLAGS} -DPACKAGE=${arg_PACKAGE}")
		if ((DEFINED ${arg_PACKAGE}_MAJOR) AND (DEFINED ${arg_PACKAGE}_MINOR) AND (DEFINED ${arg_PACKAGE}_PATCH))
			set(COMPILE_FLAGS "${COMPILE_FLAGS} -DPACKAGE_VERSION_MAJOR=${${arg_PACKAGE}_MAJOR}")
			set(COMPILE_FLAGS "${COMPILE_FLAGS} -DPACKAGE_VERSION_MINOR=${${arg_PACKAGE}_MINOR}")
			set(COMPILE_FLAGS "${COMPILE_FLAGS} -DPACKAGE_VERSION_PATCH=${${arg_PACKAGE}_PATCH}")
			math(EXPR VERSION_NUMBER "(${${arg_PACKAGE}_MAJOR}*1000+${${arg_PACKAGE}_MINOR})*100+${${arg_PACKAGE}_PATCH}")
			set(COMPILE_FLAGS "${COMPILE_FLAGS} -DPACKAGE_VERSION=${VERSION_NUMBER}")
		else()
			message("-- WARNING - Package ${arg_PACKAGE}: Version unknown or incomplete. Please use MIRA_VERSION().")
		endif()
	else()
		message("-- WARNING - Target ${target}: Cannot determine the package this target belongs to. Please set the PACKAGE parameter.")
	endif()
	set(COMPILE_FLAGS "${COMPILE_FLAGS} -DLIBRARY_TARGET_NAME=${target}")

	# set some target properties
	if (NOT arg_VERSION AND NOT arg_APIVERSION)
		set_target_properties(${target} PROPERTIES
 			DEFINE_SYMBOL ${arg_EXPORT_SYMBOL}
 			COMPILE_FLAGS "${COMPILE_FLAGS}"
		)
	else()
		set_target_properties(${target} PROPERTIES
			VERSION       ${arg_VERSION}
			SOVERSION     ${arg_APIVERSION}
			DEFINE_SYMBOL ${arg_EXPORT_SYMBOL}
			COMPILE_FLAGS "${COMPILE_FLAGS}"
		)
	endif()

	if (MIRAAutoLinkLibraries)
		list(APPEND arg_LINK_LIBS ${MIRAAutoLinkLibraries})

		# Store the MIRAAutoLinkLibraries for each target as a global
		# (CACHE INTERNAL) variable named ${target}_AutoLinkLibs.
		set(${target}_AUTOLINKLIBS ${MIRAAutoLinkLibraries} CACHE INTERNAL "" FORCE)
	endif()

	# link the library
	target_link_libraries(${target} ${arg_LINK_LIBS})

	if (arg_PACKAGE)
		# Add library to package target
		# check if target already exists
		if( NOT TARGET PACKAGE_${arg_PACKAGE} )
			add_custom_target( PACKAGE_${arg_PACKAGE} )
		endif()
		# add dependency for package target
		add_dependencies(PACKAGE_${arg_PACKAGE} ${target})
		# set label property (for CDash)
		SET_PROPERTY(TARGET ${target} PROPERTY LABELS ${arg_PACKAGE})
	endif()

	# Install the library to lib/bin directory. This must be done, before
	# the manifest will be generated. Otherwise, the lib directory might
	# contain an outdated version of the target.
	if (NOT arg_DONT_INSTALL)
		MIRA_DIST_LIBRARY(${target})
	endif()

	# add a manifest, if the target is not MIRABase
	string(COMPARE EQUAL "${target}" "MIRABase" isBaseLib)
	if (NOT isBaseLib)
		# build list of arguments for ADD_MANIFEST
		set(addManifestArgs)
		list(APPEND addManifestArgs ${target})
		if (NOT arg_DONT_INSTALL)
			list(APPEND addManifestArgs INSTALL_DIR "${CMAKE_SOURCE_DIR}/${LIBRARY_DEST_FOLDER}")
		endif()
		if (arg_MANIFEST_IGNORE_CLASSES)
			list(APPEND addManifestArgs IGNORE_CLASSES ${arg_MANIFEST_IGNORE_CLASSES})
		endif()
		list(APPEND addManifestArgs LIBRARY_VERSION ${${arg_PACKAGE}_MAJOR} ${${arg_PACKAGE}_MINOR} ${${arg_PACKAGE}_PATCH})

		ADD_MANIFEST(${addManifestArgs})
	endif()

	# install all optional files
	set(installFilesIdx 0)
	while(installFilesIdx LESS installFilesLen)
		set(fileSrc "")
		set(fileDest "")
		list(GET arg_INSTALL_FILES ${installFilesIdx} fileSrc)
		math(EXPR installFilesIdx ${installFilesIdx}+1)
		list(GET arg_INSTALL_FILES ${installFilesIdx}+1 fileDest)
		math(EXPR installFilesIdx ${installFilesIdx}+1)

		MIRA_DIST_FILE(${target} ${fileSrc} ${fileDest})
	endwhile()

	if (arg_PCH)
		MIRA_ADD_PRECOMPILED_HEADER(${target} ${arg_PCH})
	endif() 

	###########################################################################
	# documentation

	if(arg_DOC_DIRS)
		MIRA_ADD_DOCUMENTATION(${target}
			DIRS ${arg_DOC_DIRS}
			DEPENDS ${arg_DOC_DEPENDS}
		)
	endif(arg_DOC_DIRS)

	###########################################################################
	# Install lib and manifest to cpack component

	if(arg_COMPONENT AND (NOT arg_DONT_INSTALL) )
		# Install the target itself
		#
		# (Note: We must disable the "ARCHIVE DESTINATION" here. Otherwise
		#  the generated cmake_install.cmake file will use an "Unspecified"
		#  component. If "ARCHIVE DESTINATION" is disabled, the right
		#  component is use. Maybe, this is a bug in cmake.)
		#
		if (WIN32)
			mira_install(TARGETS ${target}
				RUNTIME DESTINATION ${LIBRARY_DEST_FOLDER}
				LIBRARY DESTINATION lib
				ARCHIVE DESTINATION lib
				PACKAGE ${arg_COMPONENT})
		else(WIN32) # LINUX
			mira_install(TARGETS ${target}
				RUNTIME DESTINATION ${LIBRARY_DEST_FOLDER}
				LIBRARY DESTINATION lib
				PACKAGE ${arg_COMPONENT})
		endif(WIN32)

		IF (NOT isBaseLib AND NOT CMAKE_CROSSCOMPILING)
			# Install the manifest
			INSTALL_MANIFEST(${target} ${LIBRARY_DEST_FOLDER} ${arg_COMPONENT})
		endif()
	endif()

ENDMACRO(MIRA_ADD_LIBRARY)
