/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file QtUtils.C
 *    Implementation of QtUtils.h
 *
 * @author Tim Langner
 * @date   2011/02/06
 */

#include <widgets/QtUtils.h>

#include <QImage>
#include <QFileDialog>
#include <QGraphicsItem>
#include <QMessageBox>
#include <QPainter>

#include <math/Saturate.h>

namespace mira { namespace QtUtils {

///////////////////////////////////////////////////////////////////////////////

QImage toQImage(const Img<>& img, double s_min, double s_max)
{
	if(img.isEmpty())
		return QImage();

	const size_t widthStep = img.step();
	const int width  = img.width();
	const int height = img.height();

	const int imax = (s_max >= 1.0f ? static_cast<int>(ceil(s_max)) : 65535); // default for 16 bit, 2^16
	const int imin = static_cast<int>(floor(s_min));

	// Precalculate the divisors for the scaling below
	const int imax_min = (imax - imin);
	const double smax_min = (s_max - s_min);

	QImage qimage;
	bool failed = false;
	switch (img.depth())
	{
	case CV_8U:
		if (img.channels() == 1) {
			if((width % 4) == 0) {
				// if width is multiple of 4, use faster variant
				qimage = QImage(img.data(), width, height, QImage::Format_Indexed8);
			} else {
				qimage = QImage(width, height, QImage::Format_Indexed8);

				for (int y = 0; y < height; y++)
				{
					uchar* qimagePtr      = qimage.scanLine(y);
					const uint8* imagePtr = img.data(y);
					memcpy(qimagePtr, imagePtr, width);
				}
			}
		} else if (img.channels() == 3) {
			// OpenCV image is stored with 3 byte color pixels (3 channels).
			// We convert it to a 32 bit depth QImage.
			qimage = QImage(width, height, QImage::Format_RGB32);

			for (int y = 0; y < height; y++) {
				uchar* qimagePtr      = qimage.scanLine(y);
				const uint8* imagePtr = img.data(y);
				for (int x = 0; x < width; x++) {
					qimagePtr[0] = imagePtr[0];
					qimagePtr[1] = imagePtr[1];
					qimagePtr[2] = imagePtr[2];
					qimagePtr[3] = 0;
					qimagePtr += 4;
					imagePtr  += 3;
				}
			}
		} else if(img.channels() == 4) {
			if((width % 4) == 0) {
				// if width is multiple of 4, use faster variant
				qimage = QImage(img.data(), width, height, QImage::Format_RGB32);
			}
			else {
				qimage = QImage(width, height, QImage::Format_RGB32);
				for (int y = 0; y < height; y++) {
					uchar* qimagePtr      = qimage.scanLine(y);
					const uint8* imagePtr = img.data(y);
					memcpy(qimagePtr, imagePtr, width*4);
				}
			}
		} else {
			failed = true;
		}
		break;

	case CV_8S:
		if (img.channels() == 1) {
			qimage = QImage(width, height, QImage::Format_Indexed8);

			for (int y = 0; y < height; ++y) {

				uchar *qip = qimage.scanLine(y);
				const int8* cvip = reinterpret_cast<const int8*>(img.data(y));

				for (int x = 0; x < width; ++x) {
					*qip++ = static_cast<uchar>(saturate<int>(*cvip++ + 127, 0, 255));
				}
				cvip += (widthStep / sizeof(uint16_t)) - width;
			}
		} else {
			failed = true;
		}
		break;

	case CV_16U:
		if (img.channels() == 1) {
			qimage = QImage(width, height, QImage::Format_Indexed8);

			for (int y = 0; y < height; ++y) {

				uchar *qip = qimage.scanLine(y);
				const uint16_t* cvip = reinterpret_cast<const uint16_t*>(img.data(y));

				for (int x = 0; x < width; ++x) {
					*qip++ = static_cast<uchar>(saturate<int>((*cvip++ - imin) * 255 / imax_min, 0, 255));
				}
				cvip += (widthStep / sizeof(uint16_t)) - width;
			}
		} else {
			failed = true;
		}
		break;

	case CV_16S:
		if (img.channels() == 1) {
			qimage = QImage(width, height, QImage::Format_Indexed8);

			uchar *qip = qimage.scanLine(0);
			const int* cvip = reinterpret_cast<const int*>(img.data());

			for (int y = 0; y < height; ++y) {
				for (int x = 0; x < width; ++x) {
					*qip++ = static_cast<uchar>(saturate<int>((*cvip++ - imin) * 255 / imax_min, 0, 255));
				}
				cvip += (widthStep / sizeof(int)) - width;
			}
		} else {
			failed = true;
		}
		break;

	case CV_32F:
		if (img.channels() == 1) {
			qimage = QImage(width, height, QImage::Format_Indexed8);

			for (int y = 0; y < height; ++y) {

				uchar *qip = qimage.scanLine(y);
				const float* cvip = reinterpret_cast<const float*>(img.data(y));

				for (int x = 0; x < width; ++x) {
					*qip++ = static_cast<uchar>(saturate<float>((*cvip++ - static_cast<float>(s_min)) * 255.0f / static_cast<float>(smax_min), 0.0f, 255.0f));
				}
				cvip += (widthStep / sizeof(float)) - width;
			}
		} else {
			failed = true;
		}
		break;

	case CV_64F:
		if (img.channels() == 1) {
			qimage = QImage(width, height, QImage::Format_Indexed8);

			for (int y = 0; y < height; ++y) {

				uchar *qip = qimage.scanLine(y);
				const double* cvip = reinterpret_cast<const double*>(img.data(y));

				for (int x = 0; x < width; ++x) {
					*qip++ = static_cast<uchar>(saturate<double>((*cvip++ - s_min) * 255.0 / smax_min, 0.0, 255.0));
				}
				cvip += (widthStep / sizeof(double)) - width;
			}
		} else {
			failed = true;
		}
		break;
	}

	if(!failed && img.channels() == 1) {
		QVector<QRgb> colorTable;
		for (int i = 0; i < 256; i++) {
			colorTable.push_back(qRgb(i, i, i));
		}
		qimage.setColorTable(colorTable);
	}

	return qimage;
}

void fromQImage(const QImage& qimage, Img<>& img)
{
	size_t widthStep = img.step();
	int width  = qimage.width();
	int height = qimage.height();

	switch (qimage.format())
	{
	case QImage::Format_Indexed8:
	{
		img = Img8U1(width, height);
		for (int y = 0; y < height; y++)
		{
			const uchar* qimagePtr = qimage.scanLine(y);
			uint8* imagePtr        = img.data(y);
			memcpy(imagePtr, qimagePtr, width);
		}
		break;
	}
	case QImage::Format_RGB32:
	{
		img = Img8U3(width, height);
		for (int y = 0; y < height; y++)
		{
			const uchar* qimagePtr = qimage.scanLine(y);
			uint8* imagePtr        = img.data(y);
			for (int x = 0; x < width; x++)
			{
				imagePtr[0] = qimagePtr[0];
				imagePtr[1] = qimagePtr[1];
				imagePtr[2] = qimagePtr[2];
				qimagePtr += 4;
				imagePtr  += 3;
			}
		}
		break;
	}
	default:
		MIRA_THROW(XNotImplemented, "Conversion from QImage format "
			<< qimage.format() << " to Img<> not supported.");
		break;
	}
}

void setTransform(QGraphicsItem* item, const RigidTransform2f& t)
{
	item->setPos(t.x(), t.y());
#if (QT_VERSION < QT_VERSION_CHECK(4, 6, 1))
	//item->setTransform(QTransform(cos(t.phi()),-sin(t.phi()),sin(t.phi()),cos(t.phi()),0.0,0.0));
	// for the skeptic (maybe the sign is wrong)
	item->setTransform(QTransform().rotate(rad2deg(t.phi())));
#else
	item->setRotation(rad2deg(t.phi()));
#endif
}

QString getOpenFileName(QWidget* parent,
                        const QString& caption,
                        const QString& dir,
                        const QString& filter,
                        QString* selectedFilter,
                        QFileDialog::Options options) {
	return QFileDialog::getOpenFileName(parent, caption, dir, filter, selectedFilter,
	                                    options | QFileDialog::DontUseNativeDialog);
}

QString getSaveFileName(QWidget* parent,
                        const QString& caption,
                        const QString& dir,
                        const QString& filter,
                        QString* selectedFilter,
                        QFileDialog::Options options,
                        const QStringList& enforceExtension)
{
	QString currentDir = dir;
	while (true) {
		QString filename =  QFileDialog::getSaveFileName(parent, caption, currentDir, filter, selectedFilter,
			                                         options | QFileDialog::DontUseNativeDialog);

		if (enforceExtension.isEmpty() || filename.isEmpty())
			return filename;

		const QFileInfo info(filename);
		const QString suffix(info.suffix());

		if (enforceExtension.contains(suffix, Qt::CaseInsensitive))
			return filename;

		// we reached here as a filename was selected but it does not match the required extension
		// we add the extension, have to check for file overwrite ourselves now

		filename.append("."+enforceExtension.first());

		const QFileInfo infoExt(filename);
		if (!infoExt.exists())
			return filename;

		if (QMessageBox::warning(parent, "Save as CSV",
		                         infoExt.fileName() + " already exists.\nDo you want to replace it?",
		                         QMessageBox::No | QMessageBox::Yes) == QMessageBox::Yes) {
			return filename;
		}

		// will start dialog at last selected directory for making a new choice
		currentDir = infoExt.path();
	}
}

QString getExistingDirectory(QWidget* parent,
                             const QString& caption,
                             const QString& dir,
                             QFileDialog::Options options)
{
	return QFileDialog::getExistingDirectory(parent, caption, dir,
	                                         options | QFileDialog::DontUseNativeDialog);
}

QString tr(const char* sourceText, const char* disambiguation, int n)
{
// - in Qt5, tr() and trUtf8() are equivalent
// - in Qt5.15, trUtf8() is deprecated
#if QT_VERSION > QT_VERSION_CHECK(5, 12, 0)
    return QObject::tr(sourceText, disambiguation, n);
#else
    return QObject::trUtf8(sourceText, disambiguation, n);
#endif
}

void initPainterFrom(QPainter& painter, const QWidget& widget)
{
	// taken from implementation of QWidget::initPainter(QPainter*)
	// which is doing the work for QPainter::initFrom(QPaintDevice*),
	// but is not public, and probably also deprecated

//	void QWidget::initPainter(QPainter *painter) const
//	{
//		const QPalette &pal = palette();
//		painter->d_func()->state->pen = QPen(pal.brush(foregroundRole()), 1);
//		painter->d_func()->state->bgBrush = pal.brush(backgroundRole());
//		QFont f(font(), const_cast<QWidget *>(this));
//		painter->d_func()->state->deviceFont = f;
//		painter->d_func()->state->font = f;
//	}

	const QPalette& pal = widget.palette();
	painter.setPen(QPen(pal.brush(widget.foregroundRole()), 1));
	painter.setBackground(pal.brush(widget.backgroundRole()));
	painter.setFont(widget.font());
}

///////////////////////////////////////////////////////////////////////////////

}}
