/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file OgreUtils.C
 *    Implementation of OgreUtils.h.
 *
 * @author Erik Einhorn
 * @date   2010/12/19
 */

#include <widgets/OgreUtils.h>

#include <OGRE/OgreTechnique.h>
#include <OGRE/OgreMaterialManager.h>
#include <OGRE/OgreResourceGroupManager.h>
#include <OGRE/OgreManualObject.h>
#include <OGRE/OgreSceneManager.h>
#include <OGRE/OgreSceneNode.h>
#include <OGRE/OgreEntity.h>
#include <OGRE/OgreSubEntity.h>
#include <OGRE/OgreManualObject.h>
#include <OGRE/OgreSubMesh.h>

namespace mira { namespace OgreUtils {

///////////////////////////////////////////////////////////////////////////////

void loadResource(const Path& path, const Ogre::String& resourceGroup)
{
	Path resolved = resolvePath(path);
	Ogre::ResourceGroupManager& mgr = Ogre::ResourceGroupManager::getSingleton();

	if (!mgr.resourceGroupExists(resourceGroup))
		mgr.createResourceGroup(resourceGroup);

	Ogre::String directory = resolved.string();
	mgr.addResourceLocation(directory, "FileSystem", resourceGroup, false);
	mgr.initialiseResourceGroup(resourceGroup);
	mgr.loadResourceGroup(resourceGroup);
}

void setTransform(Ogre::SceneNode* node, const RigidTransform2f& t)
{
	setPosition(node,t.t);
	setOrientation(node,t.r);
}

void setTransform(Ogre::SceneNode* node, const RigidTransform3f& t)
{
	setPosition(node,t.t);
	setOrientation(node,t.r);
}

void setPosition(Ogre::SceneNode* node, const Eigen::Vector2f& t)
{
	if(boost::math::isnan(t.x()) || boost::math::isnan(t.y()))
		MIRA_THROW(XInvalidParameter, "Specified translation contains NaN values");

	node->setPosition(t.x(), t.y(), 0.0f);
}

void setPosition(Ogre::SceneNode* node, const Eigen::Vector3f& t)
{
	if(boost::math::isnan(t.x()) || boost::math::isnan(t.y()) || boost::math::isnan(t.z()))
		MIRA_THROW(XInvalidParameter, "Specified translation contains NaN values");

	node->setPosition(t.x(), t.y(), t.z());
}

void setOrientation(Ogre::SceneNode* node, const Eigen::Rotation2D<float>& r)
{
	if(boost::math::isnan(r.angle()))
		MIRA_THROW(XInvalidParameter, "Specified rotation contains NaN values");

	Ogre::Quaternion q;
	q.FromAngleAxis(Ogre::Radian(r.angle()), Ogre::Vector3::UNIT_Z);
	node->setOrientation(q);
}

void setOrientation(Ogre::SceneNode* node, const Eigen::Quaternion<float>& r)
{
	if(boost::math::isnan(r.w()) || boost::math::isnan(r.x()) ||
	   boost::math::isnan(r.y()) || boost::math::isnan(r.z()))
		MIRA_THROW(XInvalidParameter, "Specified rotation contains NaN values");

	Ogre::Quaternion q(r.w(), r.x(), r.y(), r.z());
	node->setOrientation(q);
}

bool hasAttachedObject(Ogre::SceneNode* node, Ogre::MovableObject* object)
{
	// first check own attached objects
	Ogre::SceneNode::ObjectIterator oi = node->getAttachedObjectIterator();
	for (auto i=oi.begin(); i!=oi.end(); ++i)
	{
		if (i->second == object)
			return true;
	}
	// no direct object -> check child nodes
	Ogre::Node::ChildNodeIterator ci = node->getChildIterator();
	for (auto i=ci.begin(); i!=ci.end(); ++i)
	{
		Ogre::SceneNode* n = dynamic_cast<Ogre::SceneNode*>(i->second);
		if (n != NULL && hasAttachedObject(n, object))
			return true;
	}
	return false;
}

///////////////////////////////////////////////////////////////////////////////


void rayQuery(Ogre::SceneNode* node, const Ogre::Ray& ray, Ogre::RaySceneQueryResult& oResult, uint32 queryMask)
{
	// attached objects
	Ogre::SceneNode::ObjectIterator objectIt = node->getAttachedObjectIterator();
	for (auto it=objectIt.begin(); it!=objectIt.end(); ++it)
	{
		Ogre::MovableObject* a = it->second;
		//std::cout << "Object: " << a << ", " << a->getMovableType() << ", " << a->getQueryFlags() <<  std::endl;

		if((a->getQueryFlags() & queryMask) &&
		    a->isInScene())
		{
			// Do ray / box test
			std::pair<bool, float> r = ray.intersects(a->getWorldBoundingBox());
			if(r.first) {
				Ogre::RaySceneQueryResultEntry dets;
				dets.distance = r.second;
				dets.movable = a;
				dets.worldFragment = NULL;
				oResult.push_back(dets);
			}
		}
	}

	// recurse through child nodes
	Ogre::Node::ChildNodeIterator nodeIt = node->getChildIterator();
	for (auto it=nodeIt.begin(); it!=nodeIt.end(); ++it)
	{
		Ogre::SceneNode* n = dynamic_cast<Ogre::SceneNode*>(it->second);
		if (n != NULL)
			rayQuery(n,ray,oResult,queryMask);
	}
}

// mesh collision code taken and adapted from:
// http://www.ogre3d.org/tikiwiki/tiki-index.php?page=Raycasting+to+the+polygon+level

static void getMeshInformation( const Ogre::ManualObject* manual, size_t& vertex_count, Ogre::Vector3*& vertices, size_t& index_count, unsigned long*& indices, const Ogre::Vector3& position, const Ogre::Quaternion& orient, const Ogre::Vector3& scale );
static void getMeshInformation( const Ogre::Entity* entity, size_t& vertex_count, Ogre::Vector3*& vertices, size_t& index_count, unsigned long*& indices, const Ogre::Vector3& position, const Ogre::Quaternion& orient, const Ogre::Vector3& scale );

std::pair<bool,float> rayObjectCollision(const Ogre::Ray& ray, const Ogre::MovableObject* object)
{
	// first check intersection with bounding box
	std::pair<bool, float> bbintersection = ray.intersects(object->getWorldBoundingBox());
	if(!bbintersection.first) // bail out if bounding box is not hit
		return std::make_pair(false,0.0f);


	const std::string& movableType = object->getMovableType();

	// mesh data to retrieve
	size_t vertex_count;
	size_t index_count;
	Ogre::Vector3* vertices;
	unsigned long* indices;

	if( movableType == "ManualObject" ) {
		// get the entity to check
		const Ogre::ManualObject* pentity = static_cast<const Ogre::ManualObject*>(object);
		// get the mesh information
		getMeshInformation( pentity, vertex_count, vertices, index_count, indices,
					   pentity->getParentNode()->_getDerivedPosition(),
					   pentity->getParentNode()->_getDerivedOrientation(),
					   pentity->getParentNode()->_getDerivedScale()
					   );
	} else if( movableType == "Entity" ) {
		// get the entity to check
		const Ogre::Entity *pentity = static_cast<const Ogre::Entity*>(object);
		// get the mesh information
		getMeshInformation( pentity, vertex_count, vertices, index_count, indices,
					   pentity->getParentNode()->_getDerivedPosition(),
					   pentity->getParentNode()->_getDerivedOrientation(),
					   pentity->getParentNode()->_getDerivedScale()
					   );
	}else{
		// for all other object types we can perform the bounding box check only
		// which we have already passed above
		return std::make_pair(true,bbintersection.second);
	}

	// test for hitting individual triangles on the mesh
	Ogre::Real closest_distance = std::numeric_limits<Ogre::Real>::max();
	bool found_hit=false;
	for ( int i=0; i<static_cast<int>(index_count); i += 3)
	{
		// check for a hit against this triangle
		std::pair<bool, Ogre::Real> hit = Ogre::Math::intersects(ray, vertices[indices[i]], vertices[indices[i+1]], vertices[indices[i+2]], true, false);

		// if it was a hit check if its the closest
		if( hit.first && hit.second < closest_distance)
		{
			// this is the closest so far, save it off
			closest_distance = hit.second;
			found_hit = true;
		}
	}

	// free the verticies and indicies memory
	delete[] vertices;
	delete[] indices;

	return std::make_pair(found_hit,closest_distance);
}


static void getMeshInformation( const Ogre::Entity* entity, size_t& vertex_count, Ogre::Vector3*& vertices, size_t& index_count, unsigned long*& indices, const Ogre::Vector3& position, const Ogre::Quaternion& orient, const Ogre::Vector3& scale )
{
	bool added_shared = false;
	size_t current_offset = 0;
	size_t shared_offset = 0;
	size_t next_offset = 0;
	size_t index_offset = 0;
	vertex_count = index_count = 0;

	Ogre::MeshPtr mesh = entity->getMesh();


	bool useSoftwareBlendingVertices = entity->hasSkeleton();

	if( useSoftwareBlendingVertices )
		const_cast<Ogre::Entity*>(entity)->_updateAnimation();

	// Calculate how many vertices and indices we're going to need
	for( unsigned short i=0,size=mesh->getNumSubMeshes(); i<size; ++i )
	{
		Ogre::SubMesh* submesh = mesh->getSubMesh(i);

		// We only need to add the shared vertices once
		if( submesh->useSharedVertices ){
			if( !added_shared ){
				vertex_count += mesh->sharedVertexData->vertexCount;
				added_shared = true;
			}
		}else{
			vertex_count += submesh->vertexData->vertexCount;
		}

		// Add the indices
		index_count += submesh->indexData->indexCount;
	}


	// Allocate space for the vertices and indices
	vertices = new Ogre::Vector3[vertex_count];
	indices = new unsigned long[index_count];

	added_shared = false;

	// Run through the submeshes again, adding the data into the arrays
	for( unsigned short i=0, size=mesh->getNumSubMeshes(); i<size; ++i)
	{
		Ogre::SubMesh* submesh = mesh->getSubMesh(i);

		//----------------------------------------------------------------
		// GET VERTEXDATA
		//----------------------------------------------------------------

		//Ogre::VertexData* vertex_data = submesh->useSharedVertices ? mesh->sharedVertexData : submesh->vertexData;
		Ogre::VertexData* vertex_data;

		//When there is animation:
		if( useSoftwareBlendingVertices )
			vertex_data = submesh->useSharedVertices ? entity->_getSkelAnimVertexData() : entity->getSubEntity(i)->_getSkelAnimVertexData();
		else
			vertex_data = submesh->useSharedVertices ? mesh->sharedVertexData : submesh->vertexData;


		if( (!submesh->useSharedVertices) || !added_shared )
		{
			if( submesh->useSharedVertices ){
				added_shared = true;
				shared_offset = current_offset;
			}

			const Ogre::VertexElement* posElem = vertex_data->vertexDeclaration->findElementBySemantic(Ogre::VES_POSITION);

			Ogre::HardwareVertexBufferSharedPtr vbuf = vertex_data->vertexBufferBinding->getBuffer(posElem->getSource());

			unsigned char* vertex = static_cast<unsigned char*>(vbuf->lock(Ogre::HardwareBuffer::HBL_READ_ONLY));

			// There is _no_ baseVertexPointerToElement() which takes an Ogre::Real or a double
			//  as second argument. So make it float, to avoid trouble when Ogre::Real will
			//  be comiled/typedefed as double:
			//      Ogre::Real* pReal;
			float* pReal = 0;

			for( size_t j=0; j<vertex_data->vertexCount; ++j, vertex += vbuf->getVertexSize())
			{
				posElem->baseVertexPointerToElement(vertex, &pReal);
				Ogre::Vector3 pt(pReal[0], pReal[1], pReal[2]);
				vertices[current_offset + j] = (orient * (pt * scale)) + position;
			}

			vbuf->unlock();
			next_offset += vertex_data->vertexCount;
		}


		Ogre::IndexData* index_data = submesh->indexData;
		size_t numTris = index_data->indexCount / 3;
		Ogre::HardwareIndexBufferSharedPtr ibuf = index_data->indexBuffer;

		bool use32bitindexes = (ibuf->getType() == Ogre::HardwareIndexBuffer::IT_32BIT);

		unsigned long*  pLong = static_cast<unsigned long*>(ibuf->lock(Ogre::HardwareBuffer::HBL_READ_ONLY));
		unsigned short* pShort = reinterpret_cast<unsigned short*>(pLong);


		size_t offset = (submesh->useSharedVertices)? shared_offset : current_offset;
		size_t index_start = index_data->indexStart;
		size_t last_index = numTris*3 + index_start;

		if (use32bitindexes){
			for( size_t k = index_start; k<last_index; ++k )
				indices[index_offset++] = pLong[k] + static_cast<unsigned long>( offset );
		}else{
			for( size_t k=index_start; k<last_index; ++k )
				indices[ index_offset++ ] =
					static_cast<unsigned long>( pShort[k] ) +
					static_cast<unsigned long>( offset );
		}

		ibuf->unlock();
		current_offset = next_offset;
	}
}

static void getMeshInformation( const Ogre::ManualObject* manual, size_t& vertex_count, Ogre::Vector3*& vertices, size_t& index_count, unsigned long*& indices, const Ogre::Vector3& position, const Ogre::Quaternion& orient, const Ogre::Vector3& scale )
{
	std::vector<Ogre::Vector3> returnVertices;
	std::vector<unsigned long> returnIndices;
	unsigned long thisSectionStart = 0;
	for ( unsigned int i=0,size=manual->getNumSections(); i<size; ++i )
	{
		Ogre::ManualObject::ManualObjectSection* section = manual->getSection(i);
		Ogre::RenderOperation* renderOp = section->getRenderOperation();

		std::vector<Ogre::Vector3> pushVertices;
		//Collect the vertices
		{
			const Ogre::VertexElement* vertexElement = renderOp->vertexData->vertexDeclaration->findElementBySemantic(Ogre::VES_POSITION);
			Ogre::HardwareVertexBufferSharedPtr vertexBuffer = renderOp->vertexData->vertexBufferBinding->getBuffer(vertexElement->getSource());

			char* verticesBuffer = static_cast<char*>(vertexBuffer->lock(Ogre::HardwareBuffer::HBL_READ_ONLY));
			float* positionArrayHolder;

			thisSectionStart = returnVertices.size();

			pushVertices.reserve(renderOp->vertexData->vertexCount);

			for( unsigned int j=0; j<renderOp->vertexData->vertexCount; ++j )
			{
				vertexElement->baseVertexPointerToElement(verticesBuffer + j * vertexBuffer->getVertexSize(), &positionArrayHolder);
				Ogre::Vector3 vertexPos = Ogre::Vector3(positionArrayHolder[0],positionArrayHolder[1],positionArrayHolder[2]);
				vertexPos = (orient * (vertexPos * scale)) + position;
				pushVertices.push_back(vertexPos);
			}

			vertexBuffer->unlock();
		}
		//Collect the indices
		{
			if( renderOp->useIndexes ){
				Ogre::HardwareIndexBufferSharedPtr indexBuffer = renderOp->indexData->indexBuffer;

				if( indexBuffer.isNull() || renderOp->operationType != Ogre::RenderOperation::OT_TRIANGLE_LIST ){
					//No triangles here, so we just drop the collected vertices and move along to the next section.
					continue;
				}else{
					returnVertices.reserve(returnVertices.size() + pushVertices.size());
					returnVertices.insert(returnVertices.end(), pushVertices.begin(), pushVertices.end());
				}

				unsigned int* pLong = static_cast<unsigned int*>(indexBuffer->lock(Ogre::HardwareBuffer::HBL_READ_ONLY));
				unsigned short* pShort = reinterpret_cast<unsigned short*>(pLong);

				returnIndices.reserve(returnIndices.size() + renderOp->indexData->indexCount);

				for( size_t j=0; j<renderOp->indexData->indexCount; ++j )
				{
					unsigned long index;
					//We also have got to remember that for a multi section object, each section has
					//different vertices, so the indices will not be correct. To correct this, we
					//have to add the position of the first vertex in this section to the index

					//(At least I think so...)
					if( indexBuffer->getType() == Ogre::HardwareIndexBuffer::IT_32BIT )
						index = static_cast<unsigned long>(pLong[j]) + thisSectionStart;
					else
						index = static_cast<unsigned long>(pShort[j]) + thisSectionStart;

					returnIndices.push_back(index);
				}

				indexBuffer->unlock();
			}
		}
	}

	//Now we simply return the data.
	index_count = returnIndices.size();
	vertex_count = returnVertices.size();
	vertices = new Ogre::Vector3[vertex_count];
	for( unsigned long i = 0; i<vertex_count; ++i )
		vertices[i] = returnVertices[i];
	indices = new unsigned long[index_count];
	for( unsigned long i = 0; i<index_count; ++i )
		indices[i] = returnIndices[i];

	//All done.
	return;
}

///////////////////////////////////////////////////////////////////////////////


///////////////////////////////////////////////////////////////////////////////

}}
