/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TextObject.h
 *    Adapted from http://www.ogre3d.org/tikiwiki/MovableText&structure=Cookbook
 *
 * Original file: MovableText.h
 *
 * description: This create create a billboarding object that display a text.
 *
 * @author  2003 by cTh see gavocanov@rambler.ru
 * @update  2006 by barraq see nospam@barraquand.com
 */

#ifndef _MIRA_TEXTOBJECT_H_
#define _MIRA_TEXTOBJECT_H_

#include <visualization/GuiVisualizationExports.h>

#ifndef Q_MOC_RUN
#include <OGRE/OgreMovableObject.h>
#include <OGRE/OgreRenderable.h>
#include <OGRE/OgreVector3.h>
#include <OGRE/OgreQuaternion.h>
#endif

#if (OGRE_VERSION >= 0x010900)
#include <OGRE/Overlay/OgreFont.h>
#endif

#include <visualization/3d/VisualizationObject.h>

namespace Ogre
{
	class RenderQueue;
	class Camera;
}

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class MIRA_GUI_VISUALIZATION_EXPORT TextObject: public Ogre::MovableObject, public Ogre::Renderable, public VisualizationObject
{
	/******************************** TextObject data ****************************/
public:
	enum HorizontalAlignment
	{
		H_LEFT,
		H_CENTER,
		H_RIGHT
	};
	enum VerticalAlignment
	{
		V_BELOW,
		V_ABOVE,
		V_CENTER
	};

protected:
	Ogre::String mFontName;
	Ogre::String mType;
	Ogre::String mName;
	Ogre::String mCaption;
	HorizontalAlignment mHorizontalAlignment;
	VerticalAlignment mVerticalAlignment;

	Ogre::ColourValue mColor;
	Ogre::RenderOperation mRenderOp;
	Ogre::AxisAlignedBox mAABB;
	Ogre::LightList mLList;

	Ogre::Real mCharHeight;
	Ogre::Real mSpaceWidth;

	bool mNeedUpdate;
	bool mUpdateColors;
	bool mOnTop;

	Ogre::Real mTimeUntilNextToggle;
	Ogre::Real mRadius;

	Ogre::Vector3 mGlobalTranslation;
	Ogre::Vector3 mLocalTranslation;

	Ogre::Camera *mCam;
	Ogre::RenderWindow *mWin;
	Ogre::Font *mFont;
	Ogre::MaterialPtr mMaterial;
	Ogre::MaterialPtr mBackgroundMaterial;

	/******************************** public methods ******************************/
public:

	TextObject(const Ogre::String &caption,
	           Ogre::SceneManager* sceneManager,
	           Ogre::SceneNode* parent=NULL);

	TextObject(const Ogre::String &caption,
	           const Ogre::String &fontName,
	           Ogre::SceneManager* sceneManager,
	           Ogre::SceneNode* parent=NULL);

	virtual ~TextObject();

	virtual void visitRenderables(Ogre::Renderable::Visitor* visitor,
	                              bool debugRenderables = false);

	virtual void setColor(const Ogre::ColourValue &color);
	virtual void setVisible(bool visible, bool cascade = true);

	// Set settings
	void setFontName(const Ogre::String &fontName);
	void setCaption(const Ogre::String &caption);
	void setCharacterHeight(Ogre::Real height);
	void setSpaceWidth(Ogre::Real width);
	void setTextAlignment(const HorizontalAlignment& horizontalAlignment,
	                      const VerticalAlignment& verticalAlignment);
	void setGlobalTranslation(Ogre::Vector3 trans);
	void setLocalTranslation(Ogre::Vector3 trans);
	void showOnTop(bool show = true);

	// Get settings
	const Ogre::String& getFontName() const
	{
		return mFontName;
	}
	const Ogre::String& getCaption() const
	{
		return mCaption;
	}
	const Ogre::ColourValue& getColor() const
	{
		return mColor;
	}

	Ogre::Real getCharacterHeight() const
	{
		return mCharHeight;
	}
	Ogre::Real getSpaceWidth() const
	{
		return mSpaceWidth;
	}
	Ogre::Vector3 getGlobalTranslation() const
	{
		return mGlobalTranslation;
	}
	Ogre::Vector3 getLocalTranslation() const
	{
		return mLocalTranslation;
	}
	bool getShowOnTop() const
	{
		return mOnTop;
	}
	Ogre::AxisAlignedBox GetAABB(void)
	{
		return mAABB;
	}

	/******************************** protected methods and overload **************/
protected:

	// from TextObject, create the object
	void _setupGeometry();
	void _updateColors();

	// from MovableObject
	void getWorldTransforms(Ogre::Matrix4 *xform) const;
	Ogre::Real getBoundingRadius(void) const
	{
		return mRadius;
	}

	Ogre::Real getSquaredViewDepth(const Ogre::Camera *cam) const
	{
		return 0;
	}

	const Ogre::Quaternion& getWorldOrientation(void) const;
	const Ogre::Vector3& getWorldPosition(void) const;
	const Ogre::AxisAlignedBox& getBoundingBox(void) const
	{
		return mAABB;
	}

	const Ogre::String& getName(void) const
	{
		return mName;
	}

	const Ogre::String& getMovableType(void) const
	{
		static Ogre::String movType = "TextObject";
		return movType;
	}

	void _notifyCurrentCamera(Ogre::Camera *cam);
	void _updateRenderQueue(Ogre::RenderQueue* queue);

	// from renderable
	void getRenderOperation(Ogre::RenderOperation &op);
	const Ogre::MaterialPtr& getMaterial(void) const
	{
		assert(!mMaterial.isNull());
		return mMaterial;
	}

	const Ogre::LightList& getLights(void) const
	{
		return mLList;
	}

};

///////////////////////////////////////////////////////////////////////////////

}

#endif
