/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationView.h
 *    Declaration of VisualizationView and VisualizationViewTransformable.
 *
 * @author Erik Einhorn
 * @date   2010/12/29
 */

#ifndef _MIRA_VISUALIZATIONVIEW_H_
#define _MIRA_VISUALIZATIONVIEW_H_

#include <QBasicTimer>

#include <factory/NullClass.h>
#include <utils/Time.h>
#include <serialization/adapters/std/list>
#include <serialization/adapters/Qt/QAction>
#include <rcp/EditorPart.h>

#include <views/GuiViewsExports.h>
#include <fw/TransformProperty.h>
#include <visualization/VisualizationTool.h>

class QMimeData;
class QDragEnterEvent;
class QDropEvent;
class QFocusEvent;
class QBoxLayout;
class QToolBar;
class QToolButton;
class QButtonGroup;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class Visualization;
class VisualizationTool;
class VisualizationControlPage;

/**
 * Abstract base class for 2D, 3D, text and plot visualization views (namely
 * Visualization2DView and Visualization3DView...).
 *
 */
class MIRA_GUI_VIEWS_EXPORT VisualizationView : public EditorPart
{
	Q_OBJECT
	MIRA_OBJECT(VisualizationView)

public:

	VisualizationView();
	virtual ~VisualizationView();


public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, EditorPart);
		r.member("Visualizations", mVisualizations, "The assigned visualizations");

		r.property("Update",
		           mUpdateInterval,
		           setter(&VisualizationView::setUpdateInterval,this),
		           "Update interval in ms",
		           40);

		r.property("WindowTitle",
		           getter(&VisualizationView::getWindowTitle,this),
		           setter(&VisualizationView::setWindowTitle,this),
		           "Title of window",
		           getClass().getMetaInfo("Name"));

		r.member("HideTools",
		         getter<bool>(boost::bind(actionGetter, mHideToolBar)),
		         setter<bool>(boost::bind(actionSetter, mHideToolBar, _1)),
		         "Hide tool bar with tools", false);

		r.property("Tools", mTools, "The used tools");
	}
	/// Specialization of reflect method for XML
	void reflect(XMLDeserializer& r);
	/// Specialization of reflect method for XML
	void reflect(XMLSerializer& r);

public:

	/**
	 * Adds an existing visualization to this view. The view will take
	 * ownership of the visualization and will destroy it upon its own
	 * deletion.
	 */
	virtual void addVisualization(Visualization* vis);

	/**
	 * Remove the specified visualization from this view. The view will give
	 * up its ownership of the visualization and the Visualization must be
	 * destroyed outside of this view.
	 */
	virtual void removeVisualization(Visualization* vis);

	/**
	 * Change order of visualizations in the view, move the specified
	 * visualization up. The base VisualizationView just changes the order
	 * in the stored list of visualizations. Subclasses should overwrite this
	 * method to adapt their respective GUI accordingly.
	 */
	virtual void moveUpVisualization(Visualization* vis);

	/**
	 * Change order of visualizations in the view, move the specified
	 * visualization down. The base VisualizationView just changes the order
	 * in the stored list of visualizations. Subclasses should overwrite this
	 * method to adapt their respective GUI accordingly.
	 */
	virtual void moveDownVisualization(Visualization* vis);

	const std::list<Visualization*>& getVisualizations() const;

	/**
	 * Adds an existing tool to this view. The view will take
	 * ownership of the tool and will destroy it upon its own
	 * deletion.
	 */
	virtual void addTool(VisualizationTool* tool);

	const std::vector<VisualizationTool*>& getTools() const;

public:

	/**
	 * Returns pointer to VisualizationControlPage (or NULL, if no such page
	 * was created yet.
	 */
	VisualizationControlPage* getControl();

public:

	virtual Object* getAdapter(const Class& adapter);

public:

	/**
	 * Derived visualization views must return the base class of their supported
	 * visualizations.
	 */
	virtual const Class& supportedVisualizationClass() const = 0;

	/**
	 * If a default visualization class is provided, this class is used to
	 * instantiate a default visualization for data objects, no other specialized
	 * visualization is found for.
	 * If no default visualization class exists, such data objects can not
	 * be visualized (the default behavior).
	 */
	virtual const Class& defaultVisualizationClass() const {
		return NullClass::null();
	}

	/**
	 * Derived visualization views may return the base class of their supported
	 * tools. The default implementation returns a null-class to indicate that
	 * no tools are supported.
	 */
	virtual const Class& supportedVisualizationToolClass() const {
		return NullClass::null();
	}

protected:

	virtual QWidget* createPartControl();

	virtual QWidget* createVisualizationPart() = 0;

	virtual void resetView();
	virtual void saveContentToFile();

protected:
	/**
	 * Is called by this class within the update timer event to update
	 * the view. The visualizations of the view have been updated before
	 * this method is called.
	 * The update cycle must be enabled by calling startUpdateTimer() after
	 * the derived class and its GUI components have been enabled
	 */
	virtual void update(Duration dt) = 0;

public:

	void setUpdateInterval(int interval);
	int  getUpdateInterval();

public:

	/// Returns the current active tool, or NULL of no tool is active.
	VisualizationTool* getActiveTool() { return mActiveTool; }
	void activateTool(VisualizationTool* tool);

public:

	void setOfferAuxiliaryVisualizations(bool on) {
		mOfferAuxiliaryVisualizations = on;
	}

protected:

	void startUpdateTimer();

	virtual void timerEvent(QTimerEvent*);
	bool eventFilter(QObject *obj, QEvent *event);

	void keyPressEvent(QKeyEvent* event);
	void keyReleaseEvent(QKeyEvent* event);

	void focusOutEvent(QFocusEvent* event);

protected:

	std::string getChannelIDFromMimeData(const QMimeData* mimeData);
	std::list<ClassProxy> getVisualizationsFor(const std::string& channelID);
	void dragEnterEventImpl(QDragEnterEvent *event);
	void dropEventImpl(QDropEvent* event, QWidget* widget);

protected:
	void destroyVisualizations();
	void destroyTools();

private:
	void setToolbarArea(Qt::ToolBarArea area);
	void addToolButton(VisualizationTool* tool, const QString& name, const QString& category, const QString& description);

	// Will enable this tool as the currently active one in its category
	void switchToTool(VisualizationTool* tool, bool setChecked = false);

	// Returns the category of this tool, or the default category if none exists
	QString getCategory(VisualizationTool* tool) {
		return tool->getClass().getMetaInfo("Category").empty() ? DEFAULT_CATEGORY :
				QString::fromLocal8Bit(tool->getClass().getMetaInfo("Category").c_str());
	}

	void setWindowTitle( std::string const& title );
	std::string getWindowTitle() const;

private slots:
	void onToolAction();
	void onToolButton();
	void onHideToolbar(bool);

private:
	VisualizationControlPage* mControl;
	std::list<Visualization*> mVisualizations;

	std::vector<VisualizationTool*> mTools;
	XMLDom mToolsState;

	// Convenience mapping from tool id (<category>_<toolname>) to its tool.
	std::map<QString, VisualizationTool*> mIdToTool;

	// Stores the tool of each category that is currently in the top slot.
	std::map<QString, VisualizationTool*> mCurrentTool;

	// Convenience mapping from category string to the corresponding tool button.
	std::map<QString, QToolButton*> mCategoryToToolButton;

	// Currently active tool
	VisualizationTool* mActiveTool;

	// When e.g. CTRL is pressed, this will store the old tool while the new one is active.
	VisualizationTool* mDisplacedTool;

	// Manage our buttons, only one may be pressed at the same time
	QButtonGroup* mButtonGroup;

private:
	QBasicTimer mUpdateTimer;
	int mUpdateInterval;
	Time mLastUpdateTime;

	QBoxLayout* mLayout;
	QToolBar* mToolBar;
	QAction* mHideToolBar;
	QWidget* mVisualizationPart;
	Qt::ToolBarArea mCurrentToolboxArea;

	const QString DEFAULT_CATEGORY;

	bool mOfferAuxiliaryVisualizations;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Abstract base class for 2D, 3D, text and plot visualization views (namely
 * Visualization2DView and Visualization3DView...) that support a target frame.
 *
 */
class MIRA_GUI_VIEWS_EXPORT VisualizationViewTransformable : public VisualizationView
{
	Q_OBJECT
	MIRA_OBJECT(VisualizationViewTransformable)

public:

	VisualizationViewTransformable();
	virtual ~VisualizationViewTransformable();

public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, VisualizationView);
		r.property("Fixed Frame", mFixedFrame, "The reference frame used to denote the world frame. Should not be moving (e.g. map frame, etc.)", "");
		r.property("Camera Frame", mCameraFrame,
		           setter(&VisualizationViewTransformable::setCameraFrame, this),
		           "The frame where all other frames are transformed to for displaying the data", TransformProperty());
	}

protected:

	virtual void setCameraFrame(const TransformProperty& frame) { mCameraFrame = frame; }

protected:

	TransformProperty mFixedFrame;
	TransformProperty mCameraFrame;

private slots:

	void checkFrames();
	
private:

	std::string mLastAskedFixedFrame; // the fixed frame value, we have asked to set the frame automatically
	std::string mLastAskedCameraFrame; // the camera frame value, we have asked to set it automatically
	QTimer* mFrameCheckTimer;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
