/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationTextView.h
 *    View for text visualizations.
 *
 * @author Tim Langner, Christof Schröter
 * @date   2012/04/16
 */

#ifndef _MIRA_VISUALIZATIONTEXTVIEW_H_
#define _MIRA_VISUALIZATIONTEXTVIEW_H_

#include <QTableWidget>
#include <QPlainTextEdit>
#include <QPushButton>

#include <serialization/Serialization.h>
#include <serialization/SetterNotify.h>

#include <visualization/VisualizationText.h>
#include <visualization/VisualizationTextBasic.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class VisualizationTextItem : public QObject, public IVisualizationTextItem
{
	Q_OBJECT;
public:

	VisualizationTextItem(QTableWidget* t, int row);

public slots:
	void onSetClicked();

public:

	void setName(const std::string& name);

	void setText(const std::string& text);

	virtual void setStamped(const Time& timestamp, const std::string& frameID,
	                        uint32 sequenceID);

	int row() const;

	virtual void enableEdit(bool enable);
	virtual void registerTextEditRequestCallback(boost::function<void ()> fn);
	virtual void setSize(const Size2i& size);
	virtual Size2i getSize() const;

	QTableWidget* table;
	QTableWidgetItem* nameItem;
	QPlainTextEdit* textItem;
	QTableWidgetItem* timestampItem;
	QTableWidgetItem* sequenceidItem;
	QTableWidgetItem* frameidItem;
	QPushButton* setButton;
	QWidget* setItem;

protected:
	
	boost::function<void ()> mEditRequestCb;
};

///////////////////////////////////////////////////////////////////////////////

template<typename T>
class DefaultTextVisualizationBase : public VisualizationTextBasic<T>
{
public:

	typedef VisualizationTextBasic<T> Base;

	DefaultTextVisualizationBase(const std::string& channelName)
		: Base(channelName), mShowROProperties(true), mStringMapAsObject(true)
	{}

	template <typename Reflector>
	void reflect(Reflector& r) {
		MIRA_REFLECT_BASE(r, Base);
		r.property("ShowReadOnlyProperties", mShowROProperties,
		           setterNotify(mShowROProperties, &DefaultTextVisualizationBase::redraw, this),
		           "Show objects' read-only properties", true);
		r.property("StringMapAsObject", mStringMapAsObject,
		           setterNotify(mStringMapAsObject, &DefaultTextVisualizationBase::redraw, this),
		           "Serialize any map<string, T> to json object "
		           "(enables simple access to map items by key)", true);
	}

	virtual void dataChanged(ChannelRead<T> data)
	{
		this->mItem->setStamped(data->timestamp, data->frameID, data->sequenceID);

		try
		{
			json::Value v;
			try {
				JSONSerializer js(mShowROProperties,
				                  this->mStringMapAsObject ? JSONSerializer::STRING_MAP_AS_OBJECT
				                                           : JSONSerializer::STANDARD);
				data.readJSON(v, js);
			}
			catch(XNotImplemented&)
			{
				data.readJSON(v);
			}
			this->mLastValue.reset(v);
			this->mItem->setText(json::write(v, this->mFormatted, this->mPrecision));
		}
		catch(Exception& ex)
		{
			int typeID = MIRA_FW.getChannelManager().getTypeId(Base::mChannel.getID());
			if (typeID < 0)
				this->mItem->setText(MakeString() << "Conversion to JSON format (from untyped channel's "
				                     "serialized data + metadata) failed:\n" << ex.message());
			else
				this->mItem->setText(MakeString() << "Serialization to JSON format not supported by "
				                     "channel type:\n" << ex.message());
		}
		catch(...)
		{
			int typeID = MIRA_FW.getChannelManager().getTypeId(Base::mChannel.getID());
			if (typeID < 0)
				this->mItem->setText("Conversion to JSON format (from untyped channel's "
				                     "serialized data + metadata) failed");
			else
				this->mItem->setText("Serialization to JSON format not supported by channel type");
		}
	}

	virtual void setupScene(IVisualizationTextItem* item)
	{
		item->registerTextEditRequestCallback(boost::bind(&DefaultTextVisualizationBase::onEdit, this));
		item->enableEdit(true);
	}

	virtual void redraw()
	{
		try {
			if (this->mChannel.isValid())
				dataChanged(this->mChannel.getChannel().read());
		} catch (...) {}
	}

protected:
	bool mShowROProperties;
	bool mStringMapAsObject;
};

///////////////////////////////////////////////////////////////////////////////

template<typename T>
class TypedDefaultTextVisualizationBase : public DefaultTextVisualizationBase<T>
{
public:

	typedef DefaultTextVisualizationBase<T> Base;

	TypedDefaultTextVisualizationBase(const std::string& channelName)
		: Base(channelName)
	{}
	
public:

	virtual std::string defaultText()
	{
		JSONSerializer js(this->mShowROProperties,
		                  this->mStringMapAsObject ? JSONSerializer::STRING_MAP_AS_OBJECT
		                                           : JSONSerializer::STANDARD);
		json::Value value = js.serialize(T());
		return json::write(value, true, this->mPrecision);
	}
};

///////////////////////////////////////////////////////////////////////////////

}

#endif 
