/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ServiceLevelTest.C
 *    Test ServiceLevel
 *
 * @author Christof Schröter
 * @date   2021/10/04
 */

#include <boost/test/unit_test.hpp>

#include "ServiceLevelTestCodec.h"

#include <image/Img.h>
#include <fw/Framework.h>

using namespace mira;

// Testing ServiceLevel: codec, update interval

struct TestAuthority : public Authority
{
	TestAuthority() : Authority("/TestAuthority"), receiveCount1(0), receiveCount2(0) {}
	void onImg1(ChannelRead<Img8U1> read) {img1=*read; ++receiveCount1;}
	void onImg2(ChannelRead<Img8U1> read) {img2=*read; ++receiveCount2;};

	int receiveCount1, receiveCount2;
	Img8U1 img1, img2;
};

BOOST_AUTO_TEST_CASE(ServiceLevelTest)
{
	const char* argv[] = {"ServiceLevelTest",
	                      "${find ServiceLevelTest.xml}"};
	Framework fw(2,(char**)argv);
	fw.load();
	fw.start();

	TestAuthority tester;
	BOOST_REQUIRE(tester.waitForService("/PublisherUnit", Duration::seconds(5)));

	tester.subscribe<Img8U1>("/Img1", &TestAuthority::onImg1);
	tester.subscribe<Img8U1>("/Img2", &TestAuthority::onImg2);

	BOOST_CHECK_EQUAL(mira::fw::test::DummyCodec::sDecodeCount, 0);

	// start publishing
	tester.callService<void>("/PublisherUnit", "enablePublish", true).get();

	MIRA_SLEEP(3500)

	// we get one update every 1000ms for Img2, so we should have 4 (0ms, 1000ms, 2000ms, 3000ms)
	BOOST_CHECK_EQUAL(tester.receiveCount2, 4);

	// stop publishing so the number of updates is defined
	tester.callService<void>("/PublisherUnit", "enablePublish", false).get();

	// wait 1000ms more so we get the final update
	MIRA_SLEEP(1000)

	// now we should have 5
	BOOST_CHECK_EQUAL(tester.receiveCount2, 5);

	// on Img1 we get every update, so receiveCount1 should equal the 'PublishCount' property
	std::string countstr = tester.callService<std::string>("/PublisherUnit#builtin", "getProperty", "PublishCount");
	int publishCount = boost::lexical_cast<int>(countstr);
	BOOST_CHECK_EQUAL(tester.receiveCount1, publishCount);

	// on both channels, the final received updates should be the final published image
	// (check by comparing pixel value = publish count)
	const int HEIGHT = 100;
	const int WIDTH = 100;
	BOOST_CHECK_EQUAL(tester.img1.height(), HEIGHT);
	BOOST_CHECK_EQUAL(tester.img1.width(), WIDTH);
	BOOST_CHECK_EQUAL(tester.img2.height(), HEIGHT);
	BOOST_CHECK_EQUAL(tester.img2.width(), WIDTH);

	for (int y=0; y<HEIGHT; ++y) {
		uint8* data1 = tester.img1.data(y);
		uint8* data2 = tester.img2.data(y);
		for (int x=0; x<WIDTH; ++x) {
			BOOST_CHECK_EQUAL(data1[x], publishCount);
			BOOST_CHECK_EQUAL(data2[x], publishCount);
		}
	}

	// check the right codec was used
	BOOST_CHECK_EQUAL(mira::fw::test::DummyCodec::sDecodeCount, publishCount);
}
