/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DynamicPropertiesUnits.C
 *
 * @author Christof Schröter
 * @date   2020/06/03
 */

#include <fw/Unit.h>

#include <serialization/adapters/std/vector>
#include <serialization/GetterSetter.h>

#include <boost/interprocess/sync/scoped_lock.hpp>

namespace mira { namespace fw {

///////////////////////////////////////////////////////////////////////////////

class DynamicPropertiesUnit : public Unit
{
	MIRA_OBJECT(DynamicPropertiesUnit)
public:

	DynamicPropertiesUnit() :
		Unit(Duration::milliseconds(500)), mStatus(0), mBool(false), mInt(0)
	{
	}

	/// Reflect method for serialization and services
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Unit);
		// changing the reflected content with the call to reflect() should not be
		// a real usecase, just included here for demonstration/testing

		r.property("DynStatus", mStatus, "dynstatus", 0);

		switch(mStatus) {
			case 0: r.property("Bool", mBool, "bool", false);
			        r.property("Int", mInt, "int", 0);
			        break;
			case 1: r.property("Ints", mInts, "ints", std::vector<int>());
			        r.property("Ints2", mInts, "ints", std::vector<int>());
			        break;
			case 2: r.property("Int", mInt, "int", 0);
			        r.property("Bool", mBool, "bool", false);
					break;
			case 3: r.property("Ints2", mInts, "ints", std::vector<int>());
			        r.property("Ints", mInts, "ints", std::vector<int>());
			        break;
			default:
			        break;
		}
	}

	virtual void process(const Timer& timer)
	{
		mStatus = (mStatus+1) % 4;

		mBool = !mBool;
		++mInt;

		mInts.clear();
		for (int i = 0; i <= mStatus; ++i)
			mInts.push_back(i);

		// this will call reflect() through PropertySerializer::reflectProperties()
		getProperties()->synchronize();
	}

protected:
	int mStatus;

	bool mBool;
	int mInt;

	std::vector<int> mInts;
};

///////////////////////////////////////////////////////////////////////////////

class DynamicContainersUnit : public Unit
{
	MIRA_OBJECT(DynamicContainersUnit)
public:

	struct Content {
		template<typename Reflector>
		void reflect(Reflector& r)
		{
			r.property("n", n, "");
			r.property("n2",
			           getter(&Content::getN2, this),
			           setter(&Content::setN2, this),
			           "");
		}
		
		Content() {} // required for serialization
		Content(int i) : n(i), n2(i) {}

		int getN2() { return n2; }
		void setN2(int v) { n2 = v; }

		int n;
		int n2;
	};

	DynamicContainersUnit() :
		Unit(Duration::milliseconds(2000)), mBigVector{0}, mFluidGrow(true)
	{
	}

	~DynamicContainersUnit()
	{
		foreach (auto& e, mFluidList)
			delete e;
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Unit);

		r.property("BigVector", mBigVector, "a vector perpetually growing", std::vector<int>{0});
		r.property("FluidList", mFluidList, "a list alternately growing and shrinking", std::list<Content*>());
	}

	virtual void process(const Timer& timer)
	{
		{
			boost::interprocess::scoped_lock<RootPropertyNode> lock(mPropertiesRoot);
			mBigVector.push_back(mBigVector.size());

			mFluidGrow = mFluidGrow ? (mFluidList.size() < 5 ) :  // grow until reaching 5, then shrink
				                      (mFluidList.size() == 0);   // start growing when reaching 0 
			if (mFluidGrow)
				mFluidList.push_back(new Content(mFluidList.size()));
			else {
				delete mFluidList.back();
				mFluidList.pop_back();
			}

			// artifically large gap here to provoke errors accessing the invalid
			// container elements between resize and synchronize
			MIRA_SLEEP(500);

			// this will call reflect() through PropertySerializer::reflectProperties()
			getProperties()->synchronize();
		}
		MIRA_SLEEP(100);
	}

protected:
	std::vector<int> mBigVector;

	bool mFluidGrow;
	std::list<Content*> mFluidList;
};

///////////////////////////////////////////////////////////////////////////////

// Test/demonstrate editing one property in PropertyEditor to affect
// other properties
class DependentPropertiesUnit : public MicroUnit
{
	MIRA_OBJECT(DependentPropertiesUnit)
public:

	/// Reflect method for serialization and services
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, MicroUnit);
		r.property("PropertyName", mPropertyName,
		            setter(&DependentPropertiesUnit::setPropertyName, this),
		           "name of dependent property", "");
		r.roproperty(("P_"+mPropertyName).c_str(), mDependentProperty, "dependent property");

		r.property("ReadOnly?", mReadOnly,
		           setter(&DependentPropertiesUnit::makeReadOnly, this),
		           "make 'Int' read only?", true);
		if (mReadOnly)
			r.roproperty("Int", "Int_ro", mDependentProperty2,
			             "dependent property switching between read/write and read-only");
		else
			r.property("Int", "Int_rw", mDependentProperty2,
			           "dependent property switching between read/write and read-only");
	}

	void setPropertyName(const std::string& name) {
		if (!isValid()) {
			// this is initial deserialization, before the unit is setup/initialized
			mPropertyName = name;
			return;
		}

		// lock is not really needed in this example, but added for testing purposes ...
		boost::interprocess::scoped_lock<RootPropertyNode> lock(mPropertiesRoot);
		mPropertyName = name;
		getProperties()->synchronize();
	}

	void makeReadOnly(bool ro) {
		if (!isValid()) {
			// this is initial deserialization, before the unit is setup/initialized
			mReadOnly = ro;
			return;
		}

		// lock is not really needed in this example, but added for testing purposes ...
		boost::interprocess::scoped_lock<RootPropertyNode> lock(mPropertiesRoot);
		mReadOnly = ro;
		getProperties()->synchronize();
	}

protected:
	std::string mPropertyName;
	std::string mDependentProperty = "RO-Value";

	bool mReadOnly = true;
	int mDependentProperty2 = 1;
};

///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION( mira::fw::DynamicPropertiesUnit, mira::Unit );
MIRA_CLASS_SERIALIZATION( mira::fw::DynamicContainersUnit, mira::Unit );
MIRA_CLASS_SERIALIZATION( mira::fw::DependentPropertiesUnit, mira::MicroUnit );
