/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SpawnLoader.C
 *    Loader plugin that parses the spawn tag
 *
 * @author Erik Einhorn
 * @date   2011/11/18
 */

#include <platform/Process.h>

#include <loader/Loader.h>
#include <fw/Framework.h>

#include "private/ProcessSpawnManager.h"

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Loader plugin that allows to parse the spawn tag to spawn processes from
 * within the configuration file.
 */
class SpawnLoader : public ConfigurationLoaderPlugin
{
	MIRA_META_OBJECT(SpawnLoader, ("Tag", "spawn"))
public:

	ProcessSpawnManager mProcessManager;

	virtual void parseNode(const XMLDom::const_iterator& node,
	                       ConfigurationLoader* loader)
	{
		auto commandIt = node.find_attribute("command");
		if (commandIt == node.attribute_end() || commandIt.value().empty())
			MIRA_THROW(XInvalidConfig, "Error in <spawn>. "
			           "No or empty command attribute specified. "
			           "Usage <spawn command=\"myExecutable\" />.");
		std::string commandString = commandIt.value();

		bool required;
		bool respawn;
		std::string name;

		auto nameIt = node.find_attribute("name");
		if(nameIt != node.attribute_end())
			name = nameIt.value();

		respawn = false;
		auto respawnIt = node.find_attribute("respawn");
		if(respawnIt != node.attribute_end())
			respawn = (respawnIt.value()=="true");

		required = false;
		auto requiredIt = node.find_attribute("required");
		if(requiredIt != node.attribute_end())
			required = (requiredIt.value()=="true");

		std::string machine;
		auto machineIt = node.find_attribute("machine");
		if(machineIt != node.attribute_end())
			machine = machineIt.value();

		// get executable and arguments
		Path executable;

		std::vector<std::string> args;

		// extract specified additional parameters
		std::vector<std::string> addargs;
		assert(!commandString.empty());
		boost::split(addargs, commandString, boost::algorithm::is_any_of(" \t\n"),
		             boost::algorithm::token_compress_on);
		assert(addargs.size()>=1);
		std::vector<std::string>::iterator beginaddargs = addargs.begin();

		if(machine.empty()) { // local
			// extract the executable name from the arg list
			executable = addargs[0];
			++beginaddargs; // skip the executable filename in our additional arguments below
		} else { // remote
			executable = "ssh";
			args.push_back("-t");
			args.push_back(machine);
		}

		// append real arguments from addargs to args
		std::copy(beginaddargs,addargs.end(), std::back_inserter(args));

		std::cout << print(args) << std::endl;

		mProcessManager.startProcess(executable, args, name, respawn, required);
		mProcessManager.startWatchdog(
				boost::bind(&SpawnLoader::requiredProcessTerminated, this,_1));
	}

	void requiredProcessTerminated(const ProcessSpawnManager::ProcessInfo& info)
	{
		MIRA_LOG(WARNING) << "Required process '" << info.getName() << "' "
		                     "was terminated. Stopping ourselves ...";
		MIRA_FW.requestTermination();
	}


};

///////////////////////////////////////////////////////////////////////////////
}

MIRA_CLASS_REGISTER(mira::SpawnLoader, mira::ConfigurationLoaderPlugin)

