/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ProcessLoader.C
 *    Loader plugin that parses the process tag
 *
 * @author Erik Einhorn
 * @date   2013/03/23
 */

#include <platform/Process.h>

#include <boost/algorithm/string.hpp>
#include <boost/tuple/tuple.hpp>
#include <boost/exception/diagnostic_information.hpp>

#include <vector>
#include <list>
#include <unistd.h>

#ifndef MIRA_WINDOWS
#      include <pwd.h>
# endif

#include <utils/Functionals.h>
#include <utils/Path.h>
#include <utils/PathFinder.h>
#include <utils/MakeString.h>

#include <loader/Loader.h>
#include <fw/Framework.h>
#include <fw/RemoteModule.h>
#include "private/MaximumArgumentListSizeEstimation.h"

#include "private/ProcessSpawnManager.h"

namespace mira {

inline const Path& getMiraPath()
{
	static Path miraPath;
	if(miraPath.empty()) {
#ifdef MIRA_WINDOWS
# ifdef NDEBUG
		miraPath = findProjectFile("bin/mira.exe");
# else
		miraPath = findProjectFile("bin/mira_d.exe");
# endif
#else
		miraPath = findProjectFile("bin/mira");
#endif
	}
	return miraPath;
}

inline const Path& getMiraguiPath()
{
	static Path miraPath;
	if(miraPath.empty()) {
#ifdef MIRA_WINDOWS
		miraPath = findProjectFile("bin/miragui.exe");
#else
		miraPath = findProjectFile("bin/miragui");
#endif
	}
	return miraPath;
}

///////////////////////////////////////////////////////////////////////////////

/**
 * Loader plugin that allows to parse the process tag to start other
 * mira processes from within the same configuration file.
 *
 */
class ProcessLoader : public ConfigurationLoaderPlugin
{
	MIRA_META_OBJECT(ProcessLoader,
	                 ("Tag", "process")
	                 ("StartEndDocument", "true"))
public:

	ProcessLoader()
	:	ConfigurationLoaderPlugin()
	{
		initializeFWUsingLoader();
	}

	ProcessSpawnManager mProcessManager;

	void removeComments(XMLDom::sibling_iterator& node)
	{
		XMLDom::comment_iterator comment = node.comment_begin();
		while(comment != node.comment_end())
			comment = comment.remove();
		for(XMLDom::sibling_iterator it=node.begin(); it!=node.end(); ++it)
			removeComments(it);
	}

	virtual void parseNode(const XMLDom::const_sibling_iterator& node,
	                       ConfigurationLoader* loader)
	{
		if (*node == "process")
			parseProcessNode(node,loader);
	}

	void parseProcessNode(const XMLDom::const_sibling_iterator& node,
	                      ConfigurationLoader* loader)
	{
		std::string name;
		auto nameIt = node.find_attribute("name");
		if(nameIt != node.attribute_end())
			name = nameIt.value();

		ProcessSpec* p;
		bool specAlreadyExisted = false;

		boost::tie(p, specAlreadyExisted) = getProcessSpec(name);
		if(p->isStarted && p->machine != "thisprocess")
		{
			MIRA_THROW(XRuntime, "Process named " << p->name << " is already running. Currently, no new "
			                  << "config can be loaded to an already running process "
			                  << "nor parameters of the running process tag can be changed.");
		}

		auto miraIt = node.find_attribute("isMira");
		if (miraIt == node.attribute_end())
			miraIt = node.find_attribute("is_mira"); // backward compatibility

		if(miraIt != node.attribute_end()) {
			p->isMira = (miraIt.value()=="true");
			if (!p->isMira)
				p->connect = false; // for non-mira processes, 'connect' defaults to true,
				                    // but can still be set by attribute
		}

		auto legacyIt = node.find_attribute("legacy");
		if(legacyIt != node.attribute_end())
			p->legacy = (legacyIt.value()=="true");

		auto forcePTPIt = node.find_attribute("forcePTP");
		if(forcePTPIt != node.attribute_end())
			p->forcePTP = (forcePTPIt.value()=="true");

		auto portIt = node.find_attribute("port");
		if(portIt != node.attribute_end())
			p->port = portIt.value();

		auto connectIt = node.find_attribute("connect");
		if(connectIt != node.attribute_end())
			p->connect = (connectIt.value()=="true");

		auto guiIt = node.find_attribute("gui");
		if(guiIt != node.attribute_end())
			p->gui = (guiIt.value()=="true");

		auto argsIt = node.find_attribute("args");
		if(argsIt != node.attribute_end())
			p->extraargs += argsIt.value() + " ";

		auto respawnIt = node.find_attribute("respawn");
		if(respawnIt != node.attribute_end())
			p->respawn = (respawnIt.value()=="true");

		auto requiredIt = node.find_attribute("required");
		if(requiredIt != node.attribute_end())
			p->required = (requiredIt.value()=="true");

		auto shutdownIt = node.find_attribute("shutdownRecursively");
		if(shutdownIt != node.attribute_end())
			p->shutdownRecursively = (shutdownIt.value()=="true");

		auto userIt = node.find_attribute("user");
		if(userIt != node.attribute_end())
			p->user = userIt.value();

		auto machineIt = node.find_attribute("machine");
		if(machineIt != node.attribute_end()) {
			// check whether the machine attribute has changed
			if(specAlreadyExisted && p->machine!=machineIt.value()) {
				if(machineIt.value()=="thisprocess") {
					MIRA_THROW(XInvalidConfig, "A process with the same name was already declared with a different machine attribute. "
						"The special value 'thisprocess' must be specified already with the first occurrence of the declared process.");
				}
				if(!p->machine.empty()) {
					MIRA_THROW(XInvalidConfig, "A process with the same name was already declared with a different machine attribute.");
				}
			}
			p->machine=machineIt.value();
		}

		auto sshportIt = node.find_attribute("sshport");
		if(sshportIt != node.attribute_end())
			p->sshport = sshportIt.value();

		auto sshpassIt = node.find_attribute("sshpass");
		if(sshpassIt != node.attribute_end())
			p->sshpass = sshpassIt.value();

		auto saveconfigIt = node.find_attribute("save-processed-config");
		if(saveconfigIt != node.attribute_end())
			p->saveconfig = saveconfigIt.value();

		auto executableIt = node.find_attribute("executable");
		if(executableIt != node.attribute_end())
			p->executable = executableIt.value();

		auto envIt = node.find_attribute("env");
		if(envIt != node.attribute_end()) {
			p->env = Process::Environment();
			p->env->insertList(envIt.value());
		}

		// extract the content of the node, which will be the configuration
		// of the new mira process

		// copy the the process tag and all content into the config of the process
		XMLDom::sibling_iterator newnode = p->config.root().add_child(node);
		newnode = "namespace"; // rename node back to "namespace"

		// remove all attributes (that come with the process tag)
		while(newnode.attribute_begin()!=newnode.attribute_end())
			newnode.remove_attribute(newnode.attribute_begin());

		// add the name of the current namespace as attribute
		newnode.add_attribute("name", loader->getContext()["namespace"]);

		// 'thisprocess' is handled as special case, which disables the
		// process tag and launches the whole content within THIS process
		if(p->machine=="thisprocess") {
			// parse the content
			loader->parse(node);
		}
		else {
			// collect the usings in the process tag
			mUsingLoader.load(p->config);
		}
	}


	virtual void startDocument(ConfigurationLoader* ioLoader)
	{
		mProcessList.clear();
	}

	virtual void endDocument(ConfigurationLoader* ioLoader)
	{
		// collect all the encountered usings
		mUsings.clear();
		foreach(auto& from_to, MIRA_FW.getNameRegistry().getAliases())
		{
			const std::string& from = from_to.first.str();
			const std::string& to = from_to.second.first.str();
			mUsings.root().add_child("using").add_attribute("name", to)
			                                 .add_attribute("as", from);
		}

		mProcessManager.startWatchdog(
				boost::bind(&ProcessLoader::requiredProcessTerminated, this,_1));

		foreach(auto& p, mProcessList)
		{
			if(!p->isStarted)
			{
				startProcess(p.get());
				p->isStarted = true;
			}
		}
	}

	void requiredProcessTerminated(const ProcessSpawnManager::ProcessInfo& info)
	{
		MIRA_LOG(NOTICE) << "Required process '" << info.getName() << "' "
		                    "was terminated. Stopping ourselves ...";
		MIRA_FW.requestTermination(-1);
	}


private:

	XMLDom mUsings;

	struct ProcessSpec : boost::noncopyable
	{
		std::string name;

		bool forcePTP;
		bool legacy;
		std::string port;
		std::string user;
		std::string machine;
		std::string sshport;
		bool connect;
		bool gui;
		std::string extraargs;
		bool respawn;
		bool required;
		bool shutdownRecursively;

		std::string sshpass;
		std::string saveconfig;

		std::string executable;
		bool isMira;
		boost::optional<Process::Environment> env;

		XMLDom config;

		bool isStarted;

		ProcessSpec() : 
			// the defaults for each process
			forcePTP(false), 
			legacy(false),
			port("1234"),
			connect(true),
			gui(false),
			respawn(false),
			required(false),
			shutdownRecursively(false),
			isMira(true),
			isStarted(false) {}
	};

	std::list<std::unique_ptr<ProcessSpec>> mProcessList;

	ConfigurationLoader mUsingLoader;

private:

	void initializeFWUsingLoader()
	{
		mUsingLoader = ConfigurationLoader(false);

		ClassProxy nsl = getLoaderPlugin("mira::NamespaceLoader");
		ClassProxy ul = getLoaderPlugin("mira::UsingLoader");

		mUsingLoader.registerLoaderPlugin(nsl);
		mUsingLoader.registerLoaderPlugin(ul);
	}

	ClassProxy getLoaderPlugin(const std::string& classID)
	{
		return ConfigurationLoaderPlugin::CLASS().getClassByIdentifier(classID);
	}

	// returns the ProcessSpec and a flag whether the spec existed before or not
	boost::tuple<ProcessSpec*, bool> getProcessSpec(const std::string& name)
	{
		if(!name.empty()) {
			// look for process with the given name in our list
			foreach(auto& p, mProcessList)
				if(p->name==name) { // already exists?
					return boost::make_tuple(p.get(), true);
				}
		}

		// process does not exist yet, or name is empty: create new spec
		ProcessSpec* p = new ProcessSpec;
		p->name = name;
		mProcessList.push_back(std::unique_ptr<ProcessSpec>(p));
		return boost::make_tuple(p, false);
	}

	void startProcess(ProcessSpec* p)
	{
		// get executable and arguments
		Path executable;
		std::vector<std::string> args;
		std::string childHost;
		std::string machine;
		bool isRemote = false;
		boost::optional<Process::Environment> processEnvironment;

		childHost = p->machine;
		// THIS PROCESS
		if(p->machine=="thisprocess") {
			// 'thisprocess' is handled as special case, the content of
			// the process tag was already handled in parseProcessNode
			return;

		// LOCAL MACHINE
		} else if(p->machine.empty() || p->machine=="localhost" || p->machine=="127.0.0.1") { // local
			if(childHost.empty()) // machine param could have been empty
				childHost = "127.0.0.1";

			if (p->executable.empty()) {
				if(p->gui) executable = getMiraguiPath();
				      else executable = getMiraPath();
			}
			else
				executable = p->executable;

			if(p->env) {
				// get current process environment
				processEnvironment = Process::Environment::systemEnvironment();
				// and add/overwrite user defined variables
				processEnvironment->insert(p->env.get());
			}

		// REMOTE MACHINE
		} else {
			isRemote = true;

			std::string sshCmd = "ssh";
			getRemoteCommand(sshCmd, args, p->sshpass);
			executable = sshCmd;

			// this flag is necessary to make sure, that ssh sends the SIGHUP
			// signal when itself gets the SIGINT or SIGTERM
			args.push_back("-t");

			if(!p->sshport.empty()) {
				args.push_back("-p");
				args.push_back(p->sshport);
			}

			if(!p->user.empty())
				machine = p->user + "@" + p->machine;  // user@hostname
			else
				machine = p->machine;

			args.push_back(machine);

			// build the command that is sent via ssh (including environment variables and the executable)
			std::string command;

			// environment variables
			if(p->env) {
				auto envp = p->env->envp();
				foreach(auto e, envp)
					command+= e+" ";
			}

			if (p->executable.empty()) {
				if(p->gui) command += "miragui";
				      else command += "mira";
			} else
				command += p->executable;

			args.push_back(command);
		}


		// if connect==true, launch child with -p<port> parameter and add it
		// to our known frameworks
		if(p->connect) {
			args.push_back("-p" + p->port); // add -p<port> parameter for our child
			// ... and add the child as known framework
			if(MIRA_FW.getRemoteModule())
				MIRA_FW.getRemoteModule()->addKnownFramework(childHost+":"+p->port, p->forcePTP, p->legacy);
		}

		// extract specified additional parameters
		boost::trim(p->extraargs);
		std::vector<std::string> addargs;
		if(!p->extraargs.empty()) {
			boost::split(addargs, p->extraargs, boost::algorithm::is_any_of(" \t\n"),
			             boost::algorithm::token_compress_on);
		}
		std::copy(addargs.begin(),addargs.end(),std::back_inserter(args));

		// and add our config string as parameter
		if (p->isMira)
			args.push_back("--config-string");

		// now prepare the config xml ...
		// add all usings that we came across
		auto first = p->config.root().begin();
		for(auto it=mUsings.root().begin(); it!=mUsings.root().end(); ++it)
			first.insert_before(it);

		// create string from config xml
		std::string config = p->config.saveToString();

		if(!p->saveconfig.empty()) // optional file for debugging
			p->config.saveToFile(p->saveconfig);

		// when using ssh the shell is invoked which
		// needs quotes around our long config argument :(
		// add single quotes to make ssh happy
		std::size_t totalArgEnvLen = 0;
		if(isRemote)
		{
			if (p->isMira) {
				// escape quotes in config to avoid breaking the ssh command
				boost::replace_all(config, "'", "'\\\''");
				args.push_back("'"+config+"'");
			}

			totalArgEnvLen = computeCStringLength(args);
		}
		else
		{
			if (p->isMira)
				args.push_back(config);

			totalArgEnvLen = computeCStringLength(args);
			if(p->env)
				totalArgEnvLen += computeCStringLength(processEnvironment->envp());
		}

		std::string cleanupCmd;
		std::size_t argMax = Private::MaximumArgumentListSizeEstimation::instance().value();
		if(totalArgEnvLen > argMax)
		{
			MIRA_LOG(NOTICE) << "Cannot create process because of probably too long config string "
			                 << "exceeding the maximum argument list size of the OS. "
			                 << "Save config to file, and if necessary send it to the remote host.";

			//remove config string from argument list
			args.pop_back(); // value of config string
			args.pop_back(); // parameter: --config-string

			// prepare path for config file
			Path uniqueFile = getPseudoUniqueConfigName(p->name);
			Path tmpDir = getTempDirectory();
			Path hostConfigFile = tmpDir/uniqueFile;
			Path configfileParam = hostConfigFile;

			if(!isRemote)
			{
				if(!p->respawn)
					// add spawn tag to remove the config file after it has been read
					p->config.root().add_child("spawn").add_attribute("command", "rm -f " + hostConfigFile.string());
				else
					cleanupCmd = "rm -f " + hostConfigFile.string();

				p->config.saveToFile(hostConfigFile);
			}
			else
			{
				Path remoteTmpDir = getRemoteTempDirectory(machine, p->sshport, p->sshpass);
				Path remoteConfigFile = remoteTmpDir / uniqueFile;

				if(!p->respawn)
				{
					// add spawn tag to remove the config file after it has been read
					p->config.root().add_child("spawn").add_attribute("command", "rm -f " + remoteConfigFile.string());
				}
				else
				{
					std::vector<std::string> args;
					std::string sshCmd = "ssh";
					getRemoteCommand(sshCmd, args, p->sshpass);
					cleanupCmd = sshCmd + " ";
					foreach(const std::string& s, args)
					{
						cleanupCmd += s;
						cleanupCmd += " ";
					}
					cleanupCmd += machine;
					cleanupCmd += " ";

					cleanupCmd += "rm -f ";
					cleanupCmd += remoteConfigFile.string();
				}

				p->config.saveToFile(hostConfigFile);
				copyConfigFileToRemoteHost(hostConfigFile, remoteTmpDir, machine, p->sshport, p->sshpass);

				boost::system::error_code err;
				boost::filesystem::remove(hostConfigFile, err);

				configfileParam = remoteConfigFile;
			}

			//add config file to argument list
			args.push_back(configfileParam.string());
		}

		mProcessManager.startProcess(executable, args, p->name, p->respawn, p->required,
		                             p->shutdownRecursively, processEnvironment, cleanupCmd);
	}

	std::size_t computeCStringLength(const std::vector<std::string>& v)
	{
		std::size_t len = 0;
		foreach(const std::string& s, v)
			len += ( s.size() + 1 );

		return len;
	}

	Path getPseudoUniqueConfigName(const std::string& processName)
	{
		char cHostname[HOST_NAME_MAX];
		gethostname(cHostname, HOST_NAME_MAX);
		std::string hostname(cHostname);
		
		uid_t uid = getuid();
		std::string username = MakeString() << "uid" << uid;
		struct passwd* pw = getpwuid(uid);
		if(pw != NULL)
			username = std::string(pw->pw_name);
		
		pid_t pid = getpid();
		
		std::string uniqueNameModel = hostname+"-"+username+"-"+std::to_string((int64)pid)+"-"+processName+"-"+"%%%%-%%%%-%%%%-%%%%-%%%%.xml";

		Path uniqueFile;
		try
		{
			uniqueFile = boost::filesystem::unique_path(uniqueNameModel);
		}
		catch(boost::exception& e)
		{
			MIRA_THROW(XRuntime, "Cannot create pseudo unique file path for saving "
			                     "config string to file (boost exception: " << boost::diagnostic_information(e) << ").");
		}

		return uniqueFile;
	}

	void getRemoteCommand(std::string& oiRemoteCmd,
	                      std::vector<std::string>& oArgs,
	                      const std::string& sshPass)
	{
		if(sshPass.empty())
		{
			return;
		}
		else
		{
			// use sshpass for password authentication
			oArgs.push_back("-p");
			oArgs.push_back(sshPass);
			oArgs.push_back(oiRemoteCmd);
			oiRemoteCmd = "sshpass";
		}
	}

#ifdef MIRA_LINUX
	Path getRemoteTempDirectory(const std::string& machine,
	                            const std::string& sshport,
	                            const std::string& sshPass)
	{
		std::vector<std::string> args;
		std::string sshCmd = "ssh";
		getRemoteCommand(sshCmd, args, sshPass);

		args.push_back("-T"); // disable pseudo-terminal, important for clean outputs without error messages
		args.push_back("-q"); // suppress most warnings
		
		if(!sshport.empty())
		{
			args.push_back("-p");
			args.push_back(sshport);
		}

		args.push_back(machine); // host address

		args.push_back("mktemp -u"); //returns temporary file name in temporary directory without creating a file

		Process p = Process::createProcess(sshCmd.c_str(),
		                                   args,
		                                   Process::noflags,
		                                   Process::out);

		if(!p.wait(Duration::seconds(5)))
			MIRA_THROW(XInvalidParameter, "Timeout getting temporary directory on remote host for sending "
			                              "config exceeding the maximum argument list size.");

		Path remoteTmpDir;
		if( p.getExitStatus() == Process::NORMALEXIT && p.getExitCode() == 0 )
		{
			remoteTmpDir = Path(std::string(MakeString() << p.cout().rdbuf()));
			remoteTmpDir = remoteTmpDir.parent_path(); //only use temporary directory
		}
		else
		{
			MIRA_THROW(XInvalidParameter, "Cannot get temporary directory of remote host for sending "
			                              "config exceeding the maximum argument list size.");
		}

		if(remoteTmpDir.empty() || !remoteTmpDir.is_absolute())
			MIRA_THROW(XInvalidParameter, "Temporary directory of remote host is not valid: " << remoteTmpDir.c_str());

		return remoteTmpDir;
	}

	void copyConfigFileToRemoteHost(const Path& hostConfigFile,
	                                const Path& remoteTmpDir,
	                                const std::string& machine,
	                                const std::string& sshport,
	                                const std::string& sshPass)
	{
		std::string scpCmd = "scp";
		std::vector<std::string> args;
		getRemoteCommand(scpCmd, args, sshPass);

		if(!sshport.empty())
		{
			args.push_back("-P");
			args.push_back(sshport);
		}

		args.push_back(hostConfigFile.c_str());
		args.push_back(machine + ":" + remoteTmpDir.c_str());

		Process p = Process::createProcess(scpCmd, args, Process::noflags, Process::out);

		if(!p.wait(Duration::seconds(5)))
			MIRA_THROW(XInvalidParameter, "Timeout copying config file of process tag to remote host.");

		if(p.getExitCode() != 0 || p.getExitStatus() != Process::NORMALEXIT )
			MIRA_THROW(XInvalidParameter, "Cannot copy config file of process tag to remote host.");
	}
#endif

#ifdef MIRA_WINDOWS
	Path getRemoteTempDirectory(const std::string& machine,
	                            const std::string& sshPass)
	{
		MIRA_THROW(XNotImplemented, "ProcessLoader::getRemoteTempDirectory is not implemented yet.");
		return Path();
	}

	void copyConfigFileToRemoteHost(const Path& hostConfigFile,
	                                const Path& remoteTmpDir,
	                                const std::string& machine,
	                                const std::string& sshPass)
	{
		MIRA_THROW(XNotImplemented, "ProcessLoader::copyConfigFileToRemoteHost is not implemented yet.");
	}
#endif
};

///////////////////////////////////////////////////////////////////////////////
}

MIRA_CLASS_REGISTER(mira::ProcessLoader, mira::ConfigurationLoaderPlugin)

