/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Unit.C
 *    Implementation of Unit.h.
 *
 * @author Tim Langner
 * @date   2010/11/19
 */

#include <boost/format.hpp>

#include <fw/Unit.h>

#include <error/LoggingAux.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

void Unit::checkin(const std::string& ns, const std::string& name)
{
	MicroUnit::checkin(ns,name);
	mProcessTimer = createTimer(mCycleTime, mCycleTimeTolerance,
	                            &Unit::processIntern, this);
	mProcessTimer->stop();
}

void Unit::setCycleTime(Duration period)
{
	if(period!=mCycleTime)
	{
		mCycleTime = period;

		if(mProcessTimer)
			mProcessTimer->setPeriod(period);
	}
}

Duration Unit::getCycleTime() const
{
	return mCycleTime;
}

void Unit::setCycleTimeTolerance(Duration tolerance)
{
	if(tolerance!=mCycleTimeTolerance)
	{
		mCycleTimeTolerance = tolerance;

		if(mProcessTimer)
			mProcessTimer->setTolerance(tolerance);
	}
}

Duration Unit::getCycleTimeTolerance() const
{
	return mCycleTimeTolerance;
}

void Unit::process(const Timer& timer)
{
}

void Unit::needRecovery(const std::string& reason)
{
	if (mNeedRecover || !mProcessTimer)
		return;
	mProcessTimer->stop();
	mLastHeartbeat.reset();
	MicroUnit::needRecovery(reason);
}

void Unit::operational()
{
	MicroUnit::operational();
	if (mProcessTimer)
		mProcessTimer->start();
	mResetProcessTimer = true;
}

void Unit::start()
{
	MicroUnit::start();
	if (mProcessTimer)
		mProcessTimer->start();
	mResetProcessTimer = true;
}

void Unit::stop()
{
	if (mProcessTimer)
		mProcessTimer->stop();
	mLastHeartbeat.reset();
	MicroUnit::stop();
}

void Unit::processIntern(const Timer& timer)
{
	// Before the the first call of the process method, we have to reset the
	// expected time of the process timer to avoid a warning "cannot hold the
	// specified cycle time" after the first process loop.
	if (mResetProcessTimer) {
		if(mProcessTimer)
			mProcessTimer->currentExpected = Time::now();
		mResetProcessTimer = false;
	}

	try
	{
		process(timer);
	}
	catch(Exception& ex)
	{
		if (error("Process", ex.message(), "Exception in process()"))
		{
			MIRA_LOG_EXCEPTION(ERROR,ex) << "Exception:\n";
		}
		return;
	}
	catch(std::exception& ex)
	{
		if (error("Process", ex.what(), "Exception in process()"))
		{
			MIRA_LOG_EXCEPTION(ERROR,ex) << "Exception:\n";
		}
		return;
	}
	Duration dt = timer.getExceedance();
	if (dt > Duration::seconds(0))
	{
		uint32 us = dt.totalMicroseconds();
		warning("Process", MakeString() << "Cannot hold the specified cycle time!"
				" We are " << boost::format("%d.%03d") % (us/1000) % (us%1000)
				<< " ms behind.");
		return;
	}
	ok("Process");
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::Unit, mira::MicroUnit);
