/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RemoteConnection.C
 *    Implementation of RemoteConnection.h.
 *
 * @author Tim Langner
 * @date   2010/11/18
 */

#include <fw/RemoteConnection.h>

#include <boost/array.hpp>

#include <math/Saturate.h>
#include <stream/BufferStream.h>
#include <utils/StringAlgorithms.h>

#include <serialization/adapters/std/list>
#include <serialization/adapters/std/set>

#include <fw/Framework.h>
#include <fw/MicroUnit.h>
#include <fw/UnitManager.h>
#include <fw/RemoteModule.h>
#include <fw/ChannelBufferSlotFlags.h>

// max allowed size of a message is limited to 256MB, this limit is used
// only in checkMessageHeader() to prevent overflows of possible Denial
// of Service attacks.
#define MAX_MESSAGE_SIZE 256*1024*1024

namespace mira {

///////////////////////////////////////////////////////////////////////////////

void KnownFramework::validate() const
{
	getHostPort();
}

boost::tuple<std::string, uint16> KnownFramework::getHostPort() const
{
	std::size_t splitPos = address.rfind(':');
	if (splitPos == std::string::npos)
		MIRA_THROW(XInvalidParameter, "'" << address << "' is not a valid address.");
	std::string host, port;
	host = address.substr(0, splitPos);
	port = address.substr(splitPos+1, address.size() - splitPos - 1);
	return boost::make_tuple(host, fromString<uint16>(port));
}

///////////////////////////////////////////////////////////////////////////////

void TimeOffsetCompensation::setCompensationInterval(const Duration& interval)
{
	boost::mutex::scoped_lock lock(mMutex);
	mCompensationInterval = interval.totalMicroseconds();
}

Duration TimeOffsetCompensation::getCompensationInterval() const
{
	boost::mutex::scoped_lock lock(mMutex);
	return Duration::microseconds(mCompensationInterval);
}

void TimeOffsetCompensation::setTargetOffset(const Duration& target, const Time& ts)
{
	assert(target.isValid());
	assert(ts.isValid());

	boost::mutex::scoped_lock lock(mMutex);
	if (isInitialized())
		mStartOffset = offset(ts);
	else {
		// on first setTargetOffset(), set mStartOffset to target offset
		// --> the first target offset is effective immediately, no delayed approach
		mStartOffset = target;
	}

	mStartTime = ts;
	mTargetOffset = target;

	assert(mStartOffset.isValid());
	assert(mTargetOffset.isValid());
}

Duration TimeOffsetCompensation::queryOffset(const Time& ts) const
{
	boost::mutex::scoped_lock lock(mMutex);
	return offset(ts);
}

Duration TimeOffsetCompensation::offset(const Time& ts) const
{
	if (!isInitialized())
		return Duration::invalid();

	if (mStartOffset == mTargetOffset)
		return mTargetOffset;

	int us = (ts - mStartTime).totalMicroseconds();
	float f = float(saturate(us, 0, mCompensationInterval)) / mCompensationInterval;
	return mStartOffset + (mTargetOffset-mStartOffset) * f;
}

///////////////////////////////////////////////////////////////////////////////

void RemoteConnection::RPCRemoteFinishHandler::onRPCfinished(Buffer<uint8>&& answer)
{
	boost::mutex::scoped_lock lock(mConnectionMutex);
	if ( mConnection != NULL )
		mConnection->queueRPCMessage(RPC_RESPONSE_MSG, std::move(answer));
}

void RemoteConnection::RPCRemoteFinishHandler::setConnection(RemoteConnection* iConnection)
{
	boost::mutex::scoped_lock lock(mConnectionMutex);
	mConnection = iConnection;
}

///////////////////////////////////////////////////////////////////////////////

void RemoteConnection::RPCRemoteRequestHandler::onRPCrequested(Buffer<uint8>&& request)
{
	boost::mutex::scoped_lock lock(mConnectionMutex);
	if ( mConnection != NULL )
		mConnection->queueRPCMessage(RPC_REQUEST_MSG, std::move(request));
}

void RemoteConnection::RPCRemoteRequestHandler::setConnection(RemoteConnection* iConnection)
{
	boost::mutex::scoped_lock lock(mConnectionMutex);
	mConnection = iConnection;
}

///////////////////////////////////////////////////////////////////////////////

/// Constructs a remote connection that uses its own io service
RemoteConnection::RemoteConnection() :
	mSocket(static_cast<boost::asio::io_service&>(mService))
{
	init();
}

/// Constructs a remote connection that uses a given io service
RemoteConnection::RemoteConnection(boost::asio::io_service& service) :
	mService(service),
	mSocket(static_cast<boost::asio::io_service&>(mService))
{
	init();
}

RemoteConnection::~RemoteConnection()
{
	try {
		// as long as the stop mutex is locked it is not possible to delete the remote connection
		// if the thread is not stopped, we stop it here and delete the connection
		stop();
		mSendRPCMessagesThread.join();
		// ensure, that both ping threads are stopped
		mProcessPingThread.join();
		mCheckPingTimeoutThread.join();
		mSendChannelUpdatesThread.join();
	}
	catch (const boost::thread_interrupted&) {
		// suppress boost::thread_interrupted
	}
	catch (std::exception& ex) {
		MIRA_LOG_EXCEPTION(ERROR, ex) << "Exception in stop() call in destructor: \n";
	}
}

bool RemoteConnection::isLocal() const
{
	try
	{
		if (mSocket.local_endpoint().address() == mSocket.remote_endpoint().address())
			return true;
	}
	catch(...)
	{}
	return false;
}

void RemoteConnection::start()
{
	mSendRPCMessagesThread =
			boost::thread(boost::bind(&RemoteConnection::sendRPCMessagesThread,
			                          this));
	mSendChannelUpdatesThread =
			boost::thread(boost::bind(&RemoteConnection::sendChannelUpdatesThread,
			                          this));
}

void RemoteConnection::stop()
{
	boost::mutex::scoped_lock lock(mStopMutex);
	if (mStopped)
		return;

	onDisconnect();
	mStopped = true;
	// mSocket.shutdown(boost::asio::ip::tcp::socket::shutdown_both);
	mService.stop();
	mSocket.close();
	mRPCFinishHandler->setConnection(NULL);
	mRPCRequestHandler->setConnection(NULL);

	foreach (const std::string& service, publishedServices)
		MIRA_FW.getRPCManager().removeRemoteService(service);
	if (authority) {
		if (mSyncTimeTimer)
			authority->removeTimer(mSyncTimeTimer);

		authority.reset();
	}

	// remove all remote authorities from our local AuthorityManager
	foreach (const auto& a, mRemoteAuthorities)
		MIRA_FW.getAuthorityManager().removeAuthority(a.second.get());

	mRemoteAuthorities.clear();

	mSendRPCMessagesThread.interrupt();
	mProcessPingThread.interrupt();
	mCheckPingTimeoutThread.interrupt();
	mSendChannelUpdatesThread.interrupt();

	// if stop() was called from any of these threads,
	// add interruption point to ensure it interrupts now
	auto id = boost::this_thread::get_id();
	if (id == mSendRPCMessagesThread.get_id() ||
	    id == mProcessPingThread.get_id() ||
	    id == mCheckPingTimeoutThread.get_id() ||
	    id == mSendChannelUpdatesThread.get_id())
		boost::this_thread::interruption_point();
}

void RemoteConnection::onConnect(bool enablePTPTimeSync, bool enablePingTimeout)
{
	mEnablePTPSync = enablePTPTimeSync;
	mEnablePingTimeout = enablePingTimeout;

	if (isPingTimeoutEnabled()) {
		// Ensure, that the last ping time is initialized properly.
		mPingLastReceived = Time::now();

		// Start a thread, which send pings and syncTime
		mProcessPingThread =
				boost::thread(boost::bind(&RemoteConnection::processPingThread,
				              this));

		// Start a thread, which checks if the connection is alive.
		mCheckPingTimeoutThread =
				boost::thread(boost::bind(&RemoteConnection::checkPingTimeoutThread,
				              this));
	} else         // processPingThread will also call syncTime. only if ping timeout 
	{              // is disabled, a separate timer must be started for time sync
		startTimeSync();
	}
}

/**
 * PTP overview for protocol version 3.2 ('old'):
 * both sides of the connection estimate the offset and use it to correct time
 * when receiving a WRITE_CHANNEL message
 * 
 *                          Outgoing                                Incoming
 * -------------------------------------------------------------------------------------------
 * remote           1. send PTP_SYNC+PTP_FOLLOW_UP       1. send PTP_SYNC+PTP_FOLLOW_UP
 *                  2. wait for PTP_FINISH               2. wait for PTP_FINISH
 *                     (Incoming offset is initialized)     (Outgoing offset is initialized)
 *                  3. publish channels etc.             3. publish channels etc.
 * 
 * local            1. publish channels etc.             1. publish channels etc.
 * 
 * local, forcePTP  1. send PTP_SYNC+PTP_FOLLOW_UP       1a. send channels etc.
 *                  2. wait for PTP_FINISH               1b. when receiving PTP_SYNC:
 *                     (Incoming offset is initialized)      send PTP_SYNC+PTP_FOLLOW_UP
 *                  3. publish channels etc.             (2. wait for PTP_FINISH
 *                                                           (Outgoing offset is initialized))
 * 
 * PTP overview for protocol version 3.3 ('new'):
 * only Outgoing side estimates the offset and uses it to correct time both when
 * sending and receiving a WRITE_CHANNEL message
 * 
 *                          Outgoing                            Incoming
 * -------------------------------------------------------------------------------------------
 * remote           1. wait for offset initialization   1. send PTP_SYNC+PTP_FOLLOW_UP
 *                  2. publish channels etc.            2. wait for PTP_FINISH
 *                                                         (Outgoing offset is initialized)
 *                                                      3. publish channels etc.
 * 
 * local            1. publish channels etc.            1. publish channels etc.
 * 
 * local, forcePTP  1. send PTP_SYNC                    1a. publish channels etc.
 *                  2. wait for offset initialization   1b. when receiving PTP_SYNC:
 *                                                          send PTP_SYNC+PTP_FOLLOW_UP
 *                  3. publish channels etc.            (2. wait for PTP_FINISH
 *                                                          (Outgoing offset is initialized))
 *
 * Note: both in versions 3.2 and 3.3, for the forceptp case, Outgoing may receive
 * channel data from Incoming before its offset is initialized. This is not regarded a
 * critical issue.
 */
void RemoteConnection::syncTime()
{
	// Only sync time, if the frameworks are fully connected.
	if (mAuthState == AUTHSTATE_ACCEPTED)
	{
		if ( (Time::now()-mLastPTP) > Duration::seconds(30) )
		{
			mLastPTP = Time::now();

			// only do time synchronization if connection is to another host or forced

			if (mPTPOutgoing) { // we are the outgoing side
				if (isLocal()) {
					if (address.forcePTP) {
						if (remoteVersion >= 0x00030003) // will trigger PTP on incoming side (the incoming 
							writeMessage(PTP_SYNC_MSG);  // side will not start it itself on a local connection)
						else
							sendPTP(); 
					} else {
						clockOffset.setTargetOffset(Duration::seconds(0), Time::unixEpoch());
						synchronizeFrameworks();
					}
				} else if (remoteVersion < 0x00030003) { // connection to remote
					sendPTP();
				}
			} else { // we are the incoming side
				if (isLocal()) {
					clockOffset.setTargetOffset(Duration::seconds(0), Time::unixEpoch());
					synchronizeFrameworks();
				} else {
					if (remoteVersion >= 0x00030003) {
						// only the outgoing connection estimates and applies an actual offset
						clockOffset.setTargetOffset(Duration::seconds(0), Time::unixEpoch());
					}
					sendPTP(); // initiate offset estimation on outgoing side
				}
			}
		}
	}
}

void RemoteConnection::sendPTP()
{
	// Send ptp sync packet
	writeMessage(PTP_SYNC_MSG);
	// get the time when the packet was sent
	uint64 timestamp = Time::now().toUnixNS();
	// Send ptp follow up packet containing the time stamp when the sync packet was sent
	writeMessageFromData(PTP_FOLLOW_UP_MSG, timestamp);
}

void RemoteConnection::startTimeSync()
{
	if (mSyncTimeTimer)
		authority->removeTimer(mSyncTimeTimer);

	mSyncTimeTimer = authority->createTimer(Duration::seconds(1), boost::bind(&RemoteConnection::syncTime, this));
}

void RemoteConnection::ping()
{
	if ( (Time::now()-mPingLastSend) > MIRA_FW.getRemoteModule()->getPingInterval() )
	{
		mPingLastSend = Time::now();
		if (remoteVersion >= 0x00030001)
			writeMessage(PING_MSG);
	}
}

bool RemoteConnection::hasPingTimeout() const
{
	// PING_MSG was added in protocol version 0x00030001
	if (remoteVersion < 0x00030001)
		return false;

	return (Time::now()-mPingLastReceived) > MIRA_FW.getRemoteModule()->getPingTimeout();
}

void RemoteConnection::unpublishChannel(const std::string& channel)
{
	MIRA_LOG(DEBUG) << "Framework: Unpublishing channel to remote framework ID='"
		<< frameworkID << "'";
	writeMessageFromBuffer(UNPUBLISH_CHANNEL_MSG, channel);
}

void RemoteConnection::migrateUnit(const std::string& id)
{
	writeMessageFromBuffer(REQUEST_MIGRATION_MSG, id);
}

bool RemoteConnection::hasAuthority(const std::string& id) const
{
	if (mRemoteAuthorities.find(id) != mRemoteAuthorities.end())
		return true;
	return false;
}

void RemoteConnection::queueRPCMessage(FrameworkMessageType msg,
                                       Buffer<uint8>&& data)
{
	assert((msg == RPC_REQUEST_MSG) || (msg == RPC_RESPONSE_MSG));

	boost::mutex::scoped_lock lock(mRPCMessagesMutex);
	mOutgoingRPCMessages.emplace_back(msg, std::move(data));

	mRPCMessagesCondition.notify_one();
}

void RemoteConnection::valueChanged(ChannelRead<void> value, ServiceLevel& serviceLevel)
{
	// if data is coming from remote, don't send it again (avoid loops)
	if(value.getFlags() & SlotFlags::FROM_REMOTE)
		return;

	if (serviceLevel.interval.is_negative()) {
		sendData(value, serviceLevel);
		return;
	}

	boost::mutex::scoped_lock lock(mChannelUpdatesMutex);
	Time now = Time::now();
	Time& lastSent = subscriptions[value.getChannelID()].lastData;
	if (!lastSent.isValid() || (now - lastSent >= serviceLevel.interval)) {
		sendData(value, serviceLevel);
		mPendingChannelUpdates.erase(value.getChannelID());
		lastSent = now;
	} else {
		mPendingChannelUpdates[value.getChannelID()] = serviceLevel;
	}
}

void RemoteConnection::parseMessage()
{
	MIRA_FW.getRemoteModule()->updateIncomingStats(mHeader.messageSize);
	switch((FrameworkMessageType)mHeader.messageType)
	{
		case PTP_SYNC_MSG:
		{
			mPTPSyncLocal = mHeaderReceived;

			// if we are the incoming side, a received PTP_SYNC tells us to initiate PTP Sync,
			// unless we do it anyway in syncTime() (if the connection is to a remote address)
			if (!mPTPOutgoing && isLocal())
				sendPTP();

			break;
		}
		case PTP_FOLLOW_UP_MSG:
		{
			receivedPTPFollowUp(*reinterpret_cast<uint64*>(mMessage.data()));
			break;
		}
		case PTP_DELAY_RESPONSE_MSG:
		{
			receivedPTPDelayResponse(*reinterpret_cast<uint64*>(mMessage.data()));
			break;
		}
		case PTP_DELAY_REQUEST_MSG:
		{
			receivedPTPDelayRequest(mHeaderReceived.toUnixNS());
			break;
		}
		case PTP_FINISH_MSG:
		{
			receivedPTPFinish();
			break;
		}
		// The remote framework notifies us about a published channel
		case PUBLISH_CHANNEL_MSG:
		{
			receivedPublishChannelMsg();
			break;
		}
		case UNPUBLISH_CHANNEL_MSG:
		{
			receivedUnpublishChannelMsg();
			break;
		}
		// The remote framework notifies us about a published authority
		case PUBLISH_AUTHORITY_MSG:
		{
			receivedPublishAuthorityMsg();
			break;
		}
		// The remote framework notifies us about an authority that gets unpublished
		case UNPUBLISH_AUTHORITY_MSG:
		{
			receivedUnpublishAuthorityMsg();
			break;
		}
		case REQUEST_MIGRATION_MSG:
		{
			receivedRequestMigrationMsg();
			break;
		}
		case MIGRATION_MSG:
		{
			receivedMigrationMsg();
			break;
		}
		case MIGRATION_SINK_SUCCESS_MSG:
		{
			receivedMigrationSinkSuccessMsg();
			break;
		}
		case MIGRATION_SINK_FAILURE_MSG:
		{
			receivedMigrationSinkFailureMsg();
			break;
		}
		case MIGRATION_FINISHED_MSG:
		{
			receivedMigrationFinishedMsg();
			break;
		}
		// The remote framework notifies us about a published service
		case PUBLISH_SERVICE_MSG:
		{
			receivedPublishServiceMsg();
			break;
		}
		// The remote framework notifies us about a service that gets unpublished
		case UNPUBLISH_SERVICE_MSG:
		{
			receivedUnpublishServiceMsg();
			break;
		}
		// The remote framework subscribes on a channel
		case SUBSCRIBE_CHANNEL_MSG:
		{
			receivedSubscribeChannelRequest();
			break;
		}
		// The remote framework unsubscribes from a channel
		case UNSUBSCRIBE_CHANNEL_MSG:
		{
			uint8* ptr = &mMessage[0];
			std::string channelID(ptr, ptr + mMessage.size());
			MIRA_LOG(DEBUG) << "Framework: Remote framework ID='"
				<< frameworkID << "' unsubscribes from channel='" << channelID << "'";
			receivedUnsubscribeChannelRequest(channelID);
			break;
		}
		// Data of a channel we subscribed on has changed
		case WRITE_CHANNEL_MSG:
		{
			receivedWriteChannelMsg();
			break;
		}
		case RPC_REQUEST_MSG:
		{
			receivedRPCRequestMsg();
			break;
		}
		case RPC_RESPONSE_MSG:
		{
			receivedRPCResponseMsg();
			break;
		}
		case TYPE_METADB_MSG:
		{
			receivedTypeMetaMsg();
			break;
		}
		case CHANNEL_META_MSG:
		{
			receivedChannelMetaMsg();
			break;
		}
		case PING_MSG:
		{
			receivedPingMsg();
			break;
		}
		default:
			MIRA_LOG(WARNING) << "Framework: Unknown message type from remote framework ID='"
				<< frameworkID << "' type=" << (int)mHeader.messageType;
			break;
	}
}

void RemoteConnection::init()
{
	remoteID = boost::uuids::nil_uuid();
	remoteVersion = 0;
	authority = NULL;
	mLastPTP = Time::unixEpoch();
	mPingLastSend = Time::unixEpoch();
	mPingLastReceived = Time::unixEpoch();
	mRPCFinishHandler.reset(new RPCRemoteFinishHandler(this));
	mRPCRequestHandler.reset(new RPCRemoteRequestHandler(this));
	mAuthState = AUTHSTATE_NONE;
	mStopped = false;
}

void RemoteConnection::receivedPTPFollowUp(uint64 timestamp)
{
	mPTPSyncRemote = Time::fromUnixNS(timestamp);
	writeMessage(PTP_DELAY_REQUEST_MSG);
	mPTPDelayLocal = Time::now();
}


void RemoteConnection::receivedPTPDelayRequest(uint64 timestamp)
{
	// Send ptp delay response packet containing the time stamp
	// when delay request packet was received
	writeMessageFromData(PTP_DELAY_RESPONSE_MSG, timestamp);
}

void RemoteConnection::receivedPTPDelayResponse(uint64 timestamp)
{
	mPTPDelayRemote = Time::fromUnixNS(timestamp);

	// adapt clockOffset ONLY if PTP sync is enabled!
	if(isPTPSyncEnabled()) {

		Duration offset = (mPTPSyncLocal - mPTPSyncRemote - (mPTPDelayRemote - mPTPDelayLocal))/2;

		MIRA_LOG(DEBUG) << "Framework: measured delay to remote framework ID='"
		                << frameworkID << "' is " << offset;

		clockOffset.setTargetOffset(offset);
	}

	write(FrameworkMessageHeader(0, PTP_FINISH_MSG).asBuffer());

	// In protocol version 3.3, only the outgoing side will receive this (but not PTP_FINISH)
	// (in protocol version < 3.3, both sides wait for PTP_FINISH)
	if (remoteVersion >= 0x00030003)
		synchronizeFrameworks();
}

void RemoteConnection::receivedPTPFinish()
{
	// the opposing side has now initialised its offset estimation
	// in protocol version 3.3, only the incoming side will receive this
	// (both sides in protocol version < 3.3)
	synchronizeFrameworks();
	mLastPTP = Time::now();
}

void RemoteConnection::receivedSubscribeChannelRequest()
{
	BinaryBufferDeserializer bd(&mMessage);
	ServiceLevel serviceLevel;
	bd.deserialize(serviceLevel);
	std::string resolvedID = authority->resolveName(serviceLevel.channelID);
	MIRA_LOG(DEBUG) << "Framework: Got subscription from remote framework ID='"
			<< frameworkID << "' for channel='" << resolvedID << "'";
	if ( subscriptions.count(resolvedID) == 0 )
	{
		subscriptions[resolvedID].metaVersion = 0;
		subscriptions[resolvedID].lastData = Time::invalid();

		authority->subscribe<void>(resolvedID,
		                           boost::bind(&RemoteConnection::valueChanged,
		                                       this, _1, serviceLevel));

		try
		{
			// send last data that is contained in the channel so that the connected
			// framework has the newest data
			Channel<void> channel = authority->getChannel<void>(resolvedID);
			ChannelRead<void> value = channel.read();
			Time now = Time::now();
			sendData(value, serviceLevel);
			subscriptions[resolvedID].lastData = now;
		}
		catch (XInvalidRead&)
		{
			// No data in the channel so we will wait for the next
			// channel callback.
		}
	}
}

void RemoteConnection::receivedUnsubscribeChannelRequest(const std::string& channelID)
{
	std::string resolvedID = authority->resolveName(channelID);
	if ( subscriptions.count(resolvedID) != 0 )
	{
		subscriptions.erase(resolvedID);
		boost::mutex::scoped_lock lock(mChannelUpdatesMutex);
		mPendingChannelUpdates.erase(resolvedID);
		authority->unsubscribe<void>(resolvedID);
	}
}

void RemoteConnection::receivedPublishChannelMsg()
{
	BinaryBufferDeserializer bd(&mMessage);
	std::map<std::string, Typename> channels;
	bd.deserialize(channels);

	foreach(const auto& channel, channels)
	{
		// If we have not known about the fact, that the remote framework has
		// a publisher for this channel test if we need to subscribe
		std::string channelID = MIRA_FW.getNameRegistry().resolve(channel.first,
		                                                         authority->getNamespace());
		if ( !MIRA_FW.getChannelManager().hasPublished(authority->getGlobalID(), channelID) )
		{
			MIRA_LOG(DEBUG) << "Framework: Got notification from remote framework ID='"
					<< frameworkID << "' about a published channel='" << channel.first
					<< "'" << " type='" << channel.second << "'";

			// tell framework that we act as a publisher of this channel
			// (however, we will not subscribe at the other framework
			// unless we have a subscriber for it)
			try {
				authority->publish<void>(channel.first, channel.second);
			} catch (XBadCast& ex) {
				MIRA_RETHROW(ex, "Remote channel " << channel.first << " already exists "
				                 "in the local framework with a different type.");
			}

			if ( MIRA_FW.getChannelManager().hasSubscriber(channel.first) )
			{
				ServiceLevel serviceLevel = MIRA_FW.getRemoteModule()->getServiceLevel(channel.first,
				                                                                       channel.second);
				subscribeChannel(channel.first, serviceLevel);
			}
		}
	}
}

void RemoteConnection::receivedUnpublishChannelMsg()
{
	std::string channelID((char*)mMessage.data(), mMessage.size());
	// if we had known about the fact, that the remote framework had
	// published this channel before we can unsubscribe that channel safely
	// from that framework since there will be no more data updates.
	if ( MIRA_FW.getChannelManager().hasPublished(authority->getGlobalID(), channelID) )
	{
		writeMessageFromBuffer(UNSUBSCRIBE_CHANNEL_MSG, channelID);
		// we also no longer are a publisher of this channel
		authority->unpublish(channelID);
	}	
}

void RemoteConnection::receivedPublishAuthorityMsg()
{
	BinaryBufferDeserializer bd(&mMessage);
	AuthorityDescriptions authorities;
	bd.deserialize(authorities);

	// add authorities as RemoteAuthorities to our local AuthorityManager
	foreach(const AuthorityDescription& a, authorities)
	{
		// already published by this connection -> ignore
		if (mRemoteAuthorities.count(a.getGlobalID()) > 0)
			continue;
		RemoteAuthority* ra = new RemoteAuthority(a);
		try
		{
			MIRA_FW.getAuthorityManager().addAuthority(ra);
		}
		catch(XInvalidParameter& ex)
		{
			delete ra;
			MIRA_LOG(WARNING) << ex.message();
			continue;
		}
		mRemoteAuthorities[ra->getGlobalID()].reset(ra);
	}
}

void RemoteConnection::receivedUnpublishAuthorityMsg()
{
	BinaryBufferDeserializer bd(&mMessage);
	AuthorityDescriptions authorities;
	bd.deserialize(authorities);

	// remove authorities from our local AuthorityManager
	foreach(const AuthorityDescription& a, authorities)
	{
		auto it = mRemoteAuthorities.find(a.getGlobalID());
		if(it==mRemoteAuthorities.end())
			continue;

		MIRA_FW.getAuthorityManager().removeAuthority(it->second.get());
		mRemoteAuthorities.erase(it);
	}
}

void RemoteConnection::receivedUnpublishServiceMsg()
{
	BinaryBufferDeserializer bd(&mMessage);
	StringSet services;
	bd.deserialize(services);

	foreach(const std::string& service, services)
	{
		publishedServices.erase(service);
		MIRA_LOG(DEBUG) << "Framework: Got notification from remote framework ID='"
			<< frameworkID << "' about a service='" << service 
			<< "' that gets unpublished";
		MIRA_FW.getRPCManager().removeRemoteService(service);
	}
}

void RemoteConnection::receivedWriteChannelMsg()
{
	BinaryBufferDeserializer bd(&mMessage);
	std::string channelID;
	StampedHeader sHeader;
	uint32 dataSize;
	bd.deserialize(channelID);
	bd.deserialize(sHeader);
	bd.deserialize(dataSize);
	Buffer<uint8> data(dataSize);
	int offset = addBinaryFormatVersion(data);
	try {
		boost::asio::read(mSocket, boost::asio::buffer(data.data()+offset, dataSize));
	} catch(boost::system::system_error& e) {
		if(isPingTimeoutEnabled()) {
			MIRA_THROW(XIO, "Error while reading channel message from socket of RemoteConnection: " << e.what())
		} else {
			MIRA_LOG(ERROR) << "Error while reading from socket of RemoteConnection: " << e.what();
			stop();
			return; // abort
		}
	}
	MIRA_FW.getRemoteModule()->updateIncomingStats(dataSize);
	Channel<void> channel = authority->getChannel<void>(channelID);
	ChannelWrite<void> d = channel.write();

	try
	{
		if(isPTPSyncEnabled()) {
			// Correct the timestamp by the latest clock offset
			if (clockOffset.isInitialized()) {
				Duration offset = clockOffset.queryOffset();
				MIRA_LOG(TRACE) << "Channel " << channelID << " write message: using smoothed offset " << offset << " for timestamp";
				d->timestamp = sHeader.timestamp + offset;
			} else {
				MIRA_LOG(WARNING) << "Channel " << channelID
				                  << " write message: time not synchronized to remote framework yet, not adding any offset";
				d->timestamp = sHeader.timestamp;
			}
		} else
			d->timestamp = sHeader.timestamp;

		d->frameID = sHeader.frameID;
		d->sequenceID = sHeader.sequenceID;
		d.writeSerializedValue(std::move(data));

		d.addFlags(SlotFlags::FROM_REMOTE); // mark that the data is coming from remote
	}
	catch(Exception& ex)
	{
		MIRA_LOG(ERROR) << "Exception while receiving channel write message from remote framework (channel=" << channelID << "): " << ex.message();
		d.discard();
	}
}

void RemoteConnection::receivedRequestMigrationMsg()
{
	std::string id((char*)mMessage.data(), mMessage.size());
	XMLDom xml;
	MIRA_LOG(DEBUG) << "Received migration request for unit '" << id << "'";
	mMigrationUnit = MIRA_FW.getUnitManager()->getUnit(id);
	if (!mMigrationUnit)
		return; // TODO handle this case e.g. send a error response
	mMigrationUnit->stop();
	XMLDom::iterator root = xml.root();
	root = "unit";
	root.add_attribute("id", mMigrationUnit->getID());
	root.add_attribute("ns", mMigrationUnit->getNamespace());
	XMLSerializer s(root);
	s.serialize("instance", mMigrationUnit);
	std::string xmlData = xml.saveToString();
	writeMessageFromBuffer(MIGRATION_MSG, xmlData);
}

void RemoteConnection::receivedMigrationMsg()
{
	try
	{
		std::string xmlData((char*)mMessage.data(), mMessage.size());
		XMLDom xml;
		xml.loadFromString(xmlData);
		mMigrationID = xml.root().get_attribute<std::string>("id");
		mMigrationNS = xml.root().get_attribute<std::string>("ns");
		XMLDeserializer d(xml.root());
		d.deserialize("instance", mMigrationUnit);
		write(FrameworkMessageHeader(0, MIGRATION_SINK_SUCCESS_MSG).asBuffer());
	}
	catch(Exception& ex)
	{
		MIRA_LOG(ERROR) << "Migration failed: " << ex.message();
		// migration failed
		mMigrationUnit.reset();
		write(FrameworkMessageHeader(0, MIGRATION_SINK_FAILURE_MSG).asBuffer());
	}
}

void RemoteConnection::receivedMigrationSinkSuccessMsg()
{
	if (!mMigrationUnit)
		return;
	// unit could be migrated/restored in remote framework
	// so we can remove it from our framework
	std::string id = mMigrationUnit->getGlobalID();
	mMigrationUnit.reset();
	MIRA_FW.getUnitManager()->removeUnit(id);
	write(FrameworkMessageHeader(0, MIGRATION_FINISHED_MSG).asBuffer());
}

void RemoteConnection::receivedMigrationSinkFailureMsg()
{
	if (!mMigrationUnit)
		return;
	// migration failed so start unit again
	mMigrationUnit->start();
	mMigrationUnit.reset();
}

void RemoteConnection::receivedMigrationFinishedMsg()
{
	if (!mMigrationUnit)
		return;
	// remove authority from manager and list if unpublishAuthorityMsg was
	// not already received
	ResourceName ns(mMigrationNS);
	std::string id = ns / mMigrationID;
	auto it = mRemoteAuthorities.find(id);
	if(it!=mRemoteAuthorities.end())
	{
		MIRA_FW.getAuthorityManager().removeAuthority(it->second.get());
		mRemoteAuthorities.erase(it);
	}
	// checkin the migrated unit and start it
	mMigrationUnit->checkin(mMigrationNS, mMigrationID);
	mMigrationUnit->start();
	MIRA_FW.getUnitManager()->addUnit(mMigrationUnit, mMigrationNS, mMigrationID);
	mMigrationUnit.reset();
}

void RemoteConnection::receivedTypeMetaMsg()
{
	BinaryBufferDeserializer bd(&mMessage);
	MetaTypeDatabase db;
	bd.deserialize(db);
	MIRA_FW.getMetaDatabase()->merge(db);
}

void RemoteConnection::receivedChannelMetaMsg()
{
	BinaryBufferDeserializer bd(&mMessage);
	std::string channelName;
	TypeMetaPtr meta;
	bd.deserialize(channelName);

	Channel<void> channel = authority->getChannel<void>(channelName);

	// if the channel is typed, the correct meta will be (or has already been) created when writing to it!
	if (channel.isTyped())
		return;

	bd.deserialize(meta);

	if (channel.getTypeMeta()) {
		std::string oldMeta = channel.getTypeMeta()->toString();
		std::string newMeta = meta->toString();
		if (oldMeta != newMeta) {
			MIRA_LOG(WARNING) << "RemoteConnection: Changing type meta for untyped channel " << channelName
			                  << " from " << oldMeta << " to " << newMeta << std::endl;
		}
	}

	channel.setTypeMeta(meta);
}

void RemoteConnection::receivedPingMsg()
{
	mPingLastReceived = Time::now();
}

void RemoteConnection::synchronizeFrameworks()
{
	if (isSynchronized())
		return;

	publishChannels(MIRA_FW.getChannelManager().getPublishedChannels());
	publishServices(MIRA_FW.getRPCManager().getLocalServices());

	auto l = MIRA_FW.getAuthorityManager().getAuthorities();
	AuthorityDescriptions d;
	foreach(const std::string& a, l)
	{
		try
		{
			// only publish local authorities
			if (MIRA_FW.getAuthorityManager().isLocal(a))
				d.push_back(MIRA_FW.getAuthorityManager().getDescription(a));
		} catch(...)
		{}
	}

	publishAuthorities(d);
	synchronizedTime = Time::now();
}

void RemoteConnection::updateOutgoingStats(std::size_t size)
{
	MIRA_FW.getRemoteModule()->updateOutgoingStats(size);
}

bool RemoteConnection::checkMessageHeader() const
{
	return (mHeader.messageSize < MAX_MESSAGE_SIZE) &&
	       (mHeader.messageType < MAX_MSG_COUNT);
}

void RemoteConnection::sendConnectDenied(const std::string& msg)
{
	writeMessage(CONNECT_DENIED_MSG, msg);
}

void RemoteConnection::sendRPCMessagesThread()
{
	while(!mSendRPCMessagesThread.interruption_requested()) {
		// do nothing if the connection is stopped
		if (mStopped) {
			MIRA_SLEEP(10);
			continue;
		}

		boost::mutex::scoped_lock lock(mRPCMessagesMutex);
		// if there is no data, wait for the next RPC message to send
		if (mOutgoingRPCMessages.empty()) {
			mRPCMessagesCondition.wait(lock);
		}

		while (!mStopped &&
		       !mOutgoingRPCMessages.empty() &&
		       !mSendRPCMessagesThread.interruption_requested()) {

			RPCMessage& front = mOutgoingRPCMessages.front();
			Buffer<uint8> buf(std::move(front.second));
			FrameworkMessageType msg = front.first;
			mOutgoingRPCMessages.pop_front();

			lock.unlock();
			writeMessageFromBuffer(msg, buf);
			lock.lock();
		}
	}
}

void RemoteConnection::processPingThread()
{
	while(!mProcessPingThread.interruption_requested()) {
		syncTime();

		// Only sending ping message if the frameworks are synchronized.
		if (isSynchronized())
			ping();

		MIRA_SLEEP(10); //10ms
	}
}

void RemoteConnection::checkPingTimeoutThread()
{
	while(!mCheckPingTimeoutThread.interruption_requested()) {
		MIRA_SLEEP(1000);
		if (hasPingTimeout()) {
			MIRA_LOG(ERROR) << "Disconnect framework " << frameworkID
			                << " (from " << address.address << ") after "
			                << "ping timeout.";
			stop();
		}
	}
}

void RemoteConnection::sendChannelUpdatesThread()
{
	std::unordered_map<std::string, ServiceLevel> stillPending;

	while(!mSendChannelUpdatesThread.interruption_requested()) {
		MIRA_SLEEP(10);

		// do nothing if the connection is stopped
		if (mStopped)
			continue;

		boost::mutex::scoped_lock lock(mChannelUpdatesMutex);

		foreach(auto& entry, mPendingChannelUpdates) {
			std::string id = entry.first;
			ServiceLevel& serviceLevel = entry.second;
			Time now = Time::now();
			Time& lastSent = subscriptions[id].lastData;
			if (now - lastSent >= serviceLevel.interval) {
				ChannelRead<void> value = MIRA_FW.getChannelManager().getConcreteChannel<void>(id)->read();
				sendData(value, serviceLevel);
				lastSent = now;
			} else
				stillPending[id] = serviceLevel; // instead of erasing the ones that are done (messing up iteration),
				                                 // keep only those that are still pending in the new list
		}
		mPendingChannelUpdates.clear();
		std::swap(mPendingChannelUpdates, stillPending);
	}
}

///////////////////////////////////////////////////////////////////////////////

template <uint8 BinaryFormatVersion>
void ConcreteRemoteConnection<BinaryFormatVersion>::publishChannels(const ChannelTypeMap& channels)
{
	Buffer<uint8> buffer;
	auto bs = createBinarySerializer(&buffer);
	bs->serialize(channels);
	MIRA_LOG(DEBUG) << "Framework: Publishing channels to remote framework ID='" << frameworkID << "'";
	writeMessageFromBuffer(PUBLISH_CHANNEL_MSG, buffer);
}

template <uint8 BinaryFormatVersion>
void ConcreteRemoteConnection<BinaryFormatVersion>::subscribeChannel(const std::string& channelID,
                                                                     const ServiceLevel& serviceLevel)
{
	Buffer<uint8> buffer;
	auto bs = createBinarySerializer(&buffer);

	if (remoteVersion < 0x00030003) {
		int ms = serviceLevel.interval.totalMilliseconds();
		if (ms >= 0) {
			MIRA_LOG(WARNING) << "RemoteConnection::subscribeChannel: Remote framework "
			                     "does not support limiting channel update rate. "
			                     "Ignoring interval " << ms << "ms for channel " << channelID << ".";
		}
		bs->desireClassVersions(serialization::ClassVersionMap{{"mira::ServiceLevelBase", 0}});
	}

	bs->serialize(serviceLevel);

	writeMessageFromBuffer(SUBSCRIBE_CHANNEL_MSG, buffer);
}

template <uint8 BinaryFormatVersion>
void ConcreteRemoteConnection<BinaryFormatVersion>::publishAuthorities(const AuthorityDescriptions& authorities)
{
	Buffer<uint8> buffer;
	auto bs = createBinarySerializer(&buffer);
	bs->serialize(authorities);
	MIRA_LOG(DEBUG) << "Framework: Publishing authorities to remote framework ID='" << frameworkID << "'";
	writeMessageFromBuffer(PUBLISH_AUTHORITY_MSG, buffer);
}

template <uint8 BinaryFormatVersion>
void ConcreteRemoteConnection<BinaryFormatVersion>::unpublishAuthorities(const AuthorityDescriptions& authorities)
{
	Buffer<uint8> buffer;
	auto bs = createBinarySerializer(&buffer);
	bs->serialize(authorities);
	MIRA_LOG(DEBUG) << "Framework: Unpublishing authorities to remote framework ID='" << frameworkID << "'";
	writeMessageFromBuffer(UNPUBLISH_AUTHORITY_MSG, buffer);
}

template <uint8 BinaryFormatVersion>
void ConcreteRemoteConnection<BinaryFormatVersion>::publishServices(const StringSet& services)
{
	Buffer<uint8> buffer;
	auto bs = createBinarySerializer(&buffer);

	bs->serialize((uint32)services.size());
	foreach(const std::string& s, services)
	{
		const RPCServer::Service& service = MIRA_FW.getRPCManager().getLocalService(s);
		bs->serialize(service,false);
	}

	MIRA_LOG(DEBUG) << "Framework: Publishing services to remote framework ID='" << frameworkID << "'";
	writeMessageFromBuffer(PUBLISH_SERVICE_MSG, buffer);
}

template <uint8 BinaryFormatVersion>
void ConcreteRemoteConnection<BinaryFormatVersion>::unpublishServices(const StringSet& services)
{
	Buffer<uint8> buffer;
	auto bs = createBinarySerializer(&buffer);
	bs->serialize(services);
	MIRA_LOG(DEBUG) << "Framework: Unpublishing services to remote framework ID='" << frameworkID << "'";
	writeMessageFromBuffer(UNPUBLISH_SERVICE_MSG, buffer);
}

template <uint8 BinaryFormatVersion>
void ConcreteRemoteConnection<BinaryFormatVersion>::sendData(ChannelRead<void> value, ServiceLevel& serviceLevel)
{
	Channel<void> channel = authority->getChannel<void>(value.getChannelID());
	TypeMetaPtr meta = channel.getTypeMeta();
	// check if there is any meta information valid for the receiver and...
	if (meta && meta->version() <= BinaryFormatVersion)
	{
		// if meta information for this channel changed or was never sent yet...
		uint8& sentVersion = subscriptions[value.getChannelID()].metaVersion;
		if (sentVersion != meta->version())
		{
			// check if we already have sent detailed information about
			// this meta type (if other channels have the same type we
			// will only send detailed information once)
			if (sentMetaInformation.count(meta->getTypename()) == 0)
			{
				try
				{
					Buffer<uint8> buffer;
					auto bs = createBinarySerializer(&buffer);
					// create meta database for all types needed by channel type
					MetaTypeDatabase db = MIRA_FW.getMetaDatabase()->getDependentTypesDB(meta);
					bs->serialize(db);
					writeMessageFromBuffer(TYPE_METADB_MSG, buffer);
					sentMetaInformation.insert(meta->getTypename());
				}
				catch(Exception& ex)
				{
					MIRA_LOG_EXCEPTION(WARNING,ex) << "Exception:\n";
				}
			}
			// now send channel type meta information
			Buffer<uint8> buffer;
			auto bs = createBinarySerializer(&buffer);
			bs->serialize(value.getChannelID());
			bs->serialize(meta);
			writeMessageFromBuffer(CHANNEL_META_MSG, buffer);
			sentVersion = meta->version();
		}
	}
	Buffer<uint8> dataHeader;
	try {
		auto bs = createBinarySerializer(&dataHeader);
		const Buffer<uint8>& data = value.readSerializedValue(serviceLevel.codecs, BinaryFormatVersion, true);
		bs->serialize(value.getChannelID());
		if (isPTPSyncEnabled() && (remoteVersion >= 0x00030003)) {
			// Correct the timestamp by the latest clock offset.
			if (clockOffset.isInitialized()) {
				Duration offset = clockOffset.queryOffset();
				MIRA_LOG(TRACE) << "Channel " << value.getChannelID() << " send data: using smoothed offset " << offset << " for timestamp";
				bs->serialize(StampedHeader(value->timestamp - offset, value->frameID, value->sequenceID));
			} else {
				MIRA_LOG(WARNING) << "Channel " << value.getChannelID()
					              << " send data: time not synchronized to remote framework yet, not adding any offset";
				bs->serialize((StampedHeader)(*value));
			}
		} else
			bs->serialize((StampedHeader)(*value));
		
		bs->serialize((uint32)data.size());

		FrameworkMessageHeader header(dataHeader.size(), WRITE_CHANNEL_MSG);
		boost::array<boost::asio::const_buffer, 3> buffers =
			{{
				header.asBuffer(),
				boost::asio::buffer(dataHeader.data(), dataHeader.size()),
				boost::asio::buffer(data.data(), data.size())
			}};
		write(buffers);
	}
	catch (XIO& ex) {
		MIRA_LOG(WARNING) << "Cannot send data for channel " << value.getChannelID() << ": " << ex.what();
	}
}

template<>
int ConcreteRemoteConnection<0>::addBinaryFormatVersion(Buffer<uint8>& data)
{
	// convert to v1 (by marking as v1 ;) )
	data.resize(data.size()+3);
	data[0] = BINARY_VERSION_MARKER % 256;
	data[1] = BINARY_VERSION_MARKER / 256;
	data[2] = 1;

	return 3; // offset for data
}

template<>
int ConcreteRemoteConnection<2>::addBinaryFormatVersion(Buffer<uint8>& data)
{
	// nothing to do (it is already there), leave data unchanged
	return 0;
}

template <uint8 BinaryFormatVersion>
void ConcreteRemoteConnection<BinaryFormatVersion>::receivedPublishServiceMsg()
{
	// here we use a specific format version deserializer, because we deserialize data that
	// is NOT prefixed with the typename (-> risk to fail recognition of legacy format)
	ConcreteBinaryDeserializer<BinaryBufferIstream, BinaryFormatVersion> bd(&mMessage);

	uint32 size;
	bd.deserialize(size);
	for(uint32 i=0; i<size; ++i)
	{
		RPCManager::RemoteService serviceInfo;
		bd.deserialize(serviceInfo,false);

		publishedServices.insert(serviceInfo.name);
		MIRA_LOG(DEBUG) << "Framework: Got notification from remote framework ID='" << frameworkID
		                << "' about a published service='" << serviceInfo.name << "'.";

		MIRA_FW.getRPCManager().addRemoteService(serviceInfo, mRPCRequestHandler, BinaryFormatVersion);
	}
}

template <uint8 BinaryFormatVersion>
void ConcreteRemoteConnection<BinaryFormatVersion>::receivedRPCRequestMsg()
{
	MIRA_FW.getRPCManager().handleRemoteRequest(mMessage, mRPCFinishHandler, BinaryFormatVersion);
}

template <uint8 BinaryFormatVersion>
void ConcreteRemoteConnection<BinaryFormatVersion>::receivedRPCResponseMsg()
{
	MIRA_FW.getRPCManager().handleRemoteResponse(mMessage, BinaryFormatVersion);
}

/**
 * just instantiate each required template specialization once,
 * otherwise no instances are created at compile time, resulting in a linker error
 * (the concrete instances (template parameters) actually created for the remote connections
 * are determined at runtime, this is only possible because the set of valid parameters
 * is restricted and we make sure to create each possible specialization at compile time).
 */
void createConcreteRemoteConnectionInstances()
{
	ConcreteRemoteConnection<0> rc0;
	ConcreteRemoteConnection<2> rc2;
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::RemoteConnection, mira::Object);

