/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeRecorder.h
 *    Class for recording data to tape.
 *
 * @author Tim Langner
 * @date   2010/12/28
 */

#ifndef _MIRA_TAPERECORDER_H_
#define _MIRA_TAPERECORDER_H_

#include <serialization/Serialization.h>
#include <fw/Framework.h>
#include <fw/Tape.h>
#include <fw/PerformanceStatistics.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Contains all settings for a recorded channel like compression, codecs,...
 */
class RecordedChannelInfo
{
public:

	RecordedChannelInfo() :
		compressionLevel(0),
		interval(Duration::microseconds(0)),
		avoidDataLoss(false) {}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		serialization::VersionType v = MIRA_REFLECT_VERSION(r, 2);
		r.member("Codecs", codecs, "The codecs used for this channel",
		         std::list<BinarySerializerCodecPtr>());
		if (v < 2) {
			bool compress;
			r.member("Compress", compress, "Should the data be stored compressed", false);
			compressionLevel = (compress ? -1 : 0);
		} else {
			r.member("CompressionLevel", compressionLevel,
			         "Compression level (0=none, 1..9=best speed to best compression, -1=default)", 0);
		}
		r.member("Interval", interval, "Stores only data in that interval (if 0 stores all data)",
		         Duration::microseconds(0));
		r.member("AvoidDataLoss", avoidDataLoss,
		         "Avoid data loss while recording this channel by reading all data from a channel since last recording",
		         false);
	}

	std::list<BinarySerializerCodecPtr> codecs;
	int compressionLevel;
	Duration interval;
	bool avoidDataLoss;
};

/// A map of recorded channel settings
typedef std::map<std::string, RecordedChannelInfo> RecordedChannelInfoMap;

/**
 * A config that can be passed to the TapeRecorder which contains all settings
 * that should be used for recording
 */
class TapeRecorderConfig
{
public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Channels", channels, "The channels to be recorded");
		r.member("MaxMessageBlockSize", maxMessageBlockSize,
		         "The maximum size of a message block in bytes",
		         64*1024*1024); // default size is 64MB
		r.member("Filename", filename, "The filename for the output tape");
	}

	RecordedChannelInfoMap channels;
	uint32 maxMessageBlockSize;
	Path filename;
};
/**
 * @ingroup FWModule
 * Class for recording channel data to tape.
 */
class MIRA_FRAMEWORK_EXPORT TapeRecorder
{
public:

	/// Informations about a recorded channel
	class ChannelInfo : public RecordedChannelInfo
	{
	public:
		ChannelInfo() :
			recordCounter(0) {}
		Time expected;
		uint64 recordCounter;
		TypeMetaPtr meta;
		MetaTypeDatabase metaDB;
		PerformanceStatistics stats;
	};

	/// Typedef for ChannelInfo pointers
	typedef std::map<std::string, ChannelInfo> ChannelMap;

public:
	/**
	 * Create a TapeRecorder
	 * @param[in] initialWaitForDataDuration Max time to wait for initial data in all recorded channels.
	 *                                       The purpose is to give channels published remotely some time after
	 *                                       subscribing to receive data already published in the past.
	 *                                       The given duration is the max total waiting time for all
	 *                                       recorded channels (i.e. determination of the tape start time begins
	 *                                       after 1 data update is read from each recorded channel, or when the
	 *                                       time has passed after calling record(), whichever happens first).
	 */
	TapeRecorder(const Duration& initialWaitForDataDuration = Duration::seconds(0));

	virtual ~TapeRecorder();

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("MaxMessageBlockSize", mMaxMessageBlockSize,
		           "The maximum size of a message block in bytes");
	}

	/**
	 * Add a channel to the list of recorded channels.
	 * @param[in] channelID Name of the channel
	 * @param[in] codecs A list of used codecs or empty if none
	 * @param[in] compress Should the channel data be recorded in a compressed format (zip)
	 * @param[in] interval Interval in which the data gets recorded.
	 *            E.g. if a channel changes every 10 ms and the interval is 100 ms,
	 *            only 1 out of 10 changes are recorded.
	 * @param[in] avoidDataLoss Avoid data loss on the channel during recording
	 *            by using subscribeInterval() instead of subscribe()
	 */
	MIRA_DEPRECATED("Please use int compressionLevel(=0/-1) instead of bool compress(=false/true)",
	void addChannel(const std::string& channelID, 
	                const std::list<BinarySerializerCodecPtr>& codecs,
	                bool compress, Duration interval = Duration::microseconds(0), bool avoidDataLoss = false))
	{
		addChannel(channelID, codecs, (compress ? -1 : 0), interval, avoidDataLoss);
	}


	/**
	 * Add a channel to the list of recorded channels.
	 * @param[in] channelID Name of the channel
	 * @param[in] codecs A list of used codecs or empty if none
	 * @param[in] compressionLevel libzip compression level for recorded channel data
	 *            (0=none, 1..9=best speed to best compression, -1=default compression)
	 * @param[in] interval Interval in which the data gets recorded.
	 *            E.g. if a channel changes every 10 ms and the interval is 100 ms,
	 *            only 1 out of 10 changes are recorded.
	 * @param[in] avoidDataLoss Avoid data loss on the channel during recording
	 *            by using subscribeInterval() instead of subscribe()
	 */
	void addChannel(const std::string& channelID,
	                const std::list<BinarySerializerCodecPtr>& codecs = std::list<BinarySerializerCodecPtr>(),
	                int compressionLevel = 0,
	                Duration interval = Duration::microseconds(0), bool avoidDataLoss = false);

	/**
	 * Add a channel to the list of recorded channels by providing an ID
	 * and an info object that contains the settings for the channel.
	 * @param[in] channelID Name of the channel
	 * @param[in] info Settings for recording the channel
	 */
	void addChannel(const std::string& channelID,
	                const RecordedChannelInfo& info);

	/**
	 * Start recording to file.
	 * @param[in] file Filename of the tape
	 */
	void record(const Path& file);

	/**
	 * Start recording to file using a recorder config that
	 * contains channels and settings for recording.
	 * @param[in] config A config containing all settings for recording
	 */
	void record(const TapeRecorderConfig& config);

	/**
	 * Stop recording.
	 * @param[in] finishQueue If true, the internal storage queue is finished before returning.
	 *                        Otherwise, items remain in the queue and the tape file is not 
	 *                        finished, processStorageQueue() must be called until the queue
	 *                        is empty.
	 */
	void stop(bool finishQueue = true);

	/**
	 * Query size of internal storage queue.
	 * This can be called during recording or after stopping.
	 */
	size_t storageQueueSize();

	/**
	 * Process single element or a specified number of elements from storage queue.
	 * This should only be used to process remaining queue entries after
	 * recording was stopping with stop(finishQueue=false). This way,
	 * the external caller can control and monitor the number
	 * of elements to process (e.g. if it wants to display a progress bar).
	 *
	 * Will throw XLogical if called while still recording.
	 */
	void processStorageQueue(int numEntries = 1);

	/**
	 * Set the maximum size of the chunks
	 * @param[in] maxMessageBlockSize Size of a chunk in bytes
	 */
	void setMaxMessageBlockSize(uint32 maxMessageBlockSize)
	{
		mMaxMessageBlockSize = maxMessageBlockSize;
	}

	/**
	 * Returns whether TapeRecorder is currently gathering data after record was called.
	 */
	bool isRecording() const
	{
		return mIsRecording;
	}

	const PerformanceStatistics& getStats(const std::string& name);

protected:

	// normal callback for data changes (writes data to tape)
	void valueChanged(ChannelRead<void> value, bool forceStorage = false);
	void intervalValueChanged(ChannelReadInterval<void> values, bool forceStorage = false);

private:

	struct MessageToStore
	{
		StampedHeader header;
		std::string channelID;
		std::string typeName;
		Time enqueueTime;
		Buffer<uint8> buf;
		ChannelInfo* info;
	};

	void storageThread();
	void storeMessage(MessageToStore& msg);

protected:

	Authority mAuthority;
	uint32 mMaxMessageBlockSize;
	Tape mTape;
	bool mIsRecording;
	bool mFirstCallbackMessage;
	ChannelMap mChannels;
	Duration mInitialWaitForDataDuration;

	boost::thread mStorageThread;
	boost::condition_variable mStorageThreadCondition;
	boost::mutex mStorageMutex;
	bool mStorageThreadStarted;

	std::deque<MessageToStore> mStorageQueue;
	volatile bool mStorageThreadTerminationRequest;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
