/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file miraauth.C
 *    .
 *
 * @author Erik Einhorn
 * @date   2012/01/28
 */

#include <iostream>
#include <fstream>

#include <security/RSAKey.h>

#include <utils/ProgramOptions.h>

using namespace mira;

void printUsage()
{
	std::cout <<
			"miraauth keygen [options]\n\n"
			"Generates a new private and public key pair for authentication\n"
			"of frameworks.\n"
			"\n"
			"Options:\n"
			"   -f output.key    The file where to write the key/keys\n"
			"   -b bits          The number of bits in the key to create\n"
			"   --private        Generate the private key only\n"
			"\n"
			"Example:\n"
			"miraauth keygen --private -f output.key\n"
			<< std::endl;
}

int main(int argc, char** argv)
{
	if(argc<2) {
		printUsage();
		return -1;
	}

	ProgramOptions options(argc, argv);

	options.getDescriptions().add_options()
		("command", boost::program_options::value<std::string>(), "")
		("file,f", boost::program_options::value<std::string>(), "")
		("bits,b", boost::program_options::value<int>()->default_value(1024), "")
	    ("private", "");

	options.getPositionalDescriptions().add("command", -1);

	ProgramOptions::VariableMap vmap = options.getOptions();

	if(!vmap.count("command")) {
		printUsage();
		return -1;
	}

	if(vmap["command"].as<std::string>()!="keygen"){
		printUsage();
		return -1;
	}

	std::ofstream fstream;
	std::ostream* stream = &std::cout;

	if(vmap.count("file")!=0) {
		std::string file = vmap["file"].as<std::string>();
		fstream.open(file.c_str());
		stream = &fstream;
	}

	RSAKey publicKey, privateKey;
	RSAKey::generateKey(vmap["bits"].as<int>(), publicKey, privateKey);

	*stream << privateKey << std::endl;

	if(vmap.count("private")==0)
		*stream << publicKey << std::endl;

	return 0;
}
