/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Inspect.h
 *    Provides the MIRA_INSPECT macro.
 *
 * @author Erik Einhorn
 * @date   2010/09/28
 */

#ifndef _MIRA_INSPECT_H_
#define _MIRA_INSPECT_H_

#include <iostream>
#include <string>
#include <vector>

#ifndef Q_MOC_RUN
#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/classification.hpp>
#endif

#ifndef NDEBUG
/**
 * Helper macro that prints the value of one or more variables on the console.
 * To inspect a value of a variable you often have to write:
 * @code
 * int a, b, c;
 * cout << "a=" << a ", b=" << b << ", c=" << c << endl;
 * @endcode
 *
 * With this macro it's sufficient to write:
 * @code
 * MIRA_INSPECT(a,b,c)
 * @endcode
 *
 * The output on the console will then be:
 * @code
 * a = ...
 * b = ...
 * c = ...
 * @endcode
 *
 * @note This macro will be disabled automatically in release builds.
 */
#define MIRA_INSPECT( ... )  _MIRA_INSPECT(__VA_ARGS__,  #__VA_ARGS__)
#else
#define MIRA_INSPECT( ... )
#endif

///////////////////////////////////////////////////////////////////////////////

#define _MIRA_INSPECT_COMMON                                                  \
    std::string s(names);                                                     \
    std::vector<std::string> v;                                               \
    boost::algorithm::split(v, s, boost::algorithm::is_from_range(',',','));

#define _MIRA_INSPECT_END

template <typename A>
void _MIRA_INSPECT(const A& a, const char* names)
{
	_MIRA_INSPECT_COMMON
	std::cout << v[0] << "=" << a << std::endl;
	_MIRA_INSPECT_END
}

template <typename A, typename B>
void _MIRA_INSPECT(const A& a, const B& b, const char* names)
{
	_MIRA_INSPECT_COMMON
	std::cout << v[0] << "=" << a << std::endl;
	std::cout << v[1] << "=" << b << std::endl;
	_MIRA_INSPECT_END
}

template <typename A, typename B, typename C>
void _MIRA_INSPECT(const A& a, const B& b, const C& c, const char* names)
{
	_MIRA_INSPECT_COMMON
	std::cout << v[0] << "=" << a << std::endl;
	std::cout << v[1] << "=" << b << std::endl;
	std::cout << v[2] << "=" << c << std::endl;
	_MIRA_INSPECT_END
}

template <typename A, typename B, typename C, typename D>
void _MIRA_INSPECT(const A& a, const B& b, const C& c, const D& d, const char* names)
{
	_MIRA_INSPECT_COMMON
	std::cout << v[0] << "=" << a << std::endl;
	std::cout << v[1] << "=" << b << std::endl;
	std::cout << v[2] << "=" << c << std::endl;
	std::cout << v[3] << "=" << d << std::endl;
	_MIRA_INSPECT_END
}

template <typename A, typename B, typename C, typename D, typename E>
void _MIRA_INSPECT(const A& a, const B& b, const C& c, const D& d,
                   const E& e, const char* names)
{
	_MIRA_INSPECT_COMMON
	std::cout << v[0] << "=" << a << std::endl;
	std::cout << v[1] << "=" << b << std::endl;
	std::cout << v[2] << "=" << c << std::endl;
	std::cout << v[3] << "=" << d << std::endl;
	std::cout << v[4] << "=" << e << std::endl;
	_MIRA_INSPECT_END
}

template <typename A, typename B, typename C, typename D, typename E,
          typename F>
void _MIRA_INSPECT(const A& a, const B& b, const C& c, const D& d,
                   const E& e, const F& f, const char* names)
{
	_MIRA_INSPECT_COMMON
	std::cout << v[0] << "=" << a << std::endl;
	std::cout << v[1] << "=" << b << std::endl;
	std::cout << v[2] << "=" << c << std::endl;
	std::cout << v[3] << "=" << d << std::endl;
	std::cout << v[4] << "=" << e << std::endl;
	std::cout << v[5] << "=" << f << std::endl;
	_MIRA_INSPECT_END
}

template <typename A, typename B, typename C, typename D, typename E,
          typename F, typename G>
void _MIRA_INSPECT(const A& a, const B& b, const C& c, const D& d,
                   const E& e, const F& f, const G& g, const char* names)
{
	_MIRA_INSPECT_COMMON
	std::cout << v[0] << "=" << a << std::endl;
	std::cout << v[1] << "=" << b << std::endl;
	std::cout << v[2] << "=" << c << std::endl;
	std::cout << v[3] << "=" << d << std::endl;
	std::cout << v[4] << "=" << e << std::endl;
	std::cout << v[5] << "=" << f << std::endl;
	std::cout << v[6] << "=" << g << std::endl;
	_MIRA_INSPECT_END
}

#undef _MIRA_INSPECT_COMMON
#undef _MIRA_INSPECT_END

///////////////////////////////////////////////////////////////////////////////

#endif
