/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SQLiteQuery.h
 *    Class representing a result of a database query.
 *
 * @author Tim Langner
 * @date   2010/10/25
 */

#ifndef _MIRA_SQLITEQUERY_H_
#define _MIRA_SQLITEQUERY_H_

#include <utils/ToString.h>
#include <serialization/adapters/std/pair>
#include <serialization/adapters/std/vector>

struct sqlite3;
struct sqlite3_stmt;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup DatabaseModule
 * 
 * Class representing a result of a SQLite database query.
 */
class MIRA_BASE_EXPORT SQLiteQuery
{
public:
	/** @name Constructors and destructor */
	//@{

	/// Default-constructed query will be invalid
	SQLiteQuery();

	/**
	 * Constructs a query object given a database and a statement
	 * @param[in] db        The database pointer
	 * @param[in] statement The statement pointer
	 * @param[in] more      Does the query contain data
	 */
	SQLiteQuery(sqlite3* db, sqlite3_stmt* statement, bool more);

	// no copy as copying statement would be illegal
	SQLiteQuery(const SQLiteQuery& other) = delete;
	SQLiteQuery& operator=(const SQLiteQuery& other) = delete;

	// move, moves statement
	SQLiteQuery(SQLiteQuery&& other);
	SQLiteQuery& operator=(SQLiteQuery&& other);

	/// The destructor.
	~SQLiteQuery();

	//@}

public:
	/** @name Query operations */
	//@{

	/// SQLiteQuery is valid if statement is not null
	bool valid() const
	{
		return mStatement != NULL;
	}

	/**
	 * Return if the query contains data.
	 * @return True if there is no data in the query, false otherwise.
	 */
	bool eof() const
	{
		return mMore == false;
	}

	/**
	 * Return the column count of the query.
	 * @return The number of columns.
	 */
	int getColumnCount() const
	{
		return mColumnCount;
	}

	/**
	 * Return the name of a given column.
	 * @param[in] column The column.
	 * @return The column name.
	 */
	std::string getName(std::size_t column);

	/**
	 * Return a typecasted value for a given column in the current row.
	 * @throws XBadCast if the element in the data field is NULL.
	 * @param[in] column The column to be returned.
	 * @return The casted value.
	 */
	template<typename T>
	T getValue(std::size_t column)
	{
		const char* data = getTxtValue(column);
		if ( data == NULL )
			MIRA_THROW(XBadCast, "Value of column '" << column << "' is NULL");
		return fromString<T>(data);
	}

	/**
	 * Return a typecasted value for a given column in the current row.
	 * If this value is NULL a default value is returned.
	 * @param[in] column The column to be returned.
	 * @param[in] defaultValue The default value that is returned if the cast fails.
	 * @return The casted value.
	 */
	template<typename T>
	T getValue(std::size_t column, const T& defaultValue)
	{
		const char* data = getTxtValue(column);
		if ( data == NULL )
			return defaultValue;
		return fromString<T>(data);
	}

	/**
	 * Get the data type of a given column in the current row. 
	 * Could be one of the following:
	 *   SQLITE_INTEGER = int
	 *   SQLITE_FLOAT   = float
	 *   SQLITE_TEXT    = string
	 *   SQLITE_BLOB    = string
	 *   SQLITE_NULL    = NULL
	 * @param[in] column The column to get the datatype from
	 * @return Datatype.
	 */
	int getDataType(std::size_t column);

	/**
	 * Return if a entry in a given column is NULL.
	 * @param[in] column The column to check for NULL type
	 * @return True if value is NULL, false otherwise.
	 */
	bool isNull(std::size_t column);


	/**
	 * Advance to next row.
	 * /code
	 * while (!query.eof())
	 * {
	 *     ...
	 *     query.next();
	 * }
	 * /endcode
	 */
	void next();

	/**
	 * Preincrement operator. Same as next().
	 */
	SQLiteQuery& operator++()
	{
		next();
		return *this;
	}

private:
	const char* getTxtValue(std::size_t column);

	sqlite3* mDB;
	sqlite3_stmt* mStatement;
	bool mMore;
	int mColumnCount;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup DatabaseModule
 * 
 * A cached SQLite query object that supports serialization.
 * Can be used to store a database query in a log file or for for
 * RPC database query.
 */
class MIRA_BASE_EXPORT SQLiteCachedQuery
{
public:
	typedef std::vector<std::string> HeaderVector;
	typedef std::vector<std::vector<std::pair<int, std::string>>> ValueVector;

public:
	/** @name Constructors and destructor */
	//@{

	/// Default constructor.
	SQLiteCachedQuery();

	/// The copy constructor.
	SQLiteCachedQuery(SQLiteQuery& query);

	//@}

public:
	/** @name Query operations */
	//@{
	
	/**
	 * Return if the query contains data.
	 * @return True if there is no data in the query, false otherwise.
	 */
	bool eof() const
	{
		return mCurrent >= mCache.size();
	}

	/**
	 * Return the column count of the query.
	 * @return The number of columns.
	 */
	int getColumnCount() const
	{
		return mHeader.size();
	}

	/**
	 * Returns the name of a given column.
	 * @param[in] column The column
	 * @return The column name
	 */
	std::string getName(std::size_t column)
	{
		if(column>=mHeader.size())
			MIRA_THROW(XAccessViolation, "Column " << column << " does not exist");

		return mHeader[column];
	}
	
	/**
	 * Returns a typecasted value for a given column in the current row.
	 * @throws XBadCast if the element in the data field is NULL.
	 * @param[in] column The column to be returned.
	 * @return The casted value.
	 */
	template<typename T>
	T getValue(std::size_t column)
	{
		if (isNull(column))
			MIRA_THROW(XBadCast, "Value of column '" << column << "' is NULL");
		return fromString<T>(mCache[mCurrent][column].second);
	}

	/**
	 * Returns a typecasted value for a given column in the current row.
	 * If this value is NULL a default value is returned.
	 * @param[in] column The column to be returned.
	 * @param[in] defaultValue The default value that is returned if the cast fails.
	 * @return The casted value.
	 */
	template<typename T>
	T getValue(std::size_t column, const T& defaultValue)
	{
		if (isNull(column))
			return defaultValue;
		try
		{
			return fromString<T>(mCache[mCurrent][column].second);
		}
		catch(...)
		{
		}
		return defaultValue;
	}

	/**
	 * Get the data type of a given column in the current row.
	 * Could be one of the following:
	 *   SQLITE_INTEGER = int
	 *   SQLITE_FLOAT   = float
	 *   SQLITE_TEXT    = string
	 *   SQLITE_BLOB    = string
	 *   SQLITE_NULL    = NULL
	 * @return Datatype
	 */
	int getDataType(std::size_t column);
	
	/**
	 * Return if a entry in a given column is NULL.
	 * @return True if value is NULL, false otherwise.
	 */
	bool isNull(std::size_t column);


	/**
	 * Advance to next row.
	 * /code
	 * while (!query.eof())
	 * {
	 *     ...
	 *     query.next();
	 * }
	 * /endcode
	 */
	void next()
	{
		validate();
		++mCurrent;
	}

	/**
	 * Preincrement operator. Same as next().
	 */
	SQLiteCachedQuery& operator++()
	{
		next();
		return *this;
	}

	/**
	 * Return the cached values.
	 * @return The cached values.
	 */
	ValueVector getValues() const
	{
		return mCache;
	}

	//@}

public:
	/// Reflect method for serialization
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Header", mHeader, "The header of the query");
		r.member("Values", mCache, "The values of the query");
	}

protected:

	void validate()
	{
		if(eof())
			MIRA_THROW(XAccessViolation, "Access to invalid query object.");
	}

	void validate(std::size_t column)
	{
		validate();
		if(column>=mCache[mCurrent].size())
			MIRA_THROW(XAccessViolation, "Column " << column << " does not exist");
	}

	HeaderVector mHeader;
	ValueVector mCache;
	std::size_t mCurrent;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif

