/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Repository.C
 *    Implementation of Repository.h.
 *
 * @author Ronny Stricker
 * @date   2011/08/30
 */

#include "core/Repository.h"

#include <boost/algorithm/string.hpp>
#include <boost/algorithm/string/predicate.hpp>

#include <QUrl>
#include <QByteArray>
#include <QBuffer>
#include <QTextStream>
#include <QApplication>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QNetworkAccessManager>
#include <QSslError>

#include <quazip/quazip.h>
#include <quazip/JlCompress.h>

#include <serialization/Serialization.h>
#include <xml/XMLDom.h>

#include "core/PackageParser.h"
#include "core/Tools.h"
#include "core/Package.h"

using namespace std;
using namespace boost;

MIRA_CLASS_SERIALIZATION(mira::Repository,mira::Object);

namespace mira {

///////////////////////////////////////////////////////////////////////////////

Repository::~Repository()
{
}

void Repository::setUrl( Url const& iUrl )
{
	if ( iUrl.find("://") == std::string::npos ) {
		MIRA_THROW( XLogical, "Please specify protocol for url " << iUrl );
	}

	url = iUrl;

	// remove tailing slashes
	while ( url.size()>0 && *url.rbegin() == '/' )
		url.resize( url.size()-1 );
}

RepositoryPtr Repository::create( string const& iName,
		std::string const& iDescription,
		Url const& iUrl,
		Url const& iIndexFile,
		uint32 priority, std::string const& iType )
{
	std::string type = iType;
	if ( iType == "autoDetect" ) {
		std::string tUrl = to_lower_copy(iUrl);
		if ( find_first(tUrl,"svn") )
			type = "SVN";
		else if ( find_first(tUrl,"ftp") )
			type = "FTP";
		else if ( starts_with(tUrl, "git") or starts_with(tUrl, "http") or starts_with(tUrl, "ssh") )
			type = "Gitlab";
		else if ( find_first(tUrl,"file") )
			type = "Local";
		else {
			MIRA_THROW( XLogical, "Cannot determine repository type of repository \"" + iName + "\"!");
		}
	}
	std::vector<ClassProxy> tRepoVec = ClassFactory::getClassByMeta("RepoType",type);
	if ( tRepoVec.size() != 1 ) {
		MIRA_THROW( XLogical, "Don't know how to handle repository type " +
		            type + " of repository \"" + iName + "\"!");
	}
	Repository* tRepo =  tRepoVec[0].newInstance<Repository>();
	assert( tRepo );
	tRepo->name = iName;
	tRepo->description = iDescription;
	tRepo->setUrl( iUrl );
	tRepo->priority = priority;
	tRepo->indexFile = iIndexFile;

	return RepositoryPtr(tRepo);
}

RepositoryPtr Repository::createFromURL(const std::string& iURL)
{
	QByteArray byteArray;
	QNetworkAccessManager networkMng;
	QUrl getURL(QString::fromStdString(iURL));

	// create the network request and the reply
	QNetworkRequest request;
	request.setUrl(getURL);
	request.setRawHeader("User-Agent", "mirapackage");
	QNetworkReply* reply = networkMng.get(request);
	reply->ignoreSslErrors();

	// spin until the download is finished or an error occurred
	while(!reply->isFinished() &&
	      (reply->error() == QNetworkReply::NoError))
	{
		QCoreApplication::processEvents();
		if (reply->bytesAvailable() > 0)
			byteArray.append(reply->readAll());
	}
	// Was the download successful?
	if (reply->error() == QNetworkReply::NoError)
		byteArray.append(reply->readAll());

	// Read the XML repository information file.
	XMLDom xmlRepoInfo;
	try {
		xmlRepoInfo.loadFromString(std::string(byteArray.data()));
	} catch (Exception&) {
		MIRA_THROW(XLogical, "Can not parse data from URL \"" << iURL << "\"");
	}

	// Deserialize the repository information from XML.
	boost::shared_ptr<Repository> repo;
	try {
		XMLDeserializer deserializer(xmlRepoInfo);
		deserializer.deserialize("MIRA-Repository", repo);
	} catch (Exception&) {
		MIRA_THROW(XLogical, "Can not get repository information from URL \"" << iURL << "\"");
	}

	return(repo);
}

void Repository::examine( vector<Package*>& oPackages, bool thorough,
                          boost::function<void (std::string const&,int,int)> progressListener )
{
	MIRA_LOG( NOTICE ) << "examining: " << url;
	mountDirs.clear();

	bool doThoroughReindex = true;

	// we need a map of packages and changelogs in order to merge informations
	// after we have cycled through all the files of the zip archive
	map<string,Package*> packages;
	map<string,string> changelogs;

	// if no thorough index is required and if a index file is given, we can
	// start to evaluate the index file
	if ( !thorough && !indexFile.empty() ) {
		try {
			MIRA_LOG( NOTICE ) << "try to use index file";
			QByteArray byteArray ;
			getFile(indexFile, byteArray);
			QBuffer buffer( &byteArray ) ;

			QuaZip testZip;
			testZip.setIoDevice( &buffer );
			if ( !testZip.open(QuaZip::mdUnzip) ) {
				MIRA_THROW( XIO, "unable to open index file \""+indexFile+"\"" );
			}

			QuaZipFile file(&testZip);

			int packageCount = 0;

			Url rootUrl;

			// count package files and find rootUrl file
			for(bool more=testZip.goToFirstFile(); more; more=testZip.goToNextFile()) {
				Path p( file.getActualFileName().toStdString() );
				if( p.extension()==".package"
						|| p.extension()==".extpackage"
						|| p.filename()=="mountdir.xml" )
					++packageCount;

				if ( p.filename() == "rootUrl" ) {
					file.open(QIODevice::ReadOnly);

					QTextStream reader( &file );
					rootUrl = reader.readAll().trimmed().toStdString();
					file.close();
				}
			}

			if ( rootUrl.empty() ) {
				MIRA_THROW( XLogical, "Index file is invalid! Cannot find rootUrl.");
			}
			if ( url.find( rootUrl ) != 0 ) {
				MIRA_THROW( XLogical, "Index file is invalid! RootUrl does not match the repository url.");
			}

			int packageProgress = 0;

			for( bool more=testZip.goToFirstFile(); more; more=testZip.goToNextFile() ) {
				file.open(QIODevice::ReadOnly);

				string fileName = file.getActualFileName().toStdString();
				Path p( fileName );

				if ( p.extension()==".package"
						|| p.extension()==".extpackage"
						|| p.extension()==".changelog"
						|| p.filename() =="mountdir.xml") {
					// read the file
					QTextStream reader( &file );
					string fileContent = reader.readAll().toStdString();

					// index the package files
					if ( p.extension()==".package"
							|| p.extension()==".extpackage" ) {
						Package* tPackage = indexWebPackage(
								fileName, this, &fileContent );
						if ( tPackage ) {
							oPackages.push_back( tPackage );
							packages[ tPackage->mName ] = tPackage;
						}
					}
					else if ( p.extension()==".changelog") {
						std::string filename = getFilename( fileName );
						// extract name of package by removing the file extension
						std::string name = filename.substr(0, filename.size()
								- Path(fileName).extension().string().length() );

						changelogs[name] = changeLogToHTML(fileContent);
					} else {
						// parse branch info
						parseMountDirXml( rootUrl / Url(fileName),
								fileContent );
					}
					++packageProgress;
					if ( progressListener )
						progressListener(fileName, packageProgress, packageCount );
				}

				file.close();
			}
			testZip.close();
			// quick reindex has succeeded -> no further indexing actions required
			// -> return true
			doThoroughReindex = false;
		} catch ( Exception& ex ) {
			if ( prompt ) {
				std::string tErrMsg("Error: \n");
				tErrMsg += ex.message()+"\n\n";
				tErrMsg += "Proceed with thorough indexing?";
				if (!prompt->showYesNoErrorMessage(tErrMsg))
					// quick reindex has failed but no further indexing actions
					// are desired
					doThoroughReindex = false;
			}
		}
	}

	// merge changelogs and packages
	foreach( auto it, changelogs) {
		auto package = packages.find( it.first );
		if ( package != packages.end() ) {
			package->second->mChangeLog = it.second;
		}
	}

	// call repository specific examine if required
	if ( doThoroughReindex ) {
		deepExamine( oPackages, progressListener );
	}
}

void Repository::parseMountDirXml( Url const& url, string const& fileContent )
{
	// read info file
	XMLDom xml;
	try {
		xml.loadFromString(fileContent);
	} catch ( Exception& e ) {
		MIRA_LOG(NOTICE) << "failed to load mountdir xml!";
		MIRA_LOG(NOTICE) << e.message();
		MIRA_LOG(NOTICE) << "text: " << fileContent;
	}

	auto it = xml.croot().find("path");
	if(it!=xml.croot().end()) {
		Url path = *it.content_begin();
		std::string urlDir = getParentUrl( url );
//		path = rootPath / path;
		mountDirs.insert(std::make_pair(urlDir,path));
		MIRA_LOG(NOTICE) << "found mountdir. insert " << urlDir << " as " << path;
	}
}

Url Repository::getMountDirForPackage( Url const& subUrl, std::string* matchedUrl )
{
	std::string packagePath = url / subUrl;

	// find longest matching branch
	auto match = mountDirs.end();
	std::size_t matchlen = 0;
	for(auto it=mountDirs.begin(); it!=mountDirs.end(); ++it)
	{
		std::size_t len = it->first.size();
		if(len>matchlen && packagePath.size()>=len) {
			std::string s = packagePath.substr(0, len);
			if(s==it->first) {
				match=it;
				matchlen=len;
			}
		}
	}

	if(match==mountDirs.end()) {
		MIRA_LOG(WARNING) << "No mount dir found for " << packagePath;
		return "";
	} // nothing found

	if ( matchedUrl ) {
		*matchedUrl = match->first;
	}

	return match->second;
}

void Repository::removeGeneratedLibraryFiles( Package const& package )
{
	// does not work with windows....
	if ( PathProvider::isMiraSubPath( package.mLocalPath ) ) {
		// get the associated mira path
		Path miraPath = PathProvider::getAssociatedMiraPath( package.mLocalPath );
		// remove associated files in lib and bin directory
		vector<Path> tSubPaths;
		tSubPaths.push_back("lib");
		tSubPaths.push_back("bin");
		foreach( Path const& tSubPath, tSubPaths ) {
			QDir tDir( QString::fromStdString( (miraPath / tSubPath ).string() ) );

			QStringList tVersionSymLinks;
			if ( tDir.exists() ) {
				// cycle through all children
				QFileInfoList entries = tDir.entryInfoList(QDir::NoDotAndDotDot | QDir::Files | QDir::System);
				foreach( QFileInfo const& info, entries ) {
					// obtain the link target and remove the build/[debug|release|...]
					QString linkTarget = info.symLinkTarget();
					int length = (miraPath).string().length();
					linkTarget.remove( length, linkTarget.indexOf('/',length+7 )-length );
					// if we have found a file linking to the directory
					// which should be removed -> remove it as well
					if ( linkTarget.toStdString().find( package.mLocalPath.string() ) == 0 ) {
						QFile file( info.absoluteFilePath() );

						if ( info.fileName().contains( ".so." ) ) {
							// try to remove all the other links to this file
							// we have to check for the existence of the symlinks
							// before we delete the symlink to the real file
							// otherwise qt will report that the file does not exist
							QString tFileName = info.fileName();
							do {
								// cut filename by one subversion part
								tFileName = tFileName.left( tFileName.lastIndexOf('.') );
								if ( tDir.exists( tFileName ) ) {
									tVersionSymLinks.push_back( tDir.absoluteFilePath( tFileName ) );
								}
							}
							while ( tFileName.contains(".so.") );
						}

						MIRA_LOG(NOTICE) << "remove " << info.absoluteFilePath().toStdString();
						if (!file.remove())
							MIRA_THROW(XIO,"Error removing file "+info.absoluteFilePath().toStdString()+"!")
					}
				}
				// now we can remove the version symlinks
				foreach( QString const& tVersion, tVersionSymLinks ) {
					QFile file( tVersion );
					MIRA_LOG(NOTICE) << "remove " << tVersion.toStdString();
					if (!file.remove())
						MIRA_THROW(XIO,"Error removing file " + tVersion.toStdString()+"!")
				}

			}
		}
	}
}

Url Repository::removePrefix( Url const& path )
{
	std::string tPath = path;
	if ( tPath.find("://") != std::string::npos ) {
		return tPath.substr( tPath.find("://")+3 );
	}
	return path;
}

void Repository::getFile( Url const& url, QByteArray& byteArray )
{
	QNetworkAccessManager networkMng;

	bool finished = false;
	do {
		MIRA_LOG(NOTICE) << "getFile: " << url;
		QUrl getURL( QString::fromStdString( url ) );
		getURL.setUserName(QString::fromStdString(credential.user));
		getURL.setPassword(QString::fromStdString(credential.password));

		// create the network request
		QNetworkRequest request;
		request.setUrl(getURL);
		request.setRawHeader("User-Agent", "mirapackage");

		QNetworkReply* reply = networkMng.get(request);
		reply->ignoreSslErrors(); // TODO: Ignore not all SLL errors.

		// register import signals
		connect(reply, SIGNAL(finished()), this, SLOT(networkReplyFinished()));
		connect(reply, SIGNAL(error(QNetworkReply::NetworkError)),
		        this, SLOT(networkReplyError(QNetworkReply::NetworkError)));
		connect(reply, SIGNAL(sslErrors(const QList<QSslError>&)),
		        this, SLOT(networkReplySSLErrors(const QList<QSslError>&)));

		// spin until the download is finished or an error occurred
		while(!reply->isFinished() &&
		      (reply->error() == QNetworkReply::NoError))
		{
			QApplication::processEvents();
			if (reply->bytesAvailable() > 0)
				byteArray.append(reply->readAll());
		}

		// Was the download successful?
		if (reply->error() == QNetworkReply::NoError) {
			byteArray.append(reply->readAll());
			finished = true;
		}

		// Invalid authentication ?
		if (reply->error() == QNetworkReply::AuthenticationRequiredError) {
			credential.realm = url;
			prompt->getLogin(credential);
		} else
			// Other error => interrupt
			finished = true;

		// clean up memory
		delete reply;
	} while (!finished);
	MIRA_LOG(NOTICE) << "got " << byteArray.size() << " bytes";
}

///////////////////////////////////////////////////////////////////////////////

void Repository::networkReplyFinished()
{
//	std::cout << "Repository::networkReplyFinished()" << std::endl;
}
void Repository::networkReplyError(QNetworkReply::NetworkError code)
{
//	std::cout << "Repository::networkReplyError(): " << code << ": "
//	        /*<< errorString(code)*/ << std::endl;
}

void Repository::networkReplySSLErrors(const QList<QSslError>& errors)
{
//	std::cout << "Repository::networkReplySSLErrors(): " << std::endl;
//	for(int i = 0; i < errors.size(); i++)
//		std::cout << "\t" << errors[i].errorString().toStdString() << std::endl;
}

///////////////////////////////////////////////////////////////////////////////

bool Repository::isGeneric() const
{
	return url.size()>=7 && url.substr(0,7) == "generic";
}

void Repository::prepareCheckoutDirectory(Path dir, bool generateCMakeLists)
{
	bool foundRootCMakeList = false;
	Path p;
	foreach(Path sub, dir)
	{
		p = p/sub;

		Path cm = p/"CMakeLists.txt";
		if(generateCMakeLists && boost::filesystem::exists(cm))
			foundRootCMakeList=true;
		// if we found a root cmake already, but if there's no CMakeLists.txt in this directory
		// AND this is not the leaf of the checkout dir, then create a CMakeLists.txt
		else if(foundRootCMakeList || !generateCMakeLists) {
			// create directory, if it does not exist
			if(!boost::filesystem::exists(p)) {
				boost::filesystem::create_directory(p);
			}
			if ( generateCMakeLists && p != dir)
				generateCMakeListsTxt(cm);
		}
	}
}

void Repository::generateCMakeListsTxt(Path dir)
{
	std::ofstream file(dir.string().c_str());
	file << "# autogenerated CMakeLists.txt" << std::endl;
	file << "ADD_EXISTING_SUBDIRS()" << std::endl;
}

///////////////////////////////////////////////////////////////////////////////

}
