/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SVNRepository.h
 *    Description.
 *
 * @author Ronny Stricker
 * @date   2011/08/30
 */
 
#ifndef _MIRA_SVNREPOSITORY_H_
#define _MIRA_SVNREPOSITORY_H_

#include "core/Repository.h"

#include "svn_client.h"
#include "svn_version.h"

#include <serialization/Serialization.h>

namespace mira {

/**
 * @brief Class provided for access to a svn repository.
 * Local installed packages are tracked using build in svn features.
 */
class SVNRepository : public Repository {

	MIRA_META_OBJECT(SVNRepository,("RepoType","SVN"))
	
public:
	/**
	 * @brief Setup svn environment.
	 */
	SVNRepository();

	virtual ~SVNRepository();
	
	virtual void setUrl( Url const& iUrl );

	/**
	 * @brief Svn specific implementation of Repository::examine().
	 */
	virtual void deepExamine( std::vector<Package*>& oPackages,
			boost::function<void (std::string const&,int,int)> progressListener = NULL );

	/**
	 * @brief Svn specific implementation of Repository::catFile().
	 */
	virtual std::string catFile( Url const& relUrl );
	
	/**
	 * @brief Svn specific implementation of Repository::isResponsibleFor().
	 * Function uses build in svn features (svn info).
	 */
	virtual bool isResponsibleFor( mira::Path const& localPath );
	
	/**
	 * @brief Svn specific implementation of Repository::getRelativePathFor().
	 * Function uses build in svn features (svn info).
	 */
	virtual mira::Path getRelativePathFor( mira::Path const& localPath );

	/**
	 * @brief Return the desired local path of the package using information
	 * from the mountdir files.
	 * @see Reositroy::getMountDirForPackage() for further information. This
	 * function simply appends the subdir to the path obtained by the function
	 * from the Repository base.
	 * @param subUrl suburl of the package (related to the repository root)
	 * @param[out] matchedUrl optional Url of associated mountdir
	 * @return relative local path obtained from branch.xml
	 */
	virtual Url getMountDirForPackage( Url const& subUrl, std::string* matchedUrl = NULL );

	/**
	 * @brief Svn specific implementation of Repository::install().
	 */
	virtual void install( Package const& package, mira::Path const& destPath );

	/**
	 * @brief Svn specific implementation of Repository::uninstall().
	 * Uninstalling is only applied if the files are not modified and if no
	 * unversioned items appear in the svn tree.
	 * Furthermore, the functions checks if it steps down into a switches path.
	 */
	virtual void uninstall( Package const& package );

private:

	/**
	 * @brief Initialize APR Pool and svn client context with all callbacks.
	 */
	void initializeSVN();

	/**
	 * @brief Internal function called by subversion to request credential if needed.
	 */
	static svn_error_t * my_simple_prompt_callback (svn_auth_cred_simple_t **cred,
	                           void *baton,
	                           const char *realm,
	                           const char *username,
	                           svn_boolean_t may_save,
	                           apr_pool_t *pool);

	/**
	 * @brief Internel function called by subversion if invalid ssl certificate is found.
	 */
	static svn_error_t* my_simple_sslserver_prompt_callback(svn_auth_cred_ssl_server_trust_t **cred_p,
								void *baton,
								const char *realm,
								apr_uint32_t failures,
								const svn_auth_ssl_server_cert_info_t *cert_info,
								svn_boolean_t may_save,
								apr_pool_t *pool);
	
	/**
	 * @brief Internal function to list all files of a certain url.
	 */
	std::list<Url> svnLs( Url const& url );
	
	/**
	 * @brief get the svn status of the given local path.
	 */
	bool svnStat( mira::Path const& path );

	/*
	 * SVN callback functions
	 */
	static svn_error_t* receiveSvnInfo(void* baton, const char* path,
			const svn_info_t* info, apr_pool_t*);

#if SVN_VER_MINOR >= 7
	static svn_error_t* receiveSvnInfo2(void* baton, const char* path,
			const svn_client_info2_t* info, apr_pool_t*);
#endif

	static svn_error_t* receiveSvnStatus(void* baton, const char* path,
			svn_wc_status2_t* status, apr_pool_t*);

#if SVN_VER_MINOR >= 7
	static svn_error_t* receiveClientSvnStatus( void *baton,
	                                            const char *path,
	                                            const svn_client_status_t *status,
	                                            apr_pool_t *scratch_pool );
#endif

	static svn_error_t* receiveListInfo(void* baton, const char* path,
			const svn_dirent_t* /*dirent*/, const svn_lock_t* /*lock*/,
			const char* /*abs_path*/, apr_pool_t* /*pool*/);

	static svn_error_t* receiveListInfo2(
	                                     void *baton,
	                                     const char *path,
	                                     const svn_dirent_t *dirent,
	                                     const svn_lock_t *lock,
	                                     const char *abs_path,
	                                     const char *external_parent_url,
	                                     const char *external_target,
	                                     apr_pool_t *scratch_pool );

protected:
	struct SVNStatusInfo {
		SVNStatusInfo( std::string const& iName, std::string const& iURL,
				svn_wc_status_kind iStatus ) :
					name( iName ), URL( iURL ), status( iStatus ) {}

		std::string name; ///< name of item
		std::string URL; ///< URL of item
		svn_wc_status_kind status; ///< Status of item
	};

	std::string mInfoRoot; ///< Repository root path for the last file query issued by isResponsibleFor()
	std::string mInfoURL; ///< URL for the last file query issued by isResponsibleFor()

	std::vector<SVNStatusInfo> mStatusInfo; ///< Status information obtained by svnStatus

	apr_pool_t* pool;		///< Memory pool used by subversion
	svn_client_ctx_t* ctx;	///< Svn context object

};

}

#endif /* _MIRA_SVNREPOSITORY_H_ */

