/*
 * Copyright (C) 2019 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file GitlabAPI.h
 *
 * @author Thomas Bauer
 * @date   2019/11/22
 */

#ifndef _GITLAB_API_H_
#define _GITLAB_API_H_

#include <curl/curl.h>
#include <core/Url.h>
#include <serialization/adapters/std/vector>
#include <error/Exceptions.h>
#include <utils/Foreach.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup ExceptionModule
 * This Exception indicates an error within a network operation e.g HTTP 404.
 */
MIRA_DEFINE_SERIALIZABLE_EXCEPTION(XNetworkError, XRuntime)

/**
 * @ingroup ExceptionModule
 * This Exception indicates that a resource could not be found.
 */
MIRA_DEFINE_SERIALIZABLE_EXCEPTION(XHTTP404, XNetworkError)

///////////////////////////////////////////////////////////////////////////////

namespace gitlab {

/** These types represent the json responses from the gitlab api.
*/

struct Error {
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("error", error, "Error Type");
		r.member("error_description", error_description, "Error description", "");
	}

	std::string error;
	std::string error_description;
};

struct Message {
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("message", message, "A message without a bottle. :/");
	}

	std::string message;
};

struct AccessToken {
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("access_token", access_token, "");
		r.member("created_at", created_at, "");
		r.member("refresh_token", refresh_token, "");
		r.member("scope", scope, "");
		r.member("token_type", token_type,"");
	}

	std::string authorizationHeader() {
		return "Authorization: " + token_type + " " + access_token;
	}

	std::string access_token;
	int created_at;
	std::string refresh_token;
	std::string scope;
	std::string token_type;
};

struct Namespace {
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("full_path", full_path, "");
		r.member("id", id, "");
		r.member("kind", kind, "");
		r.member("name", name, "");
		try {
			r.member("parent_id", parent_id, "");
		} catch(...) {
			parent_id = -1;
		}
		r.member("path", path, "");
		r.member("web_url", web_url, "Visit namespace in browser.");
	}

	std::string full_path;
	int  id;
	std::string kind;
	std::string name;
	int parent_id;
	std::string path;
	std::string web_url;
};

struct ProjectStatistics {
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("commit_count", commit_count, "Number of commits", -1);
	}
	long commit_count;
};

struct Project {
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("name", name, "Name of the project.");
		r.member("http_url_to_repo", http_url_to_repo, "clone url via hhtp");
		r.member("ssh_url_to_repo", ssh_url_to_repo, "clone url via ssh");
		r.member("web_url", web_url, "Visit project in browser.");
		r.member("star_count", star_count, "Stars on gitlab.", 0);
		r.member("namespace", ns, "The namespace this projects resides in.");
		r.member("id", id, "Project id number");
		r.member("default_branch", default_branch, "Default branch");
	}

	std::string name;
	std::string http_url_to_repo;
	std::string ssh_url_to_repo;
	std::string web_url;
	int star_count;
	Namespace ns;
	int id;
	std::string default_branch;
	//ProjectStatistics stats;
};

struct Group {
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("id", id, "Project ID");
		r.member("name", name, "");
		r.member("path", path, "");
		r.member("description", description, "");
		r.member("visibility", visibility, "");
		r.member("share_with_group_lock", share_with_group_lock, "");
		r.member("require_two_factor_authentication", require_two_factor_authentication, "");
		r.member("two_factor_grace_period", two_factor_grace_period, "");
		r.member("project_creation_level", project_creation_level, "");
		r.member("auto_devops_enabled", auto_devops_enabled, "", false);
		r.member("subgroup_creation_level", subgroup_creation_level, "");
		r.member("emails_disabled", emails_disabled, "", false);
		r.member("lfs_enabled", lfs_enabled, "large file system enabled", false);
		r.member("avatar_url", avatar_url, "", "");
		r.member("web_url", web_url, "");
		r.member("request_access_enabled", request_access_enabled, "");
		r.member("full_name", full_name, "");
		r.member("full_path", full_path, "");
		r.member("file_template_project_id", file_template_project_id, "", -1);
		r.member("parent_id", parent_id, "", -1);
	}

	int id;
	std::string name;
	std::string path;
	std::string description;
	std::string visibility;
	bool share_with_group_lock;
	bool require_two_factor_authentication;
	int two_factor_grace_period;
	std::string project_creation_level;
	bool auto_devops_enabled;
	std::string subgroup_creation_level;
	bool emails_disabled;
	bool lfs_enabled;
	std::string avatar_url;
	std::string web_url;
	bool request_access_enabled;
	std::string full_name;
	std::string full_path;
	int file_template_project_id;
	int parent_id;
};

struct Tag {
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("name", name, "tag name", "");
		r.member("message", message, "tag message", "");
	}

	// the api can return a commit object not used/needed yet
	std::string name;
	std::string message;
};

struct Tree {
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("id", id, "id string");
		r.member("name", name, "file/directory name");
		r.member("type", type, "git type e.g. tree, blob etc.");
		r.member("path", path, "path to object");
		r.member("mode", mode, "file mode");
	}

	std::string id;
	std::string name;
	std::string type;
	std::string path;
	std::string mode;
};

/** @brief Helper class to access the Gitlab API Version 4
 *  By convention we end all URLs without a trailing slash.
 */ 
class GitlabAPI {
public:
	/** Create a instance of GitlabAPI.
	 *  @param[in] host, a server hosting a gitlab instance
	 */
	GitlabAPI(const std::string& host);

	/** @brief create a copy of right hand side.
	 *  All member variables will be copied, except for curl_handle.
	 *  The curl_handle of left hand side will be new initialized with curl_easy_init()
	 *  to allow simultaneous usage of both instances.
	 */
	GitlabAPI(const GitlabAPI&);
	GitlabAPI& operator=(const GitlabAPI& rhs);
	~GitlabAPI();

	/** @brief Sending a request to the instance to obtain an access token.
	 *  @param[in] username
	 *  @param[in] password
	 */
	bool login(const std::string& username = "", const std::string& password = "");

	/** You can login to gitlab via an Application Token.
	 *  Provide the token with this function to login. The token will _not_ be
	 *  saved beyond the scope of this object.
	 *  Generate a token via Settings > Access Token
	 *  @param[in] accessToken, a valid gitlab Application token.
	 */
	bool provideApplicationToken(const std::string& accessToken);

	/** @brief list all projects. The number of projects depend on their visibility
	 *  for you.
	 *  @param[in] perPage, number of results per page
	 *  @return a vector containing all received Project objects
	 */
	std::vector<Project> projects(unsigned int perPage = 100);

	/** @brief list all projects in the group given by groupID.
	 *  The number of projects depend on their visibility for you.
	 *  @param[in] groupID, id of the group to search in.
	 *  @param[in] perPage, number of results per page. Only page 1 is requested.
	 *  @param[in] subgroups, if set to true the packets of any subgroup will be also returned
	 *  @return a vector containing all received Project objects
	 */
	std::vector<Project> projectsOfGroup(unsigned int groupID, bool subgroups = false, unsigned int perPage = 100);

	/** @brief list all namespaces.
	 *  @param[in] search, the search string will be sent to the api as search criteria.
	 *  @return a vector containing all received Namespace objects
	 */
	std::vector<Namespace> namespaces(const std::string& search = "");

	/** 
	 * @brief Get a group object for a given path
	 * @param[in] path path of the group you are looking for
	 * @return a GitLab API group object
	 */
	Group group(const std::string& path);

	/** @brief list all groups. This request is limited to 100 groups per page.
	 *  @param[in] search, the search string will be sent to the api as search criteria.
	 *  @param[in] all, set ?all_available=true, this will override search
	 *  @param[in] page, the page you want to see, needed if you want to see more than 100 results.
	 *  @return a vector containing all received Group objects
	 */
	std::vector<Group> groups(const std::string& search = "", bool all = false, unsigned int page = 1);

	/** 
	 * @brief Get all subgroups of a group
	 * @param[in] path path of the group whose subgroups should be returned
	 * @return a vector of all the subgroups of the group
	 */
	std::vector<Group> subgroups(const std::string& path);

	/** @brief Get the raw file content for a file in a project
	 *  @param[in] projectId, the id of the project
	 *  @param[in] path, the path to the file inside the project (e.g. src/main.cpp)
	 *  @return the content of the file if it exists, otherwise empty
	 */
	std::string rawFile(unsigned int projectId, std::string path, const std::string& ref);

	/** @brief Get a list of files and directories in a project.
	 *  @param[in] projectId, the id of the project
	 *  @param[in] path, a path to a sub-directory
	 *  @return a vector with all elements in the directory
	 */
	std::vector<Tree> listFiles(unsigned int projectId, const std::string& path, const std::string& ref);

	/** @brief Get a list of tags that are available for a project.
	 *  @param[in] projectId, the id of the project
	 *  @param[in] search, an optional search criteria
	 *  @return a vector containing all received Tag objects
	 */
	std::vector<Tag> tags(unsigned int projectId, const std::string& search = "");

	Url instanceUrl() { return mInstanceUrl; }

	bool loggedIn() { return mLoggedIn; }

	/** @brief If verbose is true, the verbose option is
	 *  set on the underlying curl_handle, so HTTP-Headers will
	 *  be printed to command line.
	 *  @param[in] verbose, if true print more information.
	 */
	void setVerbose(bool verbose=true);

	/** @brief If set to false, the underlying curl_handle will _not_
	 *  validate the server certificate.
	 *  @param[in] check, if false certificate check will be bypassed
	 */
	void checkSSLCerts(bool check=true);

private:
	/** @brief Make a request at "https://" + instance + api-endpoint + path with given headers.
	 */
	std::string request(const std::string& path, long& http_status, const curl_slist* header = NULL);

	/**
	 * @brief handle paginated requests to the GitLab API
	 *
	 * API doc: https://docs.gitlab.com/ee/api/#pagination
	 *
	 * @param[out] oVec this vector is used to collect the results
	 * @param[in] requestPath the original, unpaginated API request
	 * @param[in] perPage the number of results per page. The default is 20.
	 */
	template<typename T>
	void requestPaginated(std::vector<T>& oVec, const std::string& requestPath, unsigned int perPage = 20);

	/** @brief Make a POST request at "https://" + instance +path with given headers.
	 */
	std::string post(const std::string& path, const std::string& data, const curl_slist* header = NULL);

	inline std::string apiVersion() const { return "/api/v4"; }

	/** @brief replace / in path with %2F to pass it as argument in url
	 */
	inline std::string encodePath(const std::string& path) {
		std::string encodedPath;

		foreach(auto c, path) {
			if(c == '/') {
				encodedPath += ("%2F");
			} else {
				encodedPath += c;
			}
		}
		return encodedPath;
	}

	Url mInstanceUrl;
	bool mLoggedIn;

	CURL* mCurlHandle;
	AccessToken mAccessToken;
};

/** @brief this is used to fill the buffer given by userdata. The function is a callback argument
*  for libcurl to store data received by a request. If user data is a nullptr,
*  the data received by curl will be written in request_buffer;
*/
size_t write_callback(char* ptr, size_t size, size_t nmemb, void* userdata);

///////////////////////////////////////////////////////////////////////////////

} // namespace gitlab
} // namespace mira

#endif
