/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file IPTU.h
 *    An interface class for objects with a PTU.
 *
 * @author Benedict Stephan
 * @date   2021/07/02
 */

#ifndef _MIRA_IPTU_H_
#define _MIRA_IPTU_H_

#include <transform/Pose.h>

namespace mira { namespace robot {

///////////////////////////////////////////////////////////////////////////////

/// An interface class for objects with a PTU
class IPTU
{
public:
	/** @name Destructor and reflection */
	//@{

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.interface("IPTU");

		r.method("setPan", &IPTU::setPan, this, "Set the pan angle.",
		         "pan", "pan angle", 0.);
		r.method("getPan", &IPTU::getPan, this, "Get the current pan angle.");
		r.method("addPan", &IPTU::addPan, this, "Adds value to current pan angle.",
		         "dPan", "pan angle increment", 0.);

		r.method("setTilt", &IPTU::setTilt, this, "Set the tilt angle.",
		         "tilt", "tilt angle", 0.);
		r.method("getTilt", &IPTU::getTilt, this, "Get the current tilt angle.");
		r.method("addTilt", &IPTU::addTilt, this, "Adds value to current tilt angle.",
		         "dTilt", "tilt angle increment", 0.);

		r.method("lookAt", &IPTU::lookAt, this,
		         "Moves the PTU to look at the specified target in the given target frame.",
		         "target", "target position", Eigen::Vector3f(1., 2., 3.),
		         "targetFrame", "transformation frame id for target position", "/maps/MapFrame",
		         "tolerance", "tolerance in deg. If difference between current view direction "
		                      "and desired direction is closer than this threshold, the PTU is not moved.", 30);

		r.method("halt", &IPTU::halt, this, "Stops all movement.");
	}

	virtual ~IPTU() {}

	//@}

public:
	/** @name IPTU interface definition */
	//@{
	virtual void setPan(double pan) = 0;
	virtual double getPan() = 0;
	virtual void addPan(double dPan) = 0;

	virtual void setTilt(double tilt) = 0;
	virtual double getTilt() = 0;
	virtual void addTilt(double dTilt) = 0;

	virtual void lookAt(const Eigen::Vector3f& target, const std::string& targetFrame, double tol) = 0;

	virtual void halt() = 0;

	//@}
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace

#endif
