/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationPlotCurveView.C
 *    .
 *
 * @author Christian Vollmer
 * @date   2013/06/04
 */

#include <boost/format.hpp>

#include <views/VisualizationPlotCurveView.h>

#include <QMenu>
#include <QFileDialog>
#include <QPen>

#include <qwt_scale_draw.h>
#if (QWT_VERSION >= 0x060100)
#include <qwt_plot_canvas.h>
#endif

#include <serialization/Serialization.h>

#include <visualization/VisualizationPlot.h>
#include <visualization/plot/VisualizationPlotCurveJSON.h>

#include <widgets/QtUtils.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

class DefaultPlotCurveVisualization : public VisualizationPlotCurveJSON<void>
{
MIRA_META_OBJECT(DefaultPlotCurveVisualization,
                 ("Name", "DefaultVisualization")
                 ("Description", "Plots (elements of) data based on its JSON representation")
                 ("Category", "General"))
public:

	typedef VisualizationPlotCurveJSON<void> Base;

	DefaultPlotCurveVisualization() : Base("Channel") {}
};

//////////////////////////////////////////////////////////////////////////////


class TimeScaleDraw: public QwtScaleDraw
{
public:

	virtual QwtText label(double v) const
	{
		uint64 t = (uint64)v;
		std::string txt;
		try {
			Duration tod = Time::fromUnixNS(t).toLocal().time_of_day();
			txt = (boost::format("%02d:%02d:%02d.%03d") % (tod.hours()) %
					(tod.minutes()) % (tod.seconds()) % (tod.milliseconds())).str();
		}
		catch (const std::out_of_range&) {
		}

		return QString(txt.c_str());
	}
};

//////////////////////////////////////////////////////////////////////////////

class TimeScaleZoomer : public QwtPlotZoomer
{

#if (QWT_VERSION >= 0x060100)
	typedef QWidget CanvasType;
#else
	typedef QwtPlotCanvas CanvasType;
#endif

#if (QWT_VERSION >= 0x060000)
	typedef QPointF PointType;
	#define TrackerTextMethod trackerTextF
#else
	typedef QwtDoublePoint PointType;
	#define TrackerTextMethod trackerText
#endif

	// workaround for protected QwtPlotPicker::invTransform
	class RightPicker : public QwtPlotPicker
	{
	public:
		RightPicker(CanvasType* canvas) : QwtPlotPicker(QwtPlot::xBottom, QwtPlot::yRight, canvas) {}
		using QwtPlotPicker::invTransform;
	};

public:
	TimeScaleZoomer(CanvasType* canvas, bool doReplot = true) :
		QwtPlotZoomer(QwtPlot::xBottom, QwtPlot::yLeft, canvas, doReplot), mRightPicker(canvas) {}

	virtual QwtText TrackerTextMethod(const PointType& value) const
	{
		uint64 t = (uint64)value.x();
		std::string txt;
		try {
			Duration tod = Time::fromUnixNS(t).toLocal().time_of_day();
			txt = (boost::format("%02d:%02d:%02d.%03d : ") % (tod.hours()) %
				(tod.minutes()) % (tod.seconds()) % (tod.milliseconds())).str();
		}
		catch (const std::out_of_range&){
			txt = "(out of time) : ";
		}

		if (plot()->axisEnabled(QwtPlot::yRight)) {
			PointType rightValue = mRightPicker.invTransform(transform(value));
			if (plot()->axisEnabled(QwtPlot::yLeft)) {
				txt += (boost::format("%d (l), %d (r)") % value.y() % rightValue.y()).str();
			} else {
				txt += (boost::format("%d") % rightValue.y()).str();
			}
		} else {
			txt += (boost::format("%d") % value.y()).str();
		}

		return QString(txt.c_str());
	}
protected:
	RightPicker mRightPicker;
};

///////////////////////////////////////////////////////////////////////////////

VisualizationPlotCurveView::VisualizationPlotCurveView()
{
}

const Class& VisualizationPlotCurveView::supportedVisualizationClass() const
{
	return VisualizationPlotCurve::CLASS();
}

const Class& VisualizationPlotCurveView::defaultVisualizationClass() const
{
	return DefaultPlotCurveVisualization::CLASS();
}

VisualizationPlotView::UI* VisualizationPlotCurveView::createUI()
{
	UI* ui = new UI(this);
	ui->setAxisScaleDraw(QwtPlot::xBottom, new TimeScaleDraw());
	return ui;
}

void VisualizationPlotCurveView::populateDefaultVisualizations()
{
	createDefaultVisualization("mira::PlotCurveGridVisualization", "Grid");
}

//////////////////////////////////////////////////////////////////////////////

VisualizationPlotCurveView::UI::UI(VisualizationPlotCurveView* parent) :
	VisualizationPlotView::UI(parent)
{
	zoomer = new TimeScaleZoomer(canvas());
	zoomer->setRubberBandPen( QColor(Qt::black ));
	zoomer->setTrackerPen( QColor(Qt::black));
	zoomer->setTrackerMode(QwtPicker::AlwaysOn);
#if (QWT_VERSION >= 0x060000)
	zoomer->setMousePattern(QwtEventPattern::MouseSelect1, Qt::LeftButton, Qt::ControlModifier);
	zoomer->setMousePattern(QwtEventPattern::MouseSelect2, Qt::MidButton);
	zoomer->setMousePattern(QwtEventPattern::MouseSelect3, Qt::NoButton);
	zoomer->setKeyPattern(QwtEventPattern::KeyHome, Qt::Key_R, Qt::ControlModifier);
	connect(zoomer, SIGNAL(zoomed(const QRectF&)), this, SLOT(zoomed(const QRectF&)));
#else
	zoomer->setMousePattern(0, Qt::LeftButton, Qt::ControlModifier);
	zoomer->setMousePattern(1, Qt::MidButton);
	zoomer->setMousePattern(2, Qt::NoButton);
	connect(zoomer, SIGNAL(zoomed(const QwtDoubleRect&)), this, SLOT(zoomed(const QwtDoubleRect&)));
#endif
}

void VisualizationPlotCurveView::UI::populatePopupMenu(QMenu& menu)
{
	menu.addAction("Save as CSV");
}

void VisualizationPlotCurveView::UI::onPopupMenuAction(QAction* action)
{
	if (action->text() == "Save as CSV")
	{
		saveAsCSV();
	}
}

void VisualizationPlotCurveView::UI::saveAsCSV()
{
	QStringList extList;
	extList << "csv" << "txt";
	QString name = QtUtils::getSaveFileName(NULL, QtUtils::tr("Save as CSV"), ".",
	                                        QtUtils::tr("Data (*.csv)")+";;"+QtUtils::tr("Text (*.txt)"),
	                                        NULL, QFileDialog::DontUseNativeDialog, extList);

	if(name.isNull() || name.isEmpty())
		return;
	Path path(name.toStdString());

	auto& vs = vis->getVisualizations();
	foreach(auto& v, vs)
	{
		VisualizationPlotCurve* vp = dynamic_cast<VisualizationPlotCurve*>(v);
		foreach (auto item, vp->getPlotItems())
		{
			QwtPlotCurve* curve = dynamic_cast<QwtPlotCurve*>(item);
			if (curve == NULL)
				// Skip elements, that don't represent a curve (e.g. the grid).
				continue;
			std::string id = curve->title().text().toStdString();
			std::replace(id.begin(), id.end(), '/','.');
			std::string vname = (path.parent_path() / (path.stem().string() + id + path.extension().string())).string();
			std::ofstream of(vname.c_str());
#if (QWT_VERSION >= 0x060000)
			QwtArraySeriesData<QPointF>* array = (QwtArraySeriesData<QPointF>*)curve->data();
			for (std::size_t i=0; i<array->size(); ++i)
			{
				QPointF sample = array->sample(i);
				of << Time::fromUnixNS((uint64)sample.x()).toLocal().toUnixNS() << "," << sample.y() << std::endl;
			}
#else
			QwtArrayData& array = (QwtArrayData&)curve->data();
			for (std::size_t i=0; i<array.size(); ++i)
				of << Time::fromUnixNS((uint64)array.xData()[i]).toLocal().toUnixNS() << "," <<  array.yData()[i] << std::endl;
#endif
		}
	}
}

#if (QWT_VERSION >= 0x060000)
void VisualizationPlotCurveView::UI::zoomed(const QRectF &rect)
#else
void VisualizationPlotCurveView::UI::zoomed(const QwtDoubleRect &rect)
#endif
{
	if (rect == zoomer->zoomBase())
	{
		// enable axis auto scale
		setAxisAutoScale(QwtPlot::xBottom);
		setAxisAutoScale(QwtPlot::yLeft);
		setAxisAutoScale(QwtPlot::yRight);

		// replot
		QwtPlot::replot();

		// get axis min and max values to calculate a rectangle for initialize the zoomer
#if (QWT_VERSION >= 0x060100)
		double xLowerBound = axisScaleDiv(QwtPlot::xBottom).lowerBound();
		double xUpperBound = axisScaleDiv(QwtPlot::xBottom).upperBound();
		double yLowerBound = axisScaleDiv(QwtPlot::yLeft).lowerBound();
		double yUpperBound = axisScaleDiv(QwtPlot::yLeft).upperBound();
#elif (QWT_VERSION >= 0x050200)
		double xLowerBound = axisScaleDiv(QwtPlot::xBottom)->lowerBound();
		double xUpperBound = axisScaleDiv(QwtPlot::xBottom)->upperBound();
		double yLowerBound = axisScaleDiv(QwtPlot::yLeft)->lowerBound();
		double yUpperBound = axisScaleDiv(QwtPlot::yLeft)->upperBound();
#else
		double xLowerBound = axisScaleDiv(QwtPlot::xBottom)->lBound();
		double xUpperBound = axisScaleDiv(QwtPlot::xBottom)->hBound();
		double yLowerBound = axisScaleDiv(QwtPlot::yLeft)->lBound();
		double yUpperBound = axisScaleDiv(QwtPlot::yLeft)->hBound();
#endif

		// initialize the zoomer with auto scale values
		zoomer->setZoomBase( QwtDoubleRect(xLowerBound, yUpperBound, yUpperBound-yLowerBound, xUpperBound-xLowerBound) );
		// reset zoom level to 0
		zoomer->setZoomBase(false);
	}
}

///////////////////////////////////////////////////////////////////////////////

}

///////////////////////////////////////////////////////////////////////////////

MIRA_CLASS_SERIALIZATION( mira::VisualizationPlotCurveView, mira::VisualizationViewTransformable);
MIRA_CLASS_SERIALIZATION( mira::DefaultPlotCurveVisualization, mira::VisualizationPlotCurve);

///////////////////////////////////////////////////////////////////////////////
