/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationPlotCurveJSON.h
 *    Macros for easy creation of JSON based plot visualizations.
 *
 * @author Tim Langner
 * @date   2011/07/26
 */

#ifndef _MIRA_VISUALIZATIONPLOTCURVEJSON_H_
#define _MIRA_VISUALIZATIONPLOTCURVEJSON_H_

#include <qwt_plot.h>
#if (QWT_VERSION >= 0x060100)
#include <qwt_point_data.h>
#endif

#include <serialization/Serialization.h>
#include <visualization/VisualizationPlotCurveBasic.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

template<typename T>
class VisualizationPlotCurveJSON : public VisualizationPlotCurveBasic<T>
{
public:
	typedef VisualizationPlotCurveBasic<T> Base;

	VisualizationPlotCurveJSON(const std::string& channelName) :
		Base(channelName), mStringMapAsObject(true)
	{
		setMember("");
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Base);
		r.property("Member", member,
		           setter<std::string>(&VisualizationPlotCurveJSON<T>::setMember, this),
		           "Member of the data to plot (e.g X for Pose channel). "
		           "Use [index] for array access and [row][col] for Eigen matrices",
		           serialization::IgnoreMissing());
		r.property("StringMapAsObject", mStringMapAsObject, "Serialize any map<string, T> to json object "
		           "(enables simple access to map items by key)", true);
	}

	virtual QString getPlotTitle()
	{
		QString title;
		title = this->mChannel.getID().c_str();
		if (!member.empty())
			title += QString("(") + member.c_str() + QString(")");
		return title;
	}

#if (QWT_VERSION >= 0x060000)
	class CurveData: public QwtArraySeriesData<QPointF>
	{
	public:
		CurveData()
		{
		}

		virtual QRectF boundingRect() const
		{
			if ( d_boundingRect.width() < 0.0 )
				d_boundingRect = qwtBoundingRect( *this );

			return d_boundingRect;
		}

		inline void append( const QPointF &point, uint64 history)
		{
			d_samples += point;
			if (history > 0)
			{
				uint64 d = d_samples.back().x() - d_samples.front().x();
				while (d>history)
				{
					d_samples.pop_front();
					d = d_samples.back().x() - d_samples.front().x();
				}
			}
			d_boundingRect = QRectF( 0.0, 0.0, -1.0, -1.0 );
		}

		void clear()
		{
			d_samples.clear();
			d_samples.squeeze();
			d_boundingRect = QRectF( 0.0, 0.0, -1.0, -1.0 );
		}
	};
#endif

	void setMember(const std::string& m)
	{
		bool useTitleAsName = ((this->getName() == getPlotTitle().toStdString())
							|| (this->getName() == this->mChannel.getID()));
		member = m;
		if (this->mCurve)
		{
#if (QWT_VERSION >= 0x060000)
			this->mCurve->setData(new CurveData());
#else
			this->mCurve->setData(QwtArray<QwtDoublePoint>());
#endif
			this->mCurve->setTitle(getPlotTitle());
			if (useTitleAsName)
				this->setName(getPlotTitle().toStdString());
			this->getSite()->updateAxes();
		}
	}

	void plot(json::Value& value, const Time& timestamp)
	{
		double plotData = json::getNumberElement(value, member);
		uint64 history = this->getSite()->getMaxPlotHistory().totalNanoseconds();
		this->ok("Plot");
#if (QWT_VERSION >= 0x060000)
		CurveData* data = (CurveData*)this->mCurve->data();
		data->append(QPointF(timestamp.toUnixNS(), plotData), history);
		this->mCurve->itemChanged();
#else
		QwtArrayData& array = (QwtArrayData&)this->mCurve->data();
		QwtArray<double> xArray, yArray;
		if (array.size() > 0)
		{
			xArray = array.xData();
			yArray = array.yData();
		}
		xArray.append(timestamp.toUnixNS());
		yArray.append(plotData);
		if (history > 0)
		{
			uint64 d = xArray.back() - xArray.front();
			while (d>history)
			{
				xArray.pop_front();
				yArray.pop_front();
				d = xArray.back() - xArray.front();
			}
		}
		this->mCurve->setData(xArray, yArray);
#endif
		this->getSite()->getPlot()->replot();
	}

	virtual void dataChanged(ChannelRead<T> data)
	{
		if (!this->getSite()->isAcceptingUpdates())
			return;

		try	{
			json::Value v;
			try {
				JSONSerializer s(true,
				                 this->mStringMapAsObject ? JSONSerializer::STRING_MAP_AS_OBJECT
				                                          : JSONSerializer::STANDARD);
				data.readJSON(v, s);
			}
			catch(XNotImplemented&) {
				data.readJSON(v);
			}

			this->plot(v, data.getTimestamp());
			this->ok("Plot");
		}
		catch(Exception& ex) {
			this->error("Plot", ex.message());
		}
		catch(std::exception& ex) {
			this->error("Plot", ex.what());
		}
	}

	virtual void dataIntervalChanged(ChannelReadInterval<T> data)
	{
		typename ChannelReadInterval<T>::const_iterator it = data.begin();
		for (; it !=data.end(); it++) {
			ChannelRead<T> r = it;
			dataChanged( r );
		}
	}

protected:
	std::string member;
	bool mStringMapAsObject;
};

/**
 * Use this macro to create a  text visualization for a data type that
 * supports serialization using JSON serializer.
 * className specifies the class name
 * type is the data type of the channel
 * channel is the name of the channel property
 * name is the name of the visualization
 * description describes the visualization
 * category is the category of the visualization
 */
#define MIRA_JSON_PLOT_CURVE_VISUALIZATION(className, type, channel, name, description, category) \
class className : public VisualizationPlotCurveJSON<type>  \
{                                                     \
	MIRA_META_OBJECT(className,                       \
		("Name", name)                                \
		("Description", description)                  \
		("Category", category))                       \
public:                                               \
	className() :                                     \
		VisualizationPlotCurveJSON<type>(channel)     \
	{}                                                \
};

//////////////////////////////////////////////////////////////////////////////

}

#endif
