/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationPlot.h
 *    TODO Add description.
 *
 * @author Tim Langner
 * @date   2011/07/26
 */

#ifndef _MIRA_VISUALIZATIONPLOT_H_
#define _MIRA_VISUALIZATIONPLOT_H_

#include <geometry/Point.h>

#include <PlotVisualizationExports.h>

#include <visualization/Visualization.h>

class QwtPlot;
class QwtPlotItem;

namespace mira {

//////////////////////////////////////////////////////////////////////////////

class IVisualizationPlotSite : public IVisualizationSiteTransformable
{
public:

	virtual QwtPlot* getPlot() = 0;
	virtual Duration getMaxPlotHistory() = 0;

	/**
	 * To be called when axis assignment changed (e.g. axes need to be shown/hidden)
	 */
	virtual void updateAxes() = 0;

	virtual bool isAcceptingUpdates() { return true; }
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Base class for plot visualizations. Intended only for internal use.
 */
class VisualizationPlot : public Visualization
{
//	MIRA_OBJECT(VisualizationPlot)

protected:

	typedef std::vector<QwtPlotItem*> QwtPlotItems;

public:

	VisualizationPlot();
	virtual ~VisualizationPlot() {}

public:

	/**
	 * Is called by the visualization view when the visualization is attached.
	 * The default implementation stores the pointer to the IVisualizationPlotSite
	 * interface that can be retrieved using the getSite() method for later use.
	 * Moreover, it calls the setupScene() method.
	 * You should never need to overwrite this method in derived classes. Use
	 * the setupScene() method instead. However, if you overwrite this method
	 * you SHOULD call this default implementation.
	 *
	 * The passed interface must be of the type IVisualizationPlotSite.
	 */
	virtual void init(IVisualizationSite* site);

public:

	/**
	 * Returns a pointer to the IVisualizationPlotSite interface that was set
	 * in the init() method.
	 * Or returns NULL, if init() was not yet called.
	 */
	virtual IVisualizationPlotSite* getSite();

public:

	/**
	 * Must be implemented in derived classes. Is called within the
	 * visualization's init() method. You can use this method to create
	 * your scene and scene objects.
	 * The passed IVisualizationPlotSite interface is the same that is
	 * returned by getSite().
	 */
	virtual void setupScene(IVisualizationPlotSite* site) = 0;

	/**
	 * Must be implemented in derived class to clear the plotted item (curve)
	 */
	virtual void clear() = 0;

	/**
	 * Must be implemented in derived class to return the used curves or other items
	 */
	virtual QwtPlotItems getPlotItems() = 0;

	/**
	 * Must be implemented in derived class to return the y axis affiliation (left/right)
	 */
	virtual int getYAxis() = 0;

	/**
	 * Must be implemented in derived class to return the title
	 */
	virtual QString getPlotTitle() = 0;

private:

	IVisualizationPlotSite* mSite;

};

///////////////////////////////////////////////////////////////////////////////

/**
 * Abstract base class for all plot visualization that are used by
 * VisualizationPlotCurveView.
 */
class MIRA_GUI_PLOTVISUALIZATION_EXPORT VisualizationPlotCurve :
	public VisualizationPlot
{
	MIRA_OBJECT(VisualizationPlotCurve)
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Abstract base class for all plot visualization that are used by
 * VisualizationPlotHistogramView.
 */
class MIRA_GUI_PLOTVISUALIZATION_EXPORT VisualizationPlotBarChart :
	public VisualizationPlot
{
	MIRA_OBJECT(VisualizationPlotBarChart)
};

}

#endif
