/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Z16TableCodec.h
 *    Codec for lossless compression of 16 bit depth images.
 *
 * @author Erik Einhorn
 * @date   2011/10/01
 */

#ifndef _MIRA_Z16TABLECODEC_H_
#define _MIRA_Z16TABLECODEC_H_

#include <serialization/BinarySerializerCodec.h>

#include <codec/CommonCodecsExports.h>

namespace mira { namespace codec {

///////////////////////////////////////////////////////////////////////////////

/**
 * Codec for lossless compression of 16 bit depth images that 
 * can be used together with a BinarySerializer.
 *
 * The codec encodes differences between consecutive pixels. Consecutive zeros
 * are stored using a form of run length encoding. Small difference values
 * are encoded using 4 bits. Larger values using more bits.
 */
class MIRA_COMMONCODECS_EXPORT Z16TableCodec : public BinarySerializerCodec
{
MIRA_META_OBJECT(Z16TableCodec,
		("FOURCC",      "Z16T")
		("Name",        "Z16T")
		("Description", "Lossless compression of 16bit depth images")
		("Category"   , "Image"))

public:

	/**
	 * As parameter you can specify the max. value, that may occur in the
	 * data. For OpenKinect depth data 4096 is sufficient, since the original
	 * 11-bit encoding is used. In OpenNI, however, the data is given in as
	 * 16-bit integers in millimeters with a maximum value of 9757. Thus, we
	 * use this as default value here.
	 *
	 * If the codec is used with depth images whose values are larger than the
	 * specified maxValue, the behavior is undefined and the resulting encoded
	 * data stream is corrupted. The encoding can end up in a segfault. For
	 * performance reasons, it is not checked if the images contain values
	 * below the specified maxValue.
	 */
	Z16TableCodec();

	virtual ~Z16TableCodec();

	template <typename Reflector>
	void reflect(Reflector &r)
	{
		r.property("MaxValue", mMaxValue, "(Promised, not checked) max depth value in images", 9757);
	}

public:

	/// Implementation of BinarySerializerCodec
	virtual TypeId getSupportedTypeId() const;

protected:

	/// Implementation of BinarySerializerCodec
	virtual Buffer<uint8> encodeBuffer(TypedVoidConstPtr objectPtr);

	/// Implementation of BinarySerializerCodec
	virtual void decodeBuffer(const Buffer<uint8>& data, TypedVoidPtr ioObjectPtr);

private:

	uint16* mTable;
	uint16 mMaxValue;
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace

#endif
