###############################################################################
# Copyright (C) 2012 by
#   MetraLabs GmbH (MLAB), GERMANY
# and
#   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
# All rights reserved.
#
# Contact: info@mira-project.org
#
# Commercial Usage:
#   Licensees holding valid commercial licenses may use this file in
#   accordance with the commercial license agreement provided with the
#   software or, alternatively, in accordance with the terms contained in
#   a written agreement between you and MLAB or NICR.
#
# GNU General Public License Usage:
#   Alternatively, this file may be used under the terms of the GNU
#   General Public License version 3.0 as published by the Free Software
#   Foundation and appearing in the file LICENSE.GPL3 included in the
#   packaging of this file. Please review the following information to
#   ensure the GNU General Public License version 3.0 requirements will be
#   met: http://www.gnu.org/copyleft/gpl.html.
#   Alternatively you may (at your option) use any later version of the GNU
#   General Public License if such license has been publicly approved by
#   MLAB and NICR (or its successors, if any).
#
# IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
# INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
# THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
# "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
# BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
# FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
# ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
# PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
#
###############################################################################
#
# Build a RPM package for a MIRA package using cmake and rpmbuild.
#
# This cmake script file needs the following variables to be defined:
#
#   MIRA_INSTALL_PACKAGE:          Name of the package
#   MIRA_INSTALL_VERSION:          API-Version
#   MIRA_INSTALL_RELEASE:          Build version
#   MIRA_INSTALL_AUTHOR:           Author(s) of the package
#   MIRA_INSTALL_DESCRIPTION_FILE: File name of the description
#   MIRA_INSTALL_CHANGELOG_FILE:   File name of the changelog
#   MIRA_INSTALL_COMPONENTS:       Components of this package
#   MIRA_INSTALL_REQUIREMENTS:     Requirements
#
#   MIRA_ROOT_DIR:                 Path to MIRA to locate some files.
#   BUILD_BINARY_DIR:              Current build environment
#
# Author: Christian Martin
#
###############################################################################

SET(INSTALL_ROOT          "${BUILD_BINARY_DIR}/install_root")
SET(CMAKEINSTALL_PATH     "${INSTALL_ROOT}/_install_")
SET(RPM_SPEC_PATH         "${INSTALL_ROOT}/SPEC")
SET(RPM_BUILD_PATH        "${INSTALL_ROOT}/BUILD")

SET(MIRA_INSTALL_PREFIX    "/opt/MIRA")

###############################################################################
# cleanup and re-create all directories

FILE(REMOVE_RECURSE ${INSTALL_ROOT})

MAKE_DIRECTORY(${CMAKEINSTALL_PATH})
MAKE_DIRECTORY(${RPM_SPEC_PATH})
MAKE_DIRECTORY(${RPM_BUILD_PATH})

###############################################################################
# Install all cmake components for the current package

FOREACH(component ${MIRA_INSTALL_COMPONENTS})
	EXEC_PROGRAM(${CMAKE_COMMAND} ARGS
		-D "BUILD_TYPE=Release"
		-D "CMAKE_INSTALL_DO_STRIP=1"
		-D "CMAKE_INSTALL_COMPONENT=${component}"
		-D "CMAKE_INSTALL_PREFIX=${CMAKEINSTALL_PATH}/${MIRA_INSTALL_PREFIX}"
		-P "${BUILD_BINARY_DIR}/cmake_install.cmake"
	)
ENDFOREACH()

###############################################################################
# Check if there is at least one file installed. Otherwise build-rpm will fail!

FILE(GLOB_RECURSE FileList "${CMAKEINSTALL_PATH}/${MIRA_INSTALL_PREFIX}/*")
LIST(LENGTH FileList FileListLen)
IF(${FileListLen} EQUAL 0)
	MESSAGE(STATUS "*** Skip building RPM for package ${MIRA_INSTALL_PACKAGE}, since there are no files to install.")
	RETURN()
ENDIF()

###############################################################################
# get some more system information

EXEC_PROGRAM(uname ARGS -m OUTPUT_VARIABLE SYSTEM_ARCH)

###############################################################################
# create the RPM spec file

SET(SPEC_BUILDROOT   ${INSTALL_ROOT})
SET(SPEC_NAME        MIRA-${MIRA_INSTALL_PACKAGE})
SET(SPEC_VERSION     ${MIRA_INSTALL_VERSION})
SET(SPEC_RELEASE     ${MIRA_INSTALL_RELEASE})
SET(SPEC_LICENSE     ${MIRA_INSTALL_LICENSE})
SET(SPEC_VENDOR      ${MIRA_INSTALL_VENDOR})
SET(SPEC_RPMFILENAME MIRA-${MIRA_INSTALL_PACKAGE}-${MIRA_INSTALL_VERSION}-${MIRA_INSTALL_RELEASE}-${SYSTEM_ARCH}.rpm)
SET(SPEC_FILES		 "${MIRA_INSTALL_PREFIX}/*")

# read description from file
FILE(READ ${MIRA_INSTALL_DESCRIPTION_FILE} SPEC_DESCRIPTION)
IF(NOT SPEC_DESCRIPTION)
	SET(SPEC_DESCRIPTION "No description available.")
ENDIF()

# read changelog from file and reformat according the RPM spec syntax
FILE(STRINGS ${MIRA_INSTALL_CHANGELOG_FILE} changelogList)
# TODO: Re-format changelog according the RPM spec syntax

# requirements
SET(SPEC_REQUIRES    "")
FOREACH(requirement ${MIRA_INSTALL_REQUIREMENTS})
	# replace a{op}b with "a {op} b"
	STRING(REGEX REPLACE "(.*[^<>])=(.*)" "\\1 = \\2"  requirement "MIRA-${requirement}")
	STRING(REGEX REPLACE "(.*)>=(.*)"     "\\1 >= \\2" requirement "MIRA-${requirement}")
	STRING(REGEX REPLACE "(.*)<=(.*)"     "\\1 <= \\2" requirement "MIRA-${requirement}")
	STRING(REGEX REPLACE "(.*)>([^<>].*)" "\\1 > \\2"  requirement "MIRA-${requirement}")
	STRING(REGEX REPLACE "(.*)<([^<>].*)" "\\1 < \\2"  requirement "MIRA-${requirement}") 

	SET(SPEC_REQUIRES "${SPEC_REQUIRES}\nRequires: ${requirement}")
ENDFOREACH()

SET(SPEC_INSTALLPATH ${CMAKEINSTALL_PATH})

CONFIGURE_FILE(
	${MIRA_ROOT_DIR}/make/rpmbuild.spec.in
	${RPM_SPEC_PATH}/${MIRA_INSTALL_PACKAGE}.spec)

###############################################################################
# build the RPM package using rpmbuild

MESSAGE(STATUS "*** Building RPM package ${SPEC_RPMFILENAME}")

EXEC_PROGRAM(rpmbuild ARGS
	-bb --buildroot ${INSTALL_ROOT}/BUILDROOT
	${RPM_SPEC_PATH}/${MIRA_INSTALL_PACKAGE}.spec
	OUTPUT_VARIABLE rpmbuild_output
	RETURN_VALUE rpmbuild_return)

SET(RPMBUILD_LOG "${INSTALL_ROOT}/${MIRA_INSTALL_PACKAGE}.log")
FILE(WRITE ${RPMBUILD_LOG} "${rpmbuild_output}")

IF(NOT ${rpmbuild_return} EQUAL 0)
	MESSAGE(FATAL_ERROR "Failed to build RPM ${SPEC_RPMFILENAME}. "
	        "For more details see ${RPMBUILD_LOG}.")
ENDIF()

###############################################################################
# Move RPM file

FILE(RENAME 
		${INSTALL_ROOT}/${SPEC_RPMFILENAME}
		${CMAKE_CURRENT_BINARY_DIR}/${SPEC_RPMFILENAME})

###############################################################################
# Clean up

FILE(REMOVE_RECURSE ${INSTALL_ROOT})
