/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PropertyEditorTreeView.h
 *    For internal use by PropertyEditor.C only!
 *
 * @author Erik Einhorn
 * @date   2011/05/09
 */

#ifndef _MIRA_PROPERTYEDITOR_H_
#  error "This file must be included by PropertyEditor.C"
#endif

#include <QPainter>
#include <QHeaderView>
#include <QTreeView>
#include <QMouseEvent>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class PropertyEditor::TreeView : public QTreeView
{
public:
	TreeView(PropertyEditor* editor) : QTreeView(editor), mEditor(editor) {
		//setEditTriggers(SelectedClicked);
		setEditTriggers(QTreeView::EditKeyPressed);
		setIconSize(QSize(18, 18));
		setAlternatingRowColors(true);
		setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOn);
		setSelectionMode(QTreeView::SingleSelection);
		setSelectionBehavior(QTreeView::SelectRows);
		setRootIsDecorated(true);

		// disable movable header
#if QT_VERSION > QT_VERSION_CHECK(5, 12, 0)
		header()->setSectionsMovable(false);
#else
		header()->setMovable(false);
#endif
	}

public:

	void updateAll()
	{
		setDirtyRegion(rect());
		update();
	}

protected:

	virtual void mousePressEvent(QMouseEvent *event);
	virtual void keyPressEvent(QKeyEvent *event);
	virtual void drawRow(QPainter* painter, const QStyleOptionViewItem& option,
	                     const QModelIndex& index) const;
	friend class PropertyItemEditor;
public:
	PropertyEditor* mEditor;
};

///////////////////////////////////////////////////////////////////////////////

void PropertyEditor::TreeView::mousePressEvent(QMouseEvent *event)
{
	QTreeView::mousePressEvent(event);

	if (event->button() == Qt::LeftButton) {
		QModelIndex index = indexAt(event->pos());
		if (index.isValid() && index.column()>=1)
			edit(index);
	}
}

void PropertyEditor::TreeView::keyPressEvent(QKeyEvent *event)
{
	switch (event->key()) {
		// edit property if one of the following keys is pressed:
		case Qt::Key_Return:
		case Qt::Key_Enter:
		case Qt::Key_Space:
		{
			QModelIndex index = currentIndex();

			//TODO: if (!m_editorPrivate->editedItem())
			if (index.isValid()) {

				// If the current position is at column 0, move to 1.
				if (index.column() == 0) {
					index = index.sibling(index.row(), 1);
					setCurrentIndex(index);
				}

				if(index.column()>=1) {
					event->accept();
					edit(index);
					return;
				}
			}
			break;
		}
		default:
			break;
	}
	QTreeView::keyPressEvent(event);
}

void PropertyEditor::TreeView::drawRow(QPainter* painter,
                                       const QStyleOptionViewItem& option,
                                       const QModelIndex& index) const
{
	QStyleOptionViewItem opt = option;

	QColor c = mEditor->mModel->getBackgroundColor(index);
	if (c.isValid()) {
		painter->fillRect(option.rect, c);
		opt.palette.setColor(QPalette::Base, c);
		opt.palette.setColor(QPalette::AlternateBase, c.lighter(108));
	}

	QTreeView::drawRow(painter, opt, index);

	QColor color = static_cast<QRgb>(QApplication::style()->styleHint(QStyle::SH_Table_GridLineColor, &opt));
	painter->save();
	painter->setPen(QPen(color));
	painter->drawLine(opt.rect.x(), opt.rect.bottom(), opt.rect.right(), opt.rect.bottom());
	painter->restore();
}

///////////////////////////////////////////////////////////////////////////////

}
