
/*
 * Copyright (C) 2014 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PoseLineEdit.C
 *    Implementation of PoseLineEdit.h.
 *
 * @author Christian Reuther
 * @date   2014/09/09
 */

#include <widgets/PoseLineEdit.h>

#include <QWidget>
#include <QLineEdit>
#include <QHBoxLayout>

#include <error/Exception.h>
#include <widgets/SpinBoxDelegate.h>
#include <widgets/QtUtils.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

Pose2LineEdit::Pose2LineEdit(QWidget* parent) : QWidget(parent) {
	createGUI();
}

Pose2 Pose2LineEdit::value() const {
	return Pose2(mEditX->value(), mEditY->value(), deg2rad<double>(mEditPhi->value()));
}

bool Pose2LineEdit::isReadOnly() const {
	return (mEditX->isReadOnly() && mEditY->isReadOnly() && mEditPhi->isReadOnly());
}

void Pose2LineEdit::setValue(const Pose2& pose) {
	mEditX->setValue(pose.x());
	mEditY->setValue(pose.y());
	mEditPhi->setValue(rad2deg<double>(static_cast<double>(pose.phi())));
}

void Pose2LineEdit::setLimits(double minX, double minY, double minPhi, double maxX, double maxY, double maxPhi) {
	mEditX->setRange(minX, maxX);
	mEditY->setRange(minY, maxY);
	mEditPhi->setRange(minPhi, maxPhi);
}

void Pose2LineEdit::setPrecision(double stepX, double stepY, double stepPhi, int decimalsX, int decimalsY, int decimalsPhi) {
	mEditX->setSingleStep(stepX);
	mEditY->setSingleStep(stepY);
	mEditPhi->setSingleStep(stepPhi);

	// Cannot allow negative decimal places
	mEditX->setDecimals(std::max<int>(0, decimalsX));
	mEditY->setDecimals(std::max<int>(0, decimalsY));
	mEditPhi->setDecimals(std::max<int>(0, decimalsPhi));
}

void Pose2LineEdit::setReadOnly(bool ro) {
	mEditX->setReadOnly(ro);
	mEditY->setReadOnly(ro);
	mEditPhi->setReadOnly(ro);
}

///////////////////////////////////////////////////////////////////////////////

void Pose2LineEdit::createGUI() {
	const double limit = std::numeric_limits<double>::max();

	mEditX = createDoubleSpinBox(this, -limit, limit, 0.1, 0.0, QtUtils::tr(" m"), false, false, 2);
	mEditY = createDoubleSpinBox(this, -limit, limit, 0.1, 0.0, QtUtils::tr(" m"), false, false, 2);
	mEditPhi = createDoubleSpinBox(this, -limit, limit, 0.1, 0.0, QtUtils::tr(" °"), false, false, 2);

	mEditX->setPrefix(QtUtils::tr("x: "));
	mEditY->setPrefix(QtUtils::tr("y: "));
	mEditPhi->setPrefix(QtUtils::tr("φ: "));

	QHBoxLayout* h = new QHBoxLayout();
	h->addWidget(mEditX);
	h->addWidget(mEditY);
	h->addWidget(mEditPhi);
	h->setMargin(0);
	h->setSpacing(0);
	setLayout(h);
}

///////////////////////////////////////////////////////////////////////////////

}
