/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationTool3D.C
 *    Implementation of VisualizationTool3D.
 *
 * @author Erik Einhorn
 * @date   2011/07/01
 */

#include <visualization/VisualizationTool3D.h>

#include <OGRE/OgreSceneManager.h>

#include <widgets/OgreUtils.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

VisualizationTool3D::VisualizationTool3D() :
	mSite(NULL)
{
}

VisualizationTool3D::~VisualizationTool3D() {
	if(getSite()==NULL || mNode==NULL)
		return;

	getSite()->getSceneManager()->destroySceneNode(mNode);
}

void VisualizationTool3D::init(IVisualizationSite* site)
{
	VisualizationTool::init(site);
	mSite = dynamic_cast<IVisualization3DSite*>(site);

	Ogre::SceneManager* mgr = mSite->getSceneManager();
	mNode = mgr->getRootSceneNode()->createChildSceneNode();

	assert(mSite!=NULL);
	setupScene(mSite);
}

IVisualization3DSite* VisualizationTool3D::getSite()
{
	return mSite;
}

void VisualizationTool3D::activate()
{
	if (mNode != NULL)
		mNode->setVisible(true);
	// try to get transform here throws an exception if there is no one available
	// this will inform the user once upon activation as we catch and ignore it in update()
	getTransform<RigidTransform3f>(
			getSite()->getFixedFrame(), getSite()->getCameraFrame(), Time::now());
}

void VisualizationTool3D::deactivate()
{
	if (mNode != NULL)
		mNode->setVisible(false);
}


void VisualizationTool3D::update(Duration dt)
{
	try
	{
		mTransform = getTransform<RigidTransform3f>(
			getSite()->getFixedFrame(), getSite()->getCameraFrame(), Time::now());

		OgreUtils::setTransform(mNode, mTransform);
	}
	catch(Exception&)
	{
		// ignore
	}
}

Ogre::Vector3 VisualizationTool3D::transformVector(const Ogre::Vector3& v)
{
	Eigen::Vector3f e(v.x,v.y,v.z);
	Eigen::Vector3f r;
	r = mTransform.inverse() * e;
	return Ogre::Vector3(r.x(),r.y(),r.z());
}


///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::VisualizationTool3D, mira::VisualizationTool)
