/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeTapePlayerWidget.C
 *    Implementation of TapePlayerWidget.h.
 *
 * @author Tim Langner
 * @date   2012/03/02
 */

#include <boost/format.hpp>

#include <fw/TapePlayerWidget.h>

#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QMenu>
#include <QMessageBox>
#include <QListWidget>

#include <widgets/SelectionListDialog.h>

#include <fw/TapeFileDialog.h>
#include <fw/TapePlayer.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

TapePlayerWidget::TapePlayerWidget(QWidget* parent) : QWidget(parent)
{
	QVBoxLayout* vLayout = new QVBoxLayout(this);
	vLayout->setContentsMargins(0, -1, 0, 0);

	slider = new QSlider(Qt::Horizontal, this);
	connect(slider, SIGNAL(sliderPressed()), this, SLOT(startJump()));
	connect(slider, SIGNAL(sliderMoved(int)), this, SLOT(moveJump(int)));
	connect(slider, SIGNAL(sliderReleased()), this, SLOT(jump()));
	slider->setVisible(false);
	vLayout->addWidget(slider);

	info = new QLabel(this);
	info->setMinimumWidth(10);
	vLayout->addWidget(info);

	QHBoxLayout* hLayout = new QHBoxLayout();
	hLayout->setSpacing(0);

	btOpen = new QToolButton(this);
	btOpen->setAutoRaise(true);
	btOpen->setFixedSize(32,32);
	btOpen->setIcon(QIcon(":/icons/Open.png"));
	btOpen->setToolTip("Open one or multiple tape files for playback");
	connect(btOpen, SIGNAL(clicked()), this, SLOT(open()) );
	hLayout->addWidget(btOpen);

	btStop = new QToolButton(this);
	btStop->setAutoRaise(true);
	btStop->setFixedSize(32,32);
	btStop->setIcon(QIcon(":/icons/Stop.png"));
	btStop->setToolTip("Stop playback");
	btStop->setEnabled(false);
	connect(btStop, SIGNAL(clicked()), this, SLOT(stop()) );
	hLayout->addWidget(btStop);

	btPlay = new QToolButton(this);
	btPlay->setAutoRaise(true);
	btPlay->setFixedSize(32,32);
	btPlay->setIcon(QIcon(":/icons/Play.png"));
	btPlay->setEnabled(false);
	btPlay->setToolTip("Start playback");
	connect(btPlay, SIGNAL(clicked()), this, SLOT(play()) );
	hLayout->addWidget(btPlay);

	btStep = new QToolButton(this);
	btStep->setAutoRaise(true);
	btStep->setFixedSize(48,32);
	btStep->setIcon(QIcon(":/icons/Step.png"));
	btStep->setToolTip("Single step mode. Step to the next data message.");
	btStep->setEnabled(false);
	btStep->setPopupMode(QToolButton::MenuButtonPopup);

	QMenu* menu = new QMenu(btStep);
	actStep = menu->addAction("Step To ...");
	actStep->setStatusTip("If enabled, the step button plays back all messages in the tape between current tape time and the next occurrence of a message in the selected channel");
	actStep->setCheckable(true);
	actStep->setChecked(false);
	connect(actStep, SIGNAL(toggled(bool)), this, SLOT(stepToChanged(bool)));

	actAutoStep = menu->addAction("Auto Step ...");
	actAutoStep->setStatusTip("If enabled, a step is made whenever data in the selected channel changes");
	actAutoStep->setCheckable(true);
	actAutoStep->setChecked(false);
	connect(actAutoStep, SIGNAL(toggled(bool)), this, SLOT(autoStepToChanged(bool)));

	actPauseControl = menu->addAction("Pause Control ...");
	actPauseControl->setStatusTip("If a bool channel is set, the channel controls the pause state (if it is set to true, the player pauses)");
	actPauseControl->setCheckable(true);
	actPauseControl->setChecked(false);
	connect(actPauseControl, SIGNAL(toggled(bool)), this, SLOT(pauseControlChanged(bool)));

	btStep->setMenu(menu);
	connect(btStep, SIGNAL(clicked()), this, SLOT(step()) );
	hLayout->addWidget(btStep);

	cbLoop = new QToolButton(this);
	cbLoop->setAutoRaise(true);
	cbLoop->setFixedSize(48,32);
	cbLoop->setIcon(QIcon(":/icons/Loop.png"));
	cbLoop->setCheckable(true);
	cbLoop->setChecked(false);
	cbLoop->setEnabled(false);
	cbLoop->setPopupMode(QToolButton::MenuButtonPopup);
	cbLoop->setToolTip("Loop the playback");

	actLoopCount.resize(5);
	loopMenu = new QMenu(cbLoop);
	QActionGroup* group = new QActionGroup(cbLoop);
	actLoopCount[0] = loopMenu->addAction("Infinite");
	actLoopCount[0]->setStatusTip("Infinite loops");
	actLoopCount[0]->setCheckable(true);
	actLoopCount[0]->setActionGroup(group);
	actLoopCount[1] = loopMenu->addAction("2");
	actLoopCount[1]->setStatusTip("Two loops");
	actLoopCount[1]->setCheckable(true);
	actLoopCount[1]->setActionGroup(group);
	actLoopCount[2] = loopMenu->addAction("5");
	actLoopCount[2]->setStatusTip("Five loops");
	actLoopCount[2]->setCheckable(true);
	actLoopCount[2]->setActionGroup(group);
	actLoopCount[3] = loopMenu->addAction("10");
	actLoopCount[3]->setStatusTip("Ten loops");
	actLoopCount[3]->setCheckable(true);
	actLoopCount[3]->setActionGroup(group);
	actLoopCount[4] = loopMenu->addAction("20");
	actLoopCount[4]->setStatusTip("Twenty loops");
	actLoopCount[4]->setCheckable(true);
	actLoopCount[4]->setActionGroup(group);
	actLoopCount[0]->setChecked(true);
	cbLoop->setMenu(loopMenu);
	connect(cbLoop, SIGNAL(toggled(bool)), this, SLOT(loopChanged(bool)));
	hLayout->addWidget(cbLoop);

	btChannels = new QToolButton(this);
	btChannels->setAutoRaise(true);
	btChannels->setFixedSize(32,32);
	btChannels->setIcon(QIcon(":/icons/SelectChannels.png"));
	btChannels->setToolTip("Select or exclude channels from playback");
	connect(btChannels, SIGNAL(clicked()), this, SLOT(selectChannels()) );
	hLayout->addWidget(btChannels);


	speed = new QSlider(Qt::Horizontal, this);
	speed->setRange(-10, 10);
	speed->setValue(0);
	speed->setSingleStep(1);
	speed->setPageStep(1);
	speed->setEnabled(false);
	connect(speed, SIGNAL(valueChanged(int)), this, SLOT(setSpeed(int)));
	hLayout->addWidget(speed);

	lblSpeed = new QLabel(this);
	lblSpeed->setText("1.00x");
	hLayout->addWidget(lblSpeed);
	vLayout->addLayout(hLayout);

	connect(this, SIGNAL(onPauseControl(bool)), this, SLOT(pause(bool)));
	connect(this, SIGNAL(onAutoStep()), this, SLOT(autoStep()));

	lastTime = Time::now();

	connect(&checkTimer, SIGNAL(timeout()), this, SLOT(check()));
}

void TapePlayerWidget::openTapes(const QStringList& tapeFiles, boost::optional<std::vector<std::string>> channelsToPlay)
{
	if (tapeFiles.size() == 0)
		return;

	lastTapes = tapeFiles;
	lastChannelsToPlay = channelsToPlay;
	player.reset(new TapePlayer());
	visitor = TapeVisitor();
	tapes.clear();
	for(int32 i=0; i<tapeFiles.size(); ++i)
	{
		tapes[tapeFiles[i]] = new Tape();
		tapes[tapeFiles[i]]->open(tapeFiles[i].toUtf8().data(), Tape::READ);
		if(!channelsToPlay)
			visitor.visit(tapes[tapeFiles[i]]);
		else
			visitor.visit(tapes[tapeFiles[i]], channelsToPlay.get());
	}
	int totalPlayTime = (int)visitor.getLastMessageTimeOffset().totalMilliseconds();
	slider->setRange(0, totalPlayTime);
	slider->setValue(0);
	setSpeed(0);
	slider->setVisible(true);
	btOpen->setEnabled(false);
	btStop->setEnabled(true);
	btPlay->setEnabled(true);
	btStep->setEnabled(true);
	cbLoop->setEnabled(true);
	speed->setEnabled(true);
	checkTimer.start(100);
	int32 loopCount = getLoopCount();
	player->setLoop(cbLoop->isChecked(), loopCount);
	if (!mAutoStepChannel.empty())
		player->getAuthority()->subscribe<void>(mAutoStepChannel, &TapePlayerWidget::onAutoStep, this);
	if (!mPauseControlChannel.empty())
		player->getAuthority()->subscribe<bool>(mPauseControlChannel, &TapePlayerWidget::onPauseControl, this);
	Time n = Time::now();
	if (lastTime < n)
		lastTime = n;

	try {
		player->load(&visitor,
		             useOriginalTimestamp ? visitor.getStartTime() : lastTime,
		             namespacePrefix);
	} catch(...) {
		player.reset();
		closeTapes();
		throw;
	}
}

void TapePlayerWidget::closeTapes()
{
	btOpen->setEnabled(true);
	btStop->setEnabled(false);
	btPlay->setEnabled(false);
	btStep->setEnabled(false);
	actStep->setChecked(false);
	cbLoop->setEnabled(false);
	speed->setEnabled(false);
	slider->setVisible(false);
	checkTimer.stop();
	slider->setValue(0);
	speed->setValue(0);
	lblSpeed->setText("1.00x");
	info->setText("");
	info->setToolTip("");
	btPlay->setIcon(QIcon(":/icons/Play.png"));

	if (player)
	{
		player->stop();
		// finished playing. open the tape again and set lastTime to the
		// last relative time of the player to start new play back at
		// and add the time of the first message plus 1 microsecond
		lastTime = player->getRelativeStartTime() +
				   player->getCurrentMessageTimeOffset()+
				   Duration::microseconds(1);
		player.reset();
	}
	visitor = TapeVisitor();
	for(auto it=tapes.begin(); it!=tapes.end(); ++it)
		delete it->second;
	tapes.clear();
}

void TapePlayerWidget::open()
{
	TapeFileDialog dialog(this);
	dialog.setNameFilter("Tape files (*.tape)");
	dialog.setFileMode(QFileDialog::ExistingFiles);
	dialog.setAcceptMode(QFileDialog::AcceptOpen);
	if (!dialog.exec())
		return;
	QStringList fileNames = dialog.selectedFiles();
	emit onOpen(fileNames, dialog.useOriginalTimestamp(), dialog.getNamespacePrefix());
	useOriginalTimestamp = dialog.useOriginalTimestamp();
	namespacePrefix = dialog.getNamespacePrefix().toStdString();
	openTapes(fileNames);
}

void TapePlayerWidget::stop()
{
	emit onStop();
	closeTapes();
}

void TapePlayerWidget::play()
{
	if(!player)
		return;

	btOpen->setEnabled(false);
	btStop->setEnabled(true);
	btPlay->setEnabled(true);
	btStep->setEnabled(false);
	cbLoop->setEnabled(true);
	speed->setEnabled(true);

	if (!player->isPaused() && !player->inStepTo())
	{
		player->pause();
		btPlay->setIcon(QIcon(":/icons/Play.png"));
		btStep->setEnabled(true);
	}
	else
	{
		player->play();
		btPlay->setIcon(QIcon(":/icons/Pause.png"));
	}
}

void TapePlayerWidget::step()
{
	if(!player)
		return;

	if (actStep->isChecked())
	{
		// if this returns false, the player just resumed normal play
		if (player->stepTo(mStepToChannel))
			btPlay->setIcon(QIcon(":/icons/Play.png"));
		else
			btPlay->setIcon(QIcon(":/icons/Pause.png"));
		btStep->setEnabled(false);
	}
	else {
		player->step();
		btPlay->setIcon(QIcon(":/icons/Play.png"));
	}
}

void TapePlayerWidget::onAutoStep(ChannelRead<void> data)
{
	if (!player || !player->isPlaying())
		return;
	emit onAutoStep();
}

void TapePlayerWidget::autoStep()
{
	if(!player)
		return;

	if (actStep->isChecked())
	{
		player->stepTo(mStepToChannel);
		btStep->setEnabled(false);
	}
	else
		player->step();
}

void TapePlayerWidget::onPauseControl(ChannelRead<bool> data)
{
	if (!player || !player->isPlaying())
		return;
	emit onPauseControl(*data);
}

void TapePlayerWidget::pause(bool pause)
{
	if(!player)
		return;

	if(actPauseControl->isChecked())
	{
		if(pause)
			player->pause();
		else
			player->play();
	}
}

void TapePlayerWidget::stepToChanged(bool state)
{
	if (state)
	{
		SelectionListDialog<std::string> dialog;
		dialog.setMultiSelection(false);
		dialog.setHeaderLabels(QStringList("Channel"));
		const TapeVisitor::ChannelMap& channelMap = visitor.getChannels();
		foreach(auto channel, channelMap)
			dialog.addItem(QStringList(channel.first.c_str()), channel.first);
		if (dialog.exec() == 0)
		{
			actStep->setChecked(false);
			return;
		}
		auto selected = dialog.selectedItems();
		if (selected.empty())
		{
			actStep->setChecked(false);
			return;
		}
		mStepToChannel = selected.begin()->first.toStdString();
		actStep->setText(QString("Step To: ") + selected.begin()->first);
	}
	else
		actStep->setText("Step To ...");
}

void TapePlayerWidget::autoStepToChanged(bool state)
{
	if (!mAutoStepChannel.empty() && player)
		player->getAuthority()->unsubscribe<void>(mAutoStepChannel);
	mAutoStepChannel.clear();
	if (state)
	{
		SelectionListDialog<std::string> dialog;
		dialog.setMultiSelection(false);
		dialog.setHeaderLabels(QStringList("Channel"));
		const auto& channelMap = MIRA_FW.getChannelManager().getChannels();
		foreach(auto channel, channelMap)
			dialog.addItem(QStringList(channel.c_str()), channel);
		
		if (dialog.exec() != 0)
		{
			auto selected = dialog.selectedItems();
			if (selected.empty())
				actAutoStep->setChecked(false);
			else
			{
				mAutoStepChannel = selected.begin()->first.toStdString();
				actAutoStep->setText(QString("Auto Step: ") + selected.begin()->first);
				if (!mAutoStepChannel.empty() && player)
					player->getAuthority()->subscribe<void>(mAutoStepChannel, &TapePlayerWidget::onAutoStep, this);
			}
		} else
			actAutoStep->setChecked(false);
	}
	else
		actAutoStep->setText("Auto Step ...");
}

void TapePlayerWidget::pauseControlChanged(bool state)
{
	if (!mPauseControlChannel.empty() && player)
		player->getAuthority()->unsubscribe<void>(mPauseControlChannel);
	mPauseControlChannel.clear();
	actPauseControl->setText("Pause Control ...");
	if (state)
	{
		SelectionListDialog<std::string> dialog;
		dialog.setMultiSelection(false);
		dialog.setHeaderLabels(QStringList("Channel"));
		const auto& channelMap = MIRA_FW.getChannelManager().getChannels();
		foreach(auto channel, channelMap)
			dialog.addItem(QStringList(channel.c_str()), channel);

		if (dialog.exec() != 0)
		{
			auto selected = dialog.selectedItems();
			if (selected.empty())
				actPauseControl->setChecked(false);
			else
			{
				mPauseControlChannel = selected.begin()->first.toStdString();
				actPauseControl->setText(QString("Pause Control: ") + selected.begin()->first);
				if (!mPauseControlChannel.empty() && player)
					player->getAuthority()->subscribe<bool>(mPauseControlChannel, &TapePlayerWidget::onPauseControl, this);
			}
		} else
			actPauseControl->setChecked(false);
	}
}


void TapePlayerWidget::selectChannels()
{
	// collect all channels from all opened tapes
	std::set<std::string> channelSet;
	foreach(auto p, tapes)
	{
		foreach(const auto& c, p.second->getChannels())
			channelSet.insert(c.first);
	}

	SelectionListDialog<std::string> dialog;
	dialog.setHeaderLabels(QStringList("Channels to play"));
	dialog.setMultiSelection(true);

	foreach(const std::string& s, channelSet)
	{
		bool select = true;
		if(lastChannelsToPlay) {
			if(std::find(lastChannelsToPlay->begin(),lastChannelsToPlay->end(),s) == lastChannelsToPlay->end())
				select = false;
		}
		dialog.addItem(QStringList(s.c_str()), s,select);
	}

	if (dialog.exec() == 0)
		return;

	std::vector<std::string> channelsToPlay;
	foreach(auto p, dialog.selectedItems())
		channelsToPlay.push_back(p.second);

	// TODO: get playing state and position in tape

	// need to stop playback and close the tapes
	closeTapes();
	// reopen by taking the selected channels into account
	openTapes(lastTapes, channelsToPlay);

	// TODO: restore position in tape and playback state

}

int32 TapePlayerWidget::getLoopCount() const
{
	int32 loopCount = -1;
	if (actLoopCount[1]->isChecked())
		loopCount = 2;
	else if (actLoopCount[2]->isChecked())
		loopCount = 5;
	else if (actLoopCount[3]->isChecked())
		loopCount = 10;
	else if (actLoopCount[4]->isChecked())
		loopCount = 20;
	return loopCount;
}

void TapePlayerWidget::loopChanged(bool state)
{
	if(!player)
		return;

	int32 loopCount = getLoopCount();
	if (state)
		loopMenu->setEnabled(false);
	else
		loopMenu->setEnabled(true);
	player->setLoop(state, loopCount);
}

void TapePlayerWidget::setSpeed(int speed)
{
	float scaler = 1.0f;
	if (speed < 0)
	{
		speed = -speed;
		int f = speed / 5;
		scaler = 1.0f / (((speed % 5)+1)*(pow(10.0f, f)));
	}
	else
		if (speed > 0)
			scaler = (float)pow(2.0f, speed);
	player->setTimeScaler(scaler);
	lblSpeed->setText((boost::format("%.3fx") % player->getTimeScaler()).str().c_str());
}

void TapePlayerWidget::check()
{
	if(!player)
		return;

	if (player->isPlaying())
	{
		// for the purpose of TapePlayerWidget UI, assume all entries with negative time offset
		// occur at the recording time (this is how playback handles them)
		Duration maxPlayTime = std::max(Duration::seconds(0), visitor.getLastMessageTimeOffset());
		Duration currentMessageTime = std::max(Duration::seconds(0), player->getCurrentMessageTimeOffset());
		Duration current = currentMessageTime;
		slider->setValue(current.totalMilliseconds());
		Time currentRealTime = (player->getRelativeStartTime()+currentMessageTime).toLocal();
		Time recTime = (visitor.getStartTime()+currentMessageTime).toLocal();
		std::stringstream s;
		s << "  Playing " << currentMessageTime << " of " << maxPlayTime;
		if (maxPlayTime > Duration::seconds(0))
			s << " (" << (round(currentMessageTime.totalSeconds() * 1000.f / maxPlayTime.totalSeconds())/10.f) << "%)";
		s << "\n";
		s << "    Message time: " << toString(currentRealTime.time_of_day()) << "\n" <<
		     "    Recording time: " << toString(recTime) << "\n" << "    ";
		if(lastTapes.size() > 1)
			s << "Files: " << lastTapes.first().toStdString() << " (+ " << (lastTapes.size() - 1) << " more)";
		else
			s << "File: " << lastTapes.first().toStdString();

		info->setText(s.str().c_str());
		info->setToolTip(lastTapes.join("\n"));
		if (!player->inStepTo())
			btStep->setEnabled(true);
	}
	else
	{
		if (player->hasError())
		{
			QMessageBox::critical(this, "Error playing back Tape", player->getErrorMessage().c_str());
			stop();
		}
		else
		{
			stop();
			emit onOpen(lastTapes, useOriginalTimestamp, QString(namespacePrefix.c_str()));
			openTapes(lastTapes, lastChannelsToPlay);
		}
	}
}

void TapePlayerWidget::startJump()
{
	if(!player)
		return;

	if (player->isPlaying())
	{
		checkTimer.stop();
	}
}

void TapePlayerWidget::moveJump(int time)
{
	Duration maxPlayTime = visitor.getLastMessageTimeOffset();
	Time recTime = (visitor.getStartTime()+Duration::milliseconds(time)).toLocal();
	std::stringstream s;
	s << "  Move to " << Duration::milliseconds(time) << " of " << maxPlayTime << "\n" <<
		"    Recording time: " << toString(recTime) << "\n";
	if(lastTapes.size() > 1)
		s << "Files: " << lastTapes.first().toStdString() << " (+ " << (lastTapes.size() - 1) << " more)";
	else
		s << "File: " << lastTapes.first().toStdString();
	info->setText(s.str().c_str());
	info->setToolTip(lastTapes.join("\n"));
}

void TapePlayerWidget::jump()
{
	if(!player)
		return;

	if (player->isPlaying())
	{
		player->jumpTo(Duration::milliseconds(slider->value()));
		checkTimer.start(100);
	}
}

void TapePlayerWidget::jumpToTime(Duration time)
{
	if(!player)
		return;

	if (player->isPlaying())
		player->jumpTo(time);
}

///////////////////////////////////////////////////////////////////////////////

}
