/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PolygonTool.h
 *    Declaration of PolygonTool.
 *
 * @author Tim Langner
 * @date   2012/10/05
 */

#ifndef _MIRA_POLYGONTOOL_H_
#define _MIRA_POLYGONTOOL_H_

#ifndef Q_MOC_RUN
#include <OGRE/OgreRenderWindow.h>
#include <OGRE/OgreSceneManager.h>
#endif

#include <geometry/Polygon.h>

#include <visualization/3d/SceneQueryTool.h>
#include <visualization/3d/PolygonObject.h>

#include <serialization/adapters/OGRE/OgreColourValue.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Tool base class for selecting a polygon in the xy-plane of a 3D scene
 * (in a Visualization3DView)
 */
class MIRA_GUI_VISUALIZATION_EXPORT PolygonTool : public SceneQueryTool
{
public:

	PolygonTool() : mOutlineColor(Ogre::ColourValue::Green) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, SceneQueryTool);
		r.property("Outline Color", mOutlineColor,
		           setter<Ogre::ColourValue>(&PolygonTool::setOutlineColor, this),
		           "The color of the polygon outline", Ogre::ColourValue::Green);
	}

	/// Called if the tool is active and a mouse button was released.
	virtual void onMouseReleased(QMouseEvent* e);

	/// Called if the tool is active and the mouse was moved.
	virtual void onMouseMoved(QMouseEvent* e);

public:

	/// called when the user has activated the tool.
	virtual void activate();

	/// called when the user has deactivated the tool by selecting another one.
	virtual void deactivate();

protected:

	virtual void setupScene(IVisualization3DSite* site);

	/// overwrite this method to be notified whenever the user enters one polygon
	virtual void onNewPolygon(const Polygon2f& pose) = 0;

	/// overwrite this method to be notified whenever the user enters multiple polygons
	virtual void onNewPolygons(const std::vector<Polygon2f>& pose) = 0;

	/// set a polygon (during editing)
	void setPolygon(const Polygon2f& polygon);

	// set color of polygon outline
	void setOutlineColor(const Ogre::ColourValue& color);

private:

	std::pair<std::string, Ogre::Vector3> getIntersection(float x, float y,
	                                                      const std::string& mask="");

	void reset();
	void finishPolygon(bool last = true);
	bool isNearFirstPoint(QMouseEvent* e, const Ogre::Vector3& p);

private:

	std::string mSelectedObject;

	// currently edited Polygon
	boost::shared_ptr<PolygonObject<float>> mPolygonObject;
	Polygon2f mPolygon;

	// previously finished polygons (for multi-polygon input)
	boost::shared_ptr<PolygonObject<float>> mPolygonsObject;
	std::vector<Polygon2f> mPolygons;

	Ogre::ColourValue mOutlineColor;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
