/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FrameworkConcurrentLoadTest.C
 *    Test concurrent calls of Framework::load()
 *
 * @author Tim van der Grinten, Christof Schröter
 * @date   2021/01/23
 */

#include <boost/test/unit_test.hpp>

#include <fw/Framework.h>

using namespace mira;

bool waitForAuthority(const std::string& authority, const Duration& timeout)
{
	Time start = Time::now();
	while (true) {
		if (MIRA_FW.getAuthorityManager().hasAuthority(authority))
			return true;

		if (Time::now() - start > timeout)
			return false;
	}
}

BOOST_AUTO_TEST_CASE(FrameworkConcurrentLoadTest)
{
	const char* argv[] = {"FrameworkConcurrentLoadTest",
	                      "${find FrameworkConcurrentLoadTest.xml}"};
	Framework fw(2,(char**)argv);
	fw.load();
	fw.start();

	auto& authmgr = MIRA_FW.getAuthorityManager();
	auto& chanmgr = MIRA_FW.getChannelManager();

	std::cout << "wait for Units /Loader1 & /Loader2" << std::endl;
	int maxWait = 5;
	int n;
	for (n = 0; n < maxWait; ++n) {
		auto authorities = chanmgr.getSubscribers("/Trigger");
		if ((find(authorities.begin(), authorities.end(), "/Loader1") != authorities.end()) &&
		    (find(authorities.begin(), authorities.end(), "/Loader2") != authorities.end()))
			break;

		MIRA_SLEEP(1000);
	}
	if (n == maxWait)
		MIRA_THROW(XRuntime, "Loader1/2 not ready");

	for (int n = 0; n < 5; ++n) {
		std::cout << "trigger concurrent load of additional config, wait for new Units" << std::endl;
		chanmgr.writeChannel("/Trigger", json::Value(true), Time::now());

		BOOST_CHECK(waitForAuthority("/namespace1/UnitNS1_"+toString(n), Duration::seconds(5)));
		BOOST_CHECK(waitForAuthority("/namespace2/UnitNS2_"+toString(n), Duration::seconds(5)));
		
		// now additional Units should be loaded
		std::cout << "list of Authorities:" << std::endl;
		auto authorities = authmgr.getAuthorities();
		foreach(const auto& a, authorities)
			std::cout << "  " << a << std::endl;

		std::cout << "----------------------------------" << std::endl;
	}	
}

